#include "manager.h"

#include <drive/backend/abstract/base.h>


namespace NDrive::NFine {
    TFinesManager::TFinesManager(const IHistoryContext& context, const TFinesManagerConfig& config, const NDrive::IServer& server)
        : TDatabaseSessionConstructor(context.GetDatabase())
        , Config(config)
        , Server(server)
        , FineArticleMatcher(nullptr)
        , FineDB(context.GetDatabase())
        , FineAttachmentManager(context.GetDatabase())
    {
    }

    TFineArticleMatcher::TPtr TFinesManager::GetFineArticleMatcherPtr(const bool ensureActual) const {
        if (!!FineArticleMatcher && !ensureActual) {
            return FineArticleMatcher;
        }

        auto actualConfig = TFineArticleMatcherConfig::Construct();
        if (!actualConfig) {
            return nullptr;
        }

        if (!FineArticleMatcher || FineArticleMatcher->GetConfig() != *actualConfig) {
            try {
                FineArticleMatcher = MakeAtomicShared<TFineArticleMatcher>(*actualConfig);
            } catch (yexception& e) {
                ERROR_LOG << "Cannot construct fine article matcher: " << e.what() << Endl;
                return nullptr;
            }
        }

        return FineArticleMatcher;
    }

    TOptionalFines TFinesManager::GetFines(const TFineFilterGroup& filterGroup, NDrive::TEntitySession& tx, NSQL::TQueryOptions options /* = {} */) const {
        filterGroup.PatchQuery(options);
        auto result = FineDB.Fetch(tx, options);
        if (!result) {
            return {};
        }
        EraseIf(*result, [&filterGroup](const auto& item) { return !filterGroup.Match(item); });
        return result;
    }

    TOptionalFine TFinesManager::GetFineById(const TString& id, NDrive::TEntitySession& tx, NSQL::TQueryOptions options /* = {} */) const {
        if (!id) {
            tx.SetErrorInfo("GetFineById", "Got empty fine", EDriveSessionResult::InternalError);
            return {};
        }
        options.AddGenericCondition("id", id);
        auto result = FineDB.Fetch(tx, options);
        if (!result || result->empty()) {
            return {};
        }
        return std::move(result->front());
    }

    TOptionalFines TFinesManager::GetFinesByUserId(const TString& userId, const TFineFilterGroup& filterGroup, NDrive::TEntitySession& tx, NSQL::TQueryOptions options /* = {} */) const {
        options.AddGenericCondition("user_id", userId);
        return GetFines(filterGroup, tx, options);
    }

    TOptionalFines TFinesManager::GetFinesByCarId(const TString& carId, const TFineFilterGroup& filterGroup, NDrive::TEntitySession& tx, NSQL::TQueryOptions options /* = {} */) const {
        options.AddGenericCondition("car_id", carId);
        return GetFines(filterGroup, tx, options);
    }

    TOptionalFines TFinesManager::GetFinesBySessionIds(const TSet<TString>& sessionIds, const TFineFilterGroup& filterGroup, NDrive::TEntitySession& tx, NSQL::TQueryOptions options /* = {} */) const {
        options.MutableSubOrOptions().emplace_back(NSQL::TQueryOptions().SetGenericCondition("session_id", sessionIds));
        options.MutableSubOrOptions().emplace_back(NSQL::TQueryOptions().SetGenericCondition("order_id", sessionIds));
        return GetFines(filterGroup, tx, options);
    }

    bool TFinesManager::UpsertFine(const TAutocodeFineEntry& fine, NDrive::TEntitySession& tx) const {
        return FineDB.Upsert(fine, tx);
    }

    bool TFinesManager::UpsertFineNotCharged(const TAutocodeFineEntry& fine, NDrive::TEntitySession& tx) const {
        NSQL::TQueryOptions options;
        options.AddGenericCondition("id", fine.GetId());
        options.SetGenericCondition("charged_at", NSQL::TNot<TNull>());
        if (auto dbFines = GetFines({}, tx, options)) {
            if (!dbFines->empty()) {
                tx.SetErrorInfo("UpsertFineNotCharged", "fine(" + fine.GetId() + ") is charged: " + fine.GetRulingNumber() , EDriveSessionResult::InternalError);
                return false;
            }
        } else {
            return false;
        }
        return FineDB.Upsert(fine, tx);
    }

    TFineAttachmentManager::TOptionalEntities TFinesManager::GetFineAttachments(const TString& fineId, const EFineAttachmentType attachmentType, NDrive::TEntitySession& tx) const {
        return GetFineAttachments(TSet<TString>{ fineId }, attachmentType, tx);
    }

    TFineAttachmentManager::TOptionalEntities TFinesManager::GetFineAttachments(const TSet<TString>& fineIds, const EFineAttachmentType attachmentType, NDrive::TEntitySession& tx) const {
        auto options = NSQL::TQueryOptions().SetGenericCondition("fine_id", fineIds);
        if (attachmentType != EFineAttachmentType::All) {
            options.AddGenericCondition("data_type", ToString(attachmentType));
        }
        return FineAttachmentManager.Fetch(tx, options);
    }

    bool TFinesManager::UpsertFineAttachment(const TAutocodeFineAttachmentEntry& fineAttachment, NDrive::TEntitySession& tx) const {
        return FineAttachmentManager.Upsert(fineAttachment, tx);
    }

    TMaybe<TVector<i64>> TFinesManager::GetFineAmounts(const TFineFilterGroup& filterGroup, NDrive::TEntitySession& tx, NSQL::TQueryOptions options /* = {} */) const {
        filterGroup.PatchQuery(options);
        return FineDB.GetFineAmountsCents(tx, options);
    }
}
