#include "user.h"

#include <drive/backend/fines/filters.h>
#include <drive/backend/fines/manager.h>
#include <drive/backend/fines/autocode/entries.h>

#include <drive/backend/actions/abstract/action.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/private_data.h>
#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/offers/offers/abstract.h>
#include <drive/backend/roles/action.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_filter.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/raw_text/datetime.h>
#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/util/json_processing.h>

#include <util/string/builder.h>

namespace {
    static tm GetLocalTimeZoneNow(const TString& tzName) {
        tm nowLocalTm;  // local to timezone time structure

        auto nowLocalTimestamp = NUtil::ConvertTimeZone(TInstant::Now(), NUtil::GetUTCTimeZone(), NUtil::GetTimeZone(tzName));
        nowLocalTimestamp.GmTime(&nowLocalTm);

        return nowLocalTm;
    }

    static bool AreYearDaysEqual(const tm& lhs, const tm& rhs) {
        return lhs.tm_year == rhs.tm_year && lhs.tm_mon == rhs.tm_mon;
    }
}

namespace NFineUserRestrictions {
    const TString TFineUserRestrictions::DefaultTimeZoneName = "Europe/Moscow";
    const TString TFineUserRestrictions::DefaultBirthdayTagName = "user_character_birth_day";
    const TString TFineUserRestrictions::DefaultLongTermOfferGroupingTags = "long_term";

    bool TFineUserRestrictions::IsTotalChargeCountLimitSet() const {
        return !!TotalChargeCountLimit || TotalChargeCountLimit == std::numeric_limits<decltype(TotalChargeCountLimit)>::max();
    }

    EUserRestrictionsStatus TFineUserRestrictions::Check(const NDrive::IServer& server, const NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& tx) const {
        const TDriveAPI* driveApi = Yensured(server.GetDriveAPI());

        if (auto check = CheckPaymentLimits(*driveApi, fine, tx); check != EUserRestrictionsStatus::Allowed) {
            return check;
        }
        if (auto check = CheckChargeCountLimits(*driveApi, fine, tx); check != EUserRestrictionsStatus::Allowed) {
            return check;
        }
        if (auto check = CheckChargeOnRide(server, fine, tx); check != EUserRestrictionsStatus::Allowed) {
            return check;
        }
        if (auto check = CheckChargeOnBirthday(*driveApi, fine, tx); check != EUserRestrictionsStatus::Allowed) {
            return check;
        }
        return EUserRestrictionsStatus::Allowed;
    }

    EUserRestrictionsStatus TFineUserRestrictions::CheckPaymentLimits(const TDriveAPI& driveApi, const NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& tx) const {
        if (!TotalCentsPaymentLimit) {
            return EUserRestrictionsStatus::Allowed;
        }
        auto amounts = GetUserChargeAmounts(driveApi, fine.GetUserId(), PaymentLimitPeriod, tx);
        if (!amounts) {
            return EUserRestrictionsStatus::Error;
        }
        i64 amountCharged = std::accumulate(amounts->cbegin(), amounts->cend(), 0);
        if (!amounts->empty() && amountCharged + fine.GetSumToPayCents() > TotalCentsPaymentLimit) {
            return EUserRestrictionsStatus::RestrictedPaymentLimits;
        }
        return EUserRestrictionsStatus::Allowed;
    }

    EUserRestrictionsStatus TFineUserRestrictions::CheckChargeCountLimits(const TDriveAPI& driveApi, const NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& tx) const {
        if (!IsTotalChargeCountLimitSet()) {
            return EUserRestrictionsStatus::Allowed;
        }
        auto amounts = GetUserChargeAmounts(driveApi, fine.GetUserId(), ChargeCountLimitPeriod, tx);
        if (!amounts) {
            return EUserRestrictionsStatus::Error;
        }
        if (amounts->size() + 1 > TotalChargeCountLimit) {
            return EUserRestrictionsStatus::RestrictedChargeCountLimits;
        }
        return EUserRestrictionsStatus::Allowed;
    }

    TMaybe<TVector<i64>> TFineUserRestrictions::GetUserChargeAmounts(const TDriveAPI& driveApi, const TString& userId, const TDuration period, NDrive::TEntitySession& tx) const {
        const auto& finesManager = driveApi.GetFinesManager();
        const TInstant now = ModelingNow();
        NDrive::NFine::TFineFilterGroup filters = { MakeAtomicShared<NDrive::NFine::TFineChargeTimeFilter>(now - period, now) };
        TVector<i64> amountsToPayCents;
        return finesManager.GetFineAmounts(filters, tx, NSQL::TQueryOptions().AddGenericCondition("user_id", userId));
    }

    EUserRestrictionsStatus TFineUserRestrictions::CheckChargeOnRide(const NDrive::IServer& server, const NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& tx) const {
        if (IsChargeOnRide()) {
            return EUserRestrictionsStatus::Allowed;
        }
        TString userId = fine.GetUserId();
        bool isPerformingTags = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().IsPerformer(userId);
        if (!isPerformingTags) {
            return EUserRestrictionsStatus::Allowed;
        }

        IOfferPtrs offers;
        if (!GetSessionOffers(server, fine.GetSessionId(), offers, tx)) {
            return EUserRestrictionsStatus::Error;
        }

        for (auto&& offer: offers) {
            bool isOfferSpecific = HasOfferBeenEvolved(offer) || IsOfferLongTerm(server, offer);
            if (isOfferSpecific) {
                return EUserRestrictionsStatus::Allowed;
            }
        }

        return EUserRestrictionsStatus::RestrictedChargeOnRide;
    }

    bool TFineUserRestrictions::GetSessionOffers(const NDrive::IServer& server, const TString& sessionId, IOfferPtrs& offers, NDrive::TEntitySession& tx) const {
        THistoryRidesContext sessionsContext(server);
        auto ydbTx = server.GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("fine_user_restrictions", &server);
        if (!sessionsContext.InitializeSession(sessionId, tx, ydbTx)) {
            return false;
        }

        auto sessionsIterator = sessionsContext.GetIterator();

        THistoryRideObject sessionInfo;
        while (sessionsIterator.GetAndNext(sessionInfo)) {
            auto offer = sessionInfo.GetOffer();
            if (offer) {
                offers.push_back(offer);
            }
        }

        return true;
    }

    bool TFineUserRestrictions::HasOfferBeenEvolved(IOfferPtr offer) const {
        return !!offer->GetParentId();
    }

    bool TFineUserRestrictions::IsOfferLongTerm(const NDrive::IServer& server, IOfferPtr offer) const {
        auto action = server.GetDriveAPI()->GetRolesManager()->GetAction(offer->GetBehaviourConstructorId());
        return action && LongTermOfferTagsFilterPtr && LongTermOfferTagsFilterPtr->IsMatching((*action)->GetGrouppingTags());
    }

    EUserRestrictionsStatus TFineUserRestrictions::CheckChargeOnBirthday(const TDriveAPI& driveApi, const NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& tx) const {
        if (IsChargeOnBirthday()) {
            return EUserRestrictionsStatus::Allowed;
        }
        tm userBithdayLocalTm;  // local to default timezone timestamp

        {
            TInstant birthdayTagAddInstant;
            if (!GetBirtdayTagTimestamp(driveApi, fine.GetUserId(), birthdayTagAddInstant, tx, /* defaultInstant = */ TInstant::Zero())) {
                return EUserRestrictionsStatus::Error;
            }

            TInstant birthdayTagLocalTimestamp = TInstant::Zero();
            if (!!birthdayTagAddInstant) {
                birthdayTagLocalTimestamp = NUtil::ConvertTimeZone(birthdayTagAddInstant, NUtil::GetUTCTimeZone(), NUtil::GetTimeZone(DefaultTimeZoneName));  // local to default timezone timestamp
            }

            birthdayTagLocalTimestamp.GmTime(&userBithdayLocalTm);
        }

        tm nowLocalTm = GetLocalTimeZoneNow(DefaultTimeZoneName);  // local to default timezone timestamp

        bool isBirthDay = AreYearDaysEqual(userBithdayLocalTm, nowLocalTm);
        if (isBirthDay) {
            return EUserRestrictionsStatus::RestrictedChargeOnBirthday;
        }

        return EUserRestrictionsStatus::Allowed;
    }

    bool TFineUserRestrictions::GetBirtdayTagTimestamp(const TDriveAPI& driveApi, const TString& userId, TInstant& tagAddInstant, NDrive::TEntitySession& tx, const TInstant& defaultInstant) const {
        const TInstant startInstant = ModelingNow() - TDuration::Days(1);

        const auto& userTags = driveApi.GetTagsManager().GetUserTags();
        auto optionalEvents = userTags.GetEventsByObject(userId, tx, 0, startInstant);
        if (!optionalEvents) {
            return false;
        }

        for (const auto& ev : *optionalEvents) {
            if (ev && ev.GetHistoryAction() == EObjectHistoryAction::Add && BirthdayTagNames.contains(ev->GetName())) {
                tagAddInstant = ev.GetHistoryInstant();
                return true;
            }
        }

        tagAddInstant = defaultInstant;
        return true;
    }

    NDrive::TScheme TFineUserRestrictions::GetScheme() {
        NDrive::TScheme scheme;

        scheme.Add<TFSNumeric>("total_cents_payment_limit", "Списать за промежуток не более чем (коп., без лимита, если ноль)").SetDefault(250000);
        scheme.Add<TFSDuration>("payment_limit_period", "Промежуток проверки ограничений на сумму").SetDefault(TDuration::Days(1));

        scheme.Add<TFSNumeric>("total_charge_count_limit", "Списаний за промежуток не более чем").SetDefault(3);
        scheme.Add<TFSDuration>("charge_count_limit_period", "Промежуток проверки ограничений на количество списаний").SetDefault(TDuration::Days(1));

        scheme.Add<TFSBoolean>("do_charge_on_ride", "Списывать во время поездки").SetDefault(false);
        scheme.Add<TFSString>("long_term_offer_grouping_tags", "Фильтр long term офферов по групповым тегам").SetDefault(DefaultLongTermOfferGroupingTags);

        scheme.Add<TFSBoolean>("do_charge_on_birthday", "Списывать в день рождения").SetDefault(false);
        scheme.Add<TFSBoolean>("do_use_tag_based_birthday_check", "Вычислять дату рождения на основании тегов").SetDefault(false);
        scheme.Add<TFSArray>("birthday_tag_names", "Имена тегов для определения дня рождения").SetElement<TFSString>();
        scheme.Add<TFSBoolean>("do_check_birthday_strict", "Не списывать в случае неуспеха получения даты рождения").SetDefault(false);

        return scheme;
    }

    bool TFineUserRestrictions::DeserializeFromJson(const NJson::TJsonValue& data) {
        JREAD_INT_OPT(data, "total_cents_payment_limit", TotalCentsPaymentLimit);
        if (data.Has("day_charge_cents_limit")) {  // deprecated
            JREAD_INT_OPT(data, "day_charge_cents_limit", TotalCentsPaymentLimit);
        }
        if (!TJsonProcessor::Read(data, "payment_limit_period", PaymentLimitPeriod)) {
            return false;
        }

        if (!TJsonProcessor::Read(data, "total_charge_count_limit", TotalChargeCountLimit)) {
            return false;
        }
        if (data.Has("day_charge_count")){  // deprecated
            if (!TJsonProcessor::Read(data, "day_charge_count", TotalChargeCountLimit)) {
                return false;
            }
        }
        if (!TJsonProcessor::Read(data, "charge_count_limit_period", ChargeCountLimitPeriod)) {
            return false;
        }

        if (!TJsonProcessor::Read(data, "do_charge_on_ride", ChargeOnRide)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "long_term_offer_grouping_tags", LongTermOfferGroupingTags)) {
            return false;
        }
        if (LongTermOfferGroupingTags) {
            LongTermOfferTagsFilterPtr = MakeAtomicShared<TTagsFilter>();
            if (!LongTermOfferTagsFilterPtr->DeserializeFromString(LongTermOfferGroupingTags) || LongTermOfferTagsFilterPtr->IsEmpty()) {
                return false;
            }
        }

        if (!TJsonProcessor::Read(data, "do_charge_on_birthday", ChargeOnBirthday)) {
            return false;
        }
        if (!TJsonProcessor::ReadContainer(data, "birthday_tag_names", BirthdayTagNames)) {
            return false;
        }

        return true;
    }

    NJson::TJsonValue TFineUserRestrictions::SerializeToJson() const {
        NJson::TJsonValue result;

        TJsonProcessor::Write(result, "total_cents_payment_limit", TotalCentsPaymentLimit);
        TJsonProcessor::Write(result, "day_charge_cents_limit", TotalCentsPaymentLimit);  // deprecated
        TJsonProcessor::WriteDurationString(result, "payment_limit_period", PaymentLimitPeriod);

        TJsonProcessor::Write(result, "total_charge_count_limit", TotalChargeCountLimit);
        TJsonProcessor::Write(result, "day_charge_count", TotalChargeCountLimit);  // deprecated
        TJsonProcessor::WriteDurationString(result, "charge_count_limit_period", ChargeCountLimitPeriod);

        TJsonProcessor::Write(result, "do_charge_on_ride", ChargeOnRide);
        TJsonProcessor::Write(result, "long_term_offer_grouping_tags", LongTermOfferGroupingTags);

        TJsonProcessor::Write(result, "do_charge_on_birthday", ChargeOnBirthday);
        TJsonProcessor::WriteContainerArray(result, "birthday_tag_names", BirthdayTagNames);

        return result;
    }
}
