#pragma once

#include <rtline/util/types/accessor.h>
#include <rtline/util/types/messages_collector.h>

#include <util/generic/map.h>
#include <util/generic/set.h>

class TDriveAPI;

class TTagsFilter;

class IOffer;

namespace NDrive {
    class TEntitySession;
    class IServer;
    class TScheme;
}

namespace NJson {
    class TJsonValue;
}

namespace NDrive::NFine {
    class TAutocodeFineEntry;
}

namespace NFineUserRestrictions {
    enum class EUserRestrictionsStatus {
        Error                           /* "error" */,
        Allowed                         /* "allowed" */,
        RestrictedPaymentLimits         /* "restricted_payment_limits" */,
        RestrictedChargeCountLimits     /* "restricted_charge_count_limits" */,
        RestrictedChargeOnBirthday      /* "restricted_charge_on_birthday" */,
        RestrictedChargeOnRide          /* "restricted_charge_on_ride" */,
    };

    class TFineUserRestrictions {
    private:
        using IOfferPtr = TAtomicSharedPtr<IOffer>;
        using IOfferPtrs = TVector<IOfferPtr>;

    private:
        static const TString DefaultTimeZoneName;
        static const TString DefaultBirthdayTagName;
        static const TString DefaultLongTermOfferGroupingTags;

    private:
        R_FIELD(i64, TotalCentsPaymentLimit, 0);
        R_FIELD(TDuration, PaymentLimitPeriod, TDuration::Days(1));

        R_FIELD(ui32, TotalChargeCountLimit, std::numeric_limits<ui32>::max());
        R_FIELD(TDuration, ChargeCountLimitPeriod, TDuration::Days(1));

        R_FIELD(bool, ChargeOnRide, false);
        R_FIELD(TString, LongTermOfferGroupingTags, DefaultLongTermOfferGroupingTags);

        R_FIELD(bool, ChargeOnBirthday, false);
        R_FIELD(TSet<TString>, BirthdayTagNames, { DefaultBirthdayTagName });

    public:
        EUserRestrictionsStatus Check(const NDrive::IServer& server, const NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& tx) const;

        static NDrive::TScheme GetScheme();

        bool DeserializeFromJson(const NJson::TJsonValue& data);
        NJson::TJsonValue SerializeToJson() const;

    private:
        bool IsTotalChargeCountLimitSet() const;

        EUserRestrictionsStatus CheckPaymentLimits(const TDriveAPI& driveApi, const NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& tx) const;
        EUserRestrictionsStatus CheckChargeCountLimits(const TDriveAPI& driveApi, const NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& tx) const;

        TMaybe<TVector<i64>> GetUserChargeAmounts(const TDriveAPI& driveApi, const TString& userId, const TDuration period, NDrive::TEntitySession& tx) const;

        EUserRestrictionsStatus CheckChargeOnBirthday(const TDriveAPI& driveApi, const NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& tx) const;

        EUserRestrictionsStatus CheckChargeOnRide(const NDrive::IServer& server, const NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& tx) const;
        bool GetSessionOffers(const NDrive::IServer& server, const TString& sessionId, IOfferPtrs& offers, NDrive::TEntitySession& tx) const;
        bool HasOfferBeenEvolved(IOfferPtr offer) const;
        bool IsOfferLongTerm(const NDrive::IServer& server, IOfferPtr offer) const;

        bool GetBirtdayTagTimestamp(const TDriveAPI& driveApi, const TString& userId, TInstant& tagAddInstant, NDrive::TEntitySession& tx, const TInstant& defaultInstant = TInstant::Zero()) const;

    private:
        TAtomicSharedPtr<TTagsFilter> LongTermOfferTagsFilterPtr;
        mutable TMap<TString, TString> UserBirthdays;
    };
};
