#pragma once

#include <drive/backend/abstract/base.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/ptr.h>

namespace NDrive {
    template <typename T>
    bool TryInitDefaultFromSettings(NDrive::IDefaultSchemeElement<T>& fieldScheme, const TString& settingKey) {
        if (!NDrive::HasServer()) {
            return false;
        }
        return TryInitDefaultFromSettings(fieldScheme, NDrive::GetServer(), settingKey);
    }

    template <typename T>
    bool TryInitDefaultFromSettings(NDrive::IDefaultSchemeElement<T>& fieldScheme, const IServerBase& server, const TString& settingKey) {
        if (!!settingKey && server.HasSettings()) {
            T value;
            if (server.GetSettings().GetValue<T>(settingKey, value)) {
                fieldScheme.SetDefault(value);
                return true;
            }
        }
        return false;
    }

    template <typename T>
    bool TryFromScheme(const NDrive::TScheme& scheme, const TString& fieldName, T& value, bool required = false) {
        if (scheme.HasField(fieldName)) {
            using TDefaultSchemeElement = NDrive::IDefaultSchemeElement<std::remove_reference_t<decltype(value)>>;
            auto schemeElementPtr = std::dynamic_pointer_cast<TDefaultSchemeElement>(scheme.Get(fieldName));
            if (schemeElementPtr != nullptr && schemeElementPtr->HasDefault()) {
                value = schemeElementPtr->GetDefaultUnsafe();
                return true;
            }
        }
        if (required) {
            ERROR_LOG << "No default value available in scheme for " << fieldName << Endl;
            return false;
        }
        return true;
    }

    template <typename T>
    bool TryCustomFromScheme(const NDrive::TScheme& scheme, const TString& fieldName, T& value, bool required = false) {
        TString serializedData;
        if (!TryFromScheme(scheme, fieldName, serializedData, /* required = */ true)) {
            return !required;
        }
        return TryFromString(serializedData, value);
    }

    template <>
    inline bool TryCustomFromScheme<NJson::TJsonValue>(const NDrive::TScheme& scheme, const TString& fieldName, NJson::TJsonValue& value, bool required) {
        TString serializedData;
        if (!TryFromScheme(scheme, fieldName, serializedData, /* required = */ true)) {
            return !required;
        }
        return NJson::ReadJsonFastTree(serializedData, &value);
    }

    template <typename T>
    bool TryCustomJsonFromScheme(const NDrive::TScheme& scheme, const TString& fieldName, T& value, bool required = false) {
        NJson::TJsonValue jsonData;
        if (!TryCustomFromScheme(scheme, fieldName, jsonData, /* required = */ true)) {
            return !required;
        }
        return NJson::TryFromJson(jsonData, value);
    }

    template <typename T>
    bool TrySerializableFromScheme(const NDrive::TScheme& scheme, const TString& fieldName, T& value, bool required = false) {
        NJson::TJsonValue jsonData;
        if (!TryCustomFromScheme(scheme, fieldName, jsonData, /* required = */ true)) {
            return !required;
        }
        return value.DeserializeFromJson(jsonData);
    }

    template <typename T>
    bool ReadFieldSchemeDefault(const NDrive::TScheme& scheme, const NJson::TJsonValue& data, const TString& fieldName, T& value, bool required = false, bool schemeRequired = false, bool skipEmpty = false) {
        if (!TryFromScheme(scheme, fieldName, value, schemeRequired)) {
            return false;
        }
        auto& field = data[fieldName];
        if (!field.IsDefined()) {
            return !required;
        }
        if (skipEmpty && !field.GetStringRobust()) {
            return true;
        }
        return TryFromJson(field, value);
    }
}
