#include <drive/backend/fines/article_matcher.h>

#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/generic/map.h>
#include <util/string/builder.h>
#include <util/system/env.h>

namespace {
    static const TString articleMatcherConfigStr = ("{"
    "\"generic_match_rule\": { \"pattern\": \"(\\\\d+(?:[ ]*\\\\.[ ]*\\\\d+)+(?:(?:[ ]*[ч|п])(?:[ ]*\\\\.[ ]*\\\\d+)+)?)\" },"
    "\"match_rules\": [ { \"name\": \"CityParking\", \"pattern\": \"платн(ой|ую)[ ]+(городск(ой|ую)[ ]+)?парковк(е|у)\", \"match_result\": \"8_14_2\" }, { \"name\": \"LawnParking\", \"pattern\": \"зелены(е|ми)[ ]+насаждени(я|ями)\", \"match_result\": \"8_25\" } ],"
    "\"readable_articles\": [ { \"article_code\": \"12_09_2\", \"article_title\": \"Превышение скорости: немножко\" }, { \"article_code\": \"12_09_3\", \"article_title\": \"Превышение скорости: неплохо так\" }, { \"article_code\": \"12_09_4\", \"article_title\": \"Превышение скорости на 60—80 км/час\" }, { \"article_code\": \"12_09_6\", \"article_title\": \"Превышение скорости: неплохо так\" }, { \"article_code\": \"12_09_7\", \"article_title\": \"Превышение скорости на 60—80 км/час\" }, { \"article_code\": \"12_12_2\", \"article_title\": \"Выезд за стоп-линию\" }, { \"article_code\": \"12_12_3\", \"article_title\": \"Проезд на красный свет\" }, { \"article_code\": \"12_15_1\", \"article_title\": \"Проезд по обочине и пересечение автомобильной или пешей колонны\" }, { \"article_code\": \"12_15_4\", \"article_title\": \"Выезд на полосу встречного движения, либо на трамвайные пути встречного направления\" }, { \"article_code\": \"12_16_1\", \"article_title\": \"Не заметили знак или разметку (или всё сразу)\" }, { \"article_code\": \"12_16_2\", \"article_title\": \"Поворот налево или разворот в неположенном месте\" }, { \"article_code\": \"12_16_4\", \"article_title\": \"Остановка или брошенная машина в запрещенной знаком или разметкой зоне\" }, { \"article_code\": \"12_16_5\", \"article_title\": \"Не заметили знак или разметку, запрещающих остановку или стоянку машины\" }, { \"article_code\": \"12_17_1_1\", \"article_title\": \"Играли в автобус или такси\" }, { \"article_code\": \"12_17_1_2\", \"article_title\": \"Играли в автобус или такси\" }, { \"article_code\": \"12_19_2\", \"article_title\": \"Брошенная машина на парковке для инвалидов\" }, { \"article_code\": \"12_19_5\", \"article_title\": \"Неправильная парковка\" }, { \"article_code\": \"12_19_6\", \"article_title\": \"Остановка или брошенная машина посреди дороги или вторым рядом\" } ],"
    "\"recognizable_articles\": [\"8_14\", \"8_14_2\", \"8_25\", \"11_21_1\", \"11_21_2\", \"11_23_1\", \"12_01_1\", \"12_02_1\", \"12_02_2\", \"12_03_1\", \"12_03_2\", \"12_03_3\", \"12_05_1\", \"12_05_2\", \"12_05_3_1\", \"12_06\", \"12_07_1\", \"12_07_2\", \"12_07_3\", \"12_08_1\", \"12_08_2\", \"12_08_3\", \"12_09_2\", \"12_09_3\", \"12_09_4\", \"12_09_5\", \"12_09_6\", \"12_09_7\", \"12_10_1\", \"12_10_2\", \"12_11_1\", \"12_11_3\", \"12_12_1\", \"12_12_2\", \"12_12_3\", \"12_13_1\", \"12_13_2\", \"12_14_1\", \"12_14_1_1\", \"12_14_2\", \"12_14_3\", \"12_15_1\", \"12_15_1_1\", \"12_15_2\", \"12_15_3\", \"12_15_4\", \"12_15_5\", \"12_16_1\", \"12_16_2\", \"12_16_3\", \"12_16_3_1\", \"12_16_4\", \"12_16_5\", \"12_17_1\", \"12_17_1_1\", \"12_17_1_2\", \"12_17_2\", \"12_18\", \"12_19_1\", \"12_19_2\", \"12_19_3\", \"12_19_3_1\", \"12_19_3_2\", \"12_19_4\", \"12_19_5\", \"12_19_6\", \"12_20\", \"12_21_2_2\", \"12_23_1\", \"12_23_3\", \"12_25_2\", \"12_26_1\", \"12_26_2\", \"12_27_1\", \"12_27_3\", \"12_28_2\", \"12_29_1\", \"12_33\", \"12_36_1\", \"12_37_1\", \"12_37_2\", \"19_03_1\", \"19_22_1\", \"20_25_1\"]"
    "}");

    static const TMap<TString, TString> articleCodeMapping = {
        { " Неуплата за размещение транспортного средства на платной городской парковке", "8_14_2" },
        { "11.21ч.1 - загрязнение полос отвода и придорожных полос автомобильных дорог, распашка земельных участков, покос травы, осуществление рубок и повреждение лесных насаждений и иных многолетних насаждений, снятие дерна и выемка грунта, за исключением работ по содержанию ", "11_21_1" },
        { "11.21ч.2 - использование водоотводных сооружений автомобильной дороги для стока или сброса вод; выполнение в границах полосы отвода автомобильной дороги, в том числе на проезжей части автомобильной дороги, работ, связанных с применением горючих веществ, а также вещес", "11_21_2" },
        { "11.23ч.1 - управление грузовым автотранспортным средством или автобусом при осуществлении международной автомобильной перевозки без контрольного устройства (тахографа) или с выключенным тахографом, а равно с незаполненными тахограммами либо без ведения регистрационны", "11_23_1" },
        { "12.02.1 - Нечитаемые, нестандартные, установленные с нарушением госзнаки", "12_02_1" },
        { "12.02.2 - Управление ТС без госзнаков", "12_02_2" },
        { "12.03.1 - Не имел при себе регистрационных документов", "12_03_1" },
        { "12.03.2 - Не имел при себе ВУ/полиса/путевого листа/товарно-трансп. док.", "12_03_2" },
        { "12.05.1 - Упр. ТС при наличии неисправностей", "12_05_1" },
        { "12.05.3.1 - Управление ТС со стеклами, светопропускание которых не соответствует требованиям тех.регламента", "12_05_3_1" },
        { "12.06 - Нар. правил пользования ремнями безопасности или мотошлемами", "12_06" },
        { "12.07.1 - Упр. ТС водителем, не имеющим права упр. ТС", "12_07_1" },
        { "12.07.3 - Передача упр. ТС лицу, не имеющему или лишенному права упр. ТС", "12_07_3" },
        { "12.08.1 - Упр. ТС водителем, находящимся в состоянии опьянения", "12_08_1" },
        { "12.09.2 - Превышение скорости движения ТС от 20 до 40 км/ч", "12_09_2" },
        { "12.09.2 - превышение скорости движения тс от 20 до 40 км/ч", "12_09_2" },
        { "12.09.3 - Превышение скорости движения ТС от 40 до 60 км/ч", "12_09_3" },
        { "12.09.4 - Превышение скорости движения ТС от 60 до 80 км/ч", "12_09_4" },
        { "12.10.2 - Нарушение правил проезда через ж/д переезды", "12_10_2" },
        { "12.10ч.1 - пересечение железнодорожного пути вне железнодорожного переезда, выезд на железнодорожный переезд при закрытом или закрывающемся шлагбауме либо при запрещающем сигнале светофора или дежурного по переезду, а равно остановка или стоянка на железнодорожном пе", "12_10_1" },
        { "12.11.1 - Движ. по автомагистрали на ТС, со скоростью менее 40 км/ч", "12_11_1" },
        { "12.11.3 - Разворот или въезд ТС в техн. разрывы разд. полосы, движ. задним ходом по автомагистрали", "12_11_3" },
        { "12.11ч.1 - движение по автомагистрали на тс, скорость которого по технической характеристике или по его состоянию менее 40 километров в час, а равно остановка тс на автомагистрали вне специальных площадок для стоянки", "12_11_1" },
        { "12.11ч.3 - разворот или въезд тс в технологические разрывы разделительной полосы на автомагистрали либо движение задним ходом по автомагистрали", "12_11_3" },
        { "12.12.1 - Проезд на запрещ. сигнал светофора или запрещ. сигнал регулировщика", "12_12_1" },
        { "12.12.2 - Неостановка перед стоп-линией", "12_12_2" },
        { "12.12.3 - Повторный проезд на запрещ.сигнал светофора (регулировщика)", "12_12_3" },
        { "12.12ч.1 - проезд на запрещающий сигнал светофора или на запрещающий жест регулировщика, за исключением случаев, предусмотренных частью 1 статьи 12.10 коап рф и частью 2 статьи 12,12", "12_12_1" },
        { "12.12ч.1 - проезд на запрещающий сигнал светофора или на запрещающий жест регулировщика, за исключением случаев, предусмотренных частью 1 статьи 12.10 настоящего кодекса и частью 2 настоящей статьи", "12_12_1" },
        { "12.12ч.2 - невыполнение требования пдд об остановке перед стоп-линией, обозначенной дорожными знаками или разметкой проезжей части дороги, при запрещающем сигнале светофора или запрещающем жесте регулировщика", "12_12_2" },
        { "12.12ч.3 - повторное совершение административного правонарушения, предусмотренного частью 1 настоящей статьи", "12_12_3" },
        { "12.13.1 - Создание препятствий для движ. ТС в поперечном направлении", "12_13_1" },
        { "12.13.2 - Непредоставл. преимущества в движ. ТС, имеющему такое право, на перекрестке", "12_13_2" },
        { "12.13ч.1 - выезд на перекресток или пересечение проезжей части дороги в случае образовавшегося затора, который вынудил водителя остановиться, создав препятствие для движения тс в поперечном направлении", "12_13_1" },
        { "12.13ч.2 - невыполнение требования пдд уступить дорогу тс, пользующемуся преимущественным правом проезда перекрестков", "12_13_2" },
        { "12.14.1 - Невыполнение треб. ПДД о подаче сигналов перед началом движ., перестроением и т.п.", "12_14_1" },
        { "12.14.1.1 - Не занял заблаговременно соответствующее крайнее положение на пр.части перед поворотом", "12_14_1_1" },
        { "12.14.2 - Разворот или движ. задним ходом в местах, где такие маневры запрещены", "12_14_2" },
        { "12.14.3 - Непредоставление преимущества в движ. ТС, имеющему такое право", "12_14_3" },
        { "12.14ч.1 - невыполнение требования пдд подать сигнал перед началом движения, перестроением, поворотом, разворотом или остановкой", "12_14_1" },
        { "12.14ч.1.1 - невыполнение требования пдд, за исключением установленных случаев, перед поворотом направо, налево или разворотом заблаговременно занять соответствующее крайнее положение на проезжей части, предназначенной для движения в данном направлении", "12_14_1_1" },
        { "12.14ч.2 - разворот или движение задним ходом в местах, где такие маневры запрещены, за исключением случаев, предусмотренных частью 3 статьи 12.11 настоящего кодекса", "12_14_2" },
        { "12.14ч.3 - невыполнение требования пдд уступить дорогу тс, пользующемуся преимущественным правом движения, за исключением случаев, предусмотренных частью 2 статьи 12.13 и статьей 12.17 настоящего кодекса", "12_14_3" },
        { "12.15.1 - Наруш.правил расп.т/с на пр.части без выезда на ст.встр.движ.,дв.по обоч.,пересечение колонны", "12_15_1" },
        { "12.15.1.1 - Невыполнение водителем тихоходного ТС требования ПДД пропустить ТС для обгона или опережения", "12_15_1_1" },
        { "12.15.2 - Движение по велосипедным или пешеходным дорожкам либо тротуарам в нарушение ПДД", "12_15_2" },
        { "12.15.3 - Выезд на ст.встр.движ. при объезде препятствия", "12_15_3" },
        { "12.15.4 - Выезд на ст.встреч.движ., либо на трамвайные пути встреч.напр.за исключением 12.15.3", "12_15_4" },
        { "12.15.5 - Повторный выезд на сторону, предназначенную для встречного движения", "12_15_5" },
        { "12.15ч.1 - нарушение правил расположения транспортного средства на проезжей части дороги, встречного разъезда, а равно движение по обочинам или пересечение организованной транспортной или пешей колонны либо занятие места в ней", "12_15_1" },
        { "12.15ч.1 - нарушение правил расположения тс на проезжей части дороги, встречного разъезда, а равно движение по обочинам или пересечение организованной транспортной или пешей колонны либо занятие места в ней", "12_15_1" },
        { "12.15ч.1.1 - невыполнение водителем тихоходного транспортного средства, транспортного средства, перевозящего крупногабаритный груз, или транспортного средства, двигающегося со скоростью, не превышающей 30 километров в час, вне населенных пунктов требования правил дорож", "12_15_1_1" },
        { "12.15ч.2 - движение по велосипедным или пешеходным дорожкам либо тротуарам в нарушение пдд", "12_15_2" },
        { "12.15ч.3 - выезд в нарушение пдд на полосу, предназначенную для встречного движения, при объезде препятствия либо на трамвайные пути встречного направления при объезде препятствия", "12_15_3" },
        { "12.15ч.4 - выезд в нарушение пдд на полосу, предназначенную для встречного движения, либо на трамвайные пути встречного направления, за исключением случаев, предусмотренных частью 3 настоящей статьи", "12_15_4" },
        { "12.15ч.4 - выезд в нарушение пдд на полосу, предназначенную для встречного движения, либо на трамвайные пути встречного направления, за исключением случаев, предусмотренных частью 3 статьи 12.15 коап рф", "12_15_4" },
        { "12.15ч.5 - повторное совершение административного правонарушения, предусмотренного ч. 4 ст. 12.15 коап рф", "12_15_5" },
        { "12.16.1 - Несоблюдение требований знаков или разметки, за искл.случаев, предусм.др.статьями гл.12", "12_16_1" },
        { "12.16.2 - Поворот налево или разворот в нарушение знаков или разметки", "12_16_2" },
        { "12.16.3 - Движение во встречном направлении по дороге с односторонним движением", "12_16_3" },
        { "12.16.3.1 - Повторное движение во встречном направлении по дороге с односторонним движением", "12_16_3_1" },
        { "12.16.4 - Несоблюдение требований дорожных знаков или разметки, запрещающими остановку или стоянку", "12_16_4" },
        { "12.16.5 - Нарушение, предусмотренное частью 4 настоящей статьи, совершенное в городе федерального значения Москве или Санкт-Петербурге", "12_16_5" },
        { "12.16.5 - Нарушение, предусмотренное.частью 4.настоящей статьи, совершенное в городе федерального значения Москве или Санкт-Петербурге", "12_16_5" },
        { "12.16.5 - Несоблюдение треб.дор.знаков или разметки, запрещ.остановку или стоянку в Москве, Санкт-Петербурге", "12_16_5" },
        { "12.16ч.1 - несоблюдение требований, предписанных дорожными знаками или разметкой проезжей части дороги, за исключением случаев, предусмотренных частями 2 и 3 настоящей статьи и другими статьями настоящей главы", "12_16_1" },
        { "12.16ч.2 - поворот налево или разворот в нарушение требований, предписанных дорожными знаками или разметкой проезжей части дороги", "12_16_2" },
        { "12.16ч.3 - движение во встречном направлении по дороге с односторонним движением", "12_16_3" },
        { "12.16ч.3.1 - повторное совершение административного правонарушения, предусмотренного ч. 3 ст. 12.16 коап рф", "12_16_3_1" },
        { "12.16ч.4 - несоблюдение требований, предписанных дорожными знаками или разметкой проезжей части дороги, запрещающими остановку или стоянку тс, за исключением случая, предусмотренного частью 5 настоящей статьи", "12_16_4" },
        { "12.16ч.5 - нарушение, предусмотренное частью 4 настоящей статьи, совершенное в городе федерального значения москве или санкт-петербурге", "12_16_5" },
        { "12.17 ч.1.1 - Движение или остановка на полосе для маршрутных ТС", "12_17_1_1" },
        { "12.17 ч.1.2 - Движение или остановка на полосе для маршрутных ТС в Москве или в Санкт-Петербурге", "12_17_1_2" },
        { "12.17.2 - Непредоставление преимущ. в движ. ТС с цветограф. схемами с вкл. пробл. маячком и звук. сигн.", "12_17_2" },
        { "12.17ч.1 - непредоставление преимущества в движении маршрутному тс, а равно тс с одновременно включенными проблесковым маячком синего цвета и специальным звуковым сигналом", "12_17_1" },
        { "12.17ч.1.1 - движение транспортных средств по полосе для маршрутных транспортных средств или остановка на указанной полосе в нарушение правил дорожного движения", "12_17_1_1" },
        { "12.17ч.1.2 - движение транспортных средств по полосе для маршрутных транспортных средств или остановка на указанной полосе в нарушение правил дорожного движения", "12_17_1_2" },
        { "12.18 - Непредоставление преимущества в движ. пешеходам, велосипедистам, имеющим такое право", "12_18" },
        { "12.18 - невыполнение требования пдд уступить дорогу пешеходам, велосипедистам или иным участникам дорожного движения (за исключением водителей тс), пользующимся преимуществом в движении", "12_18" },
        { "12.19 ч.1 - Нарушение правил стоянки, остановки", "12_19_1" },
        { "12.19 ч.5 - Нарушение правил стоянки, остановки в Москве или в Санкт-Петербурге", "12_19_5" },
        { "12.19 ч.6 - Стоянка, остановка ТС на проезжей части, либо далее 1-й полосы в Москве или в Санкт-Петербурге", "12_19_6" },
        { "12.19.2 - Нар. правил остановки или стоянки в местах, отведенных для ТС инвалидов", "12_19_2" },
        { "12.19ч. 6", "12_19_6" },
        { "12.19ч.1 - другие нарушения правил остановки или стоянки транспортных средств", "12_19_1" },
        { "12.19ч.2 - нарушение правил остановки или стоянки тс в местах, отведенных для остановки или стоянки тс инвалидов", "12_19_2" },
        { "12.19ч.3 - остановка или стоянка тс на пешеходном переходе, за исключением вынужденной остановки, либо нарушение правил остановки или стоянки тс на тротуаре, повлекшее создание препятствий для движения пешеходов", "12_19_3" },
        { "12.19ч.3.1 - остановка или стоянка транспортных средств в местах остановки маршрутных транспортных средств или стоянки легковых такси либо ближе 15 метров от мест остановки маршрутных транспортных средств или стоянки легковых такси, за исключением остановки для посадки", "12_19_3_1" },
        { "12.19ч.3.2 - остановка или стоянка транспортных средств на трамвайных путях либо остановка или стоянка транспортных средств далее первого ряда от края проезжей части, за исключением вынужденной остановки", "12_19_3_2" },
        { "12.19ч.4 - нарушение правил остановки или стоянки тс на проезжей части, повлекшее создание препятствий для движения других тс, а равно остановка или стоянка тс в тоннеле", "12_19_4" },
        { "12.19ч.5 - другие нарушения правил остановки или стоянки транспортных средств", "12_19_5" },
        { "12.19ч.6 - остановка или стоянка тс на пешеходном переходе, за исключением вынужденной остановки, либо нарушение правил остановки или стоянки тс на тротуаре, повлекшее создание препятствий для движения пешеходов", "12_19_6" },
        { "12.1ч.1 - управление тс, не зарегистрированным в установленном порядке", "12_01_1" },
        { "12.20 - Нар. правил пользования внешними свет. приборами, звук. сигналами и т.п.", "12_20" },
        { "12.20 - нарушение правил пользования внешними световыми приборами, звуковыми сигналами, аварийной сигнализацией или знаком аварийной остановки", "12_20" },
        { "12.21.2ч.2 - нарушение правил перевозки опасных грузов, за исключением случаев, предусмотренных частью 1 настоящей статьи", "12_21_2_2" },
        { "12.23.1 - Нарушение правил перевозки людей", "12_23_1" },
        { "12.23.3 - Нарушение правил перевозки детей", "12_23_3" },
        { "12.23ч.1 - нарушение правил перевозки людей, за исключением случаев, предусмотренных частью 2 настоящей статьи", "12_23_1" },
        { "12.23ч.3 - нарушение требований к перевозке детей, установленных пдд", "12_23_3" },
        { "12.23ч.3 - нарушение требований к перевозке детей, установленных правилами дорожного движения", "12_23_3" },
        { "12.25.2 - Невыполнение требования сотрудника полиции об остановке ТС", "12_25_2" },
        { "12.25ч.2 - невыполнение законного требования сотрудника полиции об остановке тс", "12_25_2" },
        { "12.26.1 - Отказ водит. от мед.освидет. на состояние опьянения.", "12_26_1" },
        { "12.26.2 - Отказ от мед.освидет. на состояние опьянения водит., не имеющ. или лиш.права управл. ТС", "12_26_2" },
        { "12.26ч.1 - невыполнение водителем законного требования сотрудника полиции о прохождении медицинского освидетельствования на состояние опьянения", "12_26_1" },
        { "12.26ч.1 - невыполнение водителем законного требования сотрудника полиции о прохождении медицинского освидетельствования на состояние опьянения или невыполнение водителем тс вооруженных сил российской федерации, внутренних войск министерства внутренних дел российской", "12_26_1" },
        { "12.26ч.2 - невыполнение водителем, не имеющим права управления тс либо лишенным права управления тс, законного требования сотрудника полиции о прохождении медицинского освидетельствования на состояние опьянения или невыполнение не имеющим права управления тс либо лиш", "12_26_2" },
        { "12.27.1 - Невып. водителем обязанностей в связи с ДТП, участником которого он является", "12_27_1" },
        { "12.27ч.1 - невыполнение водителем обязанностей, предусмотренных пдд, в связи с дорожно-транспортным происшествием, участником которого он является, за исключением случаев, предусмотренных частью 2 настоящей статьи", "12_27_1" },
        { "12.27ч.3 - невыполнение требования пдд о запрещении водителю употреблять алкогольные напитки, наркотические или психотропные вещества после дорожно-транспортного происшествия, к которому он причастен, либо после того, как тс было остановлено по требованию сотрудника ", "12_27_3" },
        { "12.27ч.3 - невыполнение требования пдд о запрещении водителю употреблять алкогольные напитки, наркотические или психотропные вещества после дтп, к которому он причастен, либо после  того, как транспортное средство было остановлено по требованию сотрудника полиции, до", "12_27_3" },
        { "12.28 ч.2 - Нарушение правил движения ТС в жилих зонах г.Москва или г.Санкт-Петербург", "12_28_2" },
        { "12.28ч.2 - нарушение правил, установленных для движения тс в жилых зонах", "12_28_2" },
        { "12.29.1 - Нарушение ПДД пешеходом или пассажиром ТС", "12_29_1" },
        { "12.29ч.1 - нарушение пешеходом или пассажиром тс пдд", "12_29_1" },
        { "12.2ч.1 - управление тс с нечитаемыми, нестандартными или установленными с нарушением требований государственного стандарта государственными регистрационными знаками, за исключением случаев, предусмотренных частью 2 настоящей статьи", "12_02_1" },
        { "12.2ч.2 - предупреждение или штраф 500 руб.", "12_02_2" },
        { "12.33 - Повреждение дорог, ж/д переездов и т.п., умышленное созд. помех в дорожном движ.", "12_33" },
        { "12.33 - повреждение дорог, железнодорожных переездов или других дорожных сооружений либо технических средств организации дорожного движения, которое создает угрозу безопасности дорожного движения, а равно умышленное создание помех в дорожном движении, в том числе ", "12_33" },
        { "12.36.1 - Польз.водит.во время движ.т/с телеф.,не оборуд.тех.устр.,позв.вести переговоры без исп.рук", "12_36_1" },
        { "12.36.1 - пользование водителем во время движения тс телефоном, не оборудованным техническим устройством, позволяющим вести переговоры без использования рук", "12_36_1" },
        { "12.37.1 - Упр. ТС с нарушением условий полиса ОСАГО", "12_37_1" },
        { "12.37.2 - Упр. ТС без ОСАГО", "12_37_2" },
        { "12.37ч.1 - управление тс в период его использования, не предусмотренный страховым полисом обязательного страхования гражданской ответственности владельцев тс, а равно управление тс с нарушением предусмотренного данным страховым полисом условия управления этим тс толь", "12_37_1" },
        { "12.37ч.2 - неисполнение владельцем тс установленной федеральным законом обязанности по страхованию своей гражданской ответственности, а равно управление тс, если такое обязательное страхование заведомо отсутствует", "12_37_2" },
        { "12.3ч.1 - управление тс водителем, не имеющим при себе регистрационных документов на тс, а в установленных случаях документов, предусмотренных таможенным законодательством таможенного союза, с отметками таможенных органов, подтверждающими временный ввоз тс", "12_03_1" },
        { "12.3ч.2 - управление транспортным средством водителем, не имеющим при себе талона о прохождении государственного технического осмотра,  страхового полиса осаго, за исключением случая, предусмотренного частью 2 статьи 12.37 коап рф, а в случаях, предусмотренных закон", "12_03_2" },
        { "12.3ч.2 - управление тс водителем, не имеющим при себе документов на право управления им, страхового полиса обязательного страхования гражданской ответственности владельцев тс, за исключением случая, предусмотренного частью 2 статьи 12.37 настоящего кодекса, а в слу", "12_03_2" },
        { "12.3ч.3 - передача управления тс лицу, не имеющему при себе документов на право управления им", "12_03_3" },
        { "12.5ч.1 - управление транспортным средством при наличии неисправностей или условий, при которых в соответствии с оп, за исключением неисправностей и условий, указанных в частях 2-6 статьи 12.5 коап рф", "12_05_1" },
        { "12.5ч.1 - управление тс при наличии неисправностей или условий, при которых в соответствии с основными положениями по допуску тс к эксплуатации и обязанностями должностных лиц по обеспечению безопасности дорожного движения эксплуатация тс запрещена, за исключением н", "12_05_1" },
        { "12.5ч.2 - управление тс с заведомо неисправными тормозной системой (за исключением стояночного тормоза), рулевым управлением или сцепным устройством (в составе поезда)", "12_05_2" },
        { "12.5ч.3.1 - управление тс, на котором установлены стекла (в том числе покрытые прозрачными цветными пленками), светопропускание которых не соответствует требованиям технического регламента о безопасности колесных тс", "12_05_3_1" },
        { "12.6 - управление транспортным средством водителем, не пристегнутым ремнем безопасности, перевозка пассажиров, не пристегнутых ремнями безопасности, если конструкцией транспортного средства предусмотрены ремни безопасности, а равно управление мотоциклом либо пере", "12_06" },
        { "12.6 - управление тс водителем, не пристегнутым ремнем безопасности, перевозка пассажиров, не пристегнутых ремнями безопасности, если конструкцией тс предусмотрены ремни безопасности, а равно управление мотоциклом или мопедом либо перевозка на мотоцикле пассажиро", "12_06" },
        { "12.7ч.1 - управление тс водителем, не имеющим права управления тс (за исключением учебной езды)", "12_07_1" },
        { "12.7ч.2 - управление тс водителем, лишенным права управления тс", "12_07_2" },
        { "12.7ч.3 - передача управления тс лицу, заведомо не имеющему права управления тс (за исключением учебной езды) или лишенному такого права", "12_07_3" },
        { "12.8ч.1 - управление транспортным средством водителем, находящимся в состоянии опьянения", "12_08_1" },
        { "12.8ч.1 - управление тс водителем, находящимся в состоянии опьянения.", "12_08_1" },
        { "12.8ч.2 - передача управления тс лицу, находящемуся в состоянии опьянения", "12_08_2" },
        { "12.8ч.3 - управление тс водителем, находящимся в состоянии опьянения и не имеющим права управления тс либо лишенным права управления тс", "12_08_3" },
        { "12.9.5 - Превышение скорости более чем на 80 км/час", "12_09_5" },
        { "12.9.6 - Повторное превышение скорости на 40-60 км/час", "12_09_6" },
        { "12.9.7 - Повторное превышение скорости на 60-80 км/час и более 80 км/ч", "12_09_7" },
        { "12.9ч.2 - превышение установленной скорости движения транспортного средства на величину более 20, но не более 40 километров в час", "12_09_2" },
        { "12.9ч.3 - превышение установленной скорости движения тс на величину более 40, но не более 60 километров в час", "12_09_3" },
        { "12.9ч.4 - превышение установленной скорости движения тс на величину более 60 километров в час", "12_09_4" },
        { "12.9ч.5 - превышение установленной скорости движения транспортного средства на величину более 80 километров в час", "12_09_5" },
        { "12.9ч.6 - повторное совершение административного правонарушения, предусмотренного частью 3 настоящей статьи", "12_09_6" },
        { "12.9ч.7 - повторное совершение административного правонарушения, предусмотренного частями 4 и 5 настоящей статьи", "12_09_7" },
        { "19.03.1 - Неповиновение треб. сотр.полиции при исполн. ими обязан. по охр.общ.пор. и обесп.общ.безопасн.", "19_03_1" },
        { "19.22ч.1 - нарушение правил государственной регистрации транспортных средств всех видов (за исключением морских судов и судов смешанного (река - море) плавания), механизмов и установок в случае, если регистрация обязательна", "19_22_1" },
        { "19.3ч.1 - неповиновение законному распоряжению или требованию сотрудника полиции, военнослужащего либо сотрудника органа или учреждения уголовно-исполнительной системы в связи с исполнением ими обязанностей по охране общественного порядка и обеспечению общественной ", "19_03_1" },
        { "20.25ч.1 - неуплата административного штрафа в срок, предусмотренный настоящим кодексом.", "20_25_1" },
        { "8.14ч.2 - коап г. москвы. неуплата за размещение транспортного средства на платной городской парковке", "8_14_2" },
        { "8.25 - коап г. москвы. размещение транспортных средств на газоне или иной территории, занятой зелеными насаждениями", "8_25" },
        { "ЗГМ45-8.14 п.2 Неуплата или нарушение порядка оплаты за платную парковку г.Москвы", "8_14_2" },
        { "ЗГМ45-8.25 Размещение ТС на газоне или иной территории, занятой зелеными насаждениями", "8_25" },
        { "КоАП 12.12 ч.2, за пересечение стоп-линии на светофоре", "12_12_2" },
        { "КоАП 12.12 ч.3, за повторный проезд на красный или жёлтый", "12_12_3" },
        { "КоАП 12.13 ч.2, не уступили дорогу на перекрёстке", "12_13_2" },
        { "КоАП 12.14 ч.2, за запрещённый разворот или проезд задним ходом", "12_14_2" },
        { "КоАП 12.14 ч.3, не уступили дорогу авто с преимуществом", "12_14_3" },
        { "КоАП 12.15 ч.1, за проезд по обочине или нарушение при встречном разъезде", "12_15_1" },
        { "КоАП 12.16 ч.1, за игнорирование знаков или разметки", "12_16_1" },
        { "КоАП 12.16 ч.2, за запрещённый разворот или поворот налево", "12_16_2" },
        { "КоАП 12.16 ч.4, за остановку или стоянку в запрещённом месте", "12_16_4" },
        { "КоАП 12.16 ч.5, за остановку или стоянку в запрещённом месте", "12_16_5" },
        { "КоАП 12.17 ч.1.2, за выезд на выделенную полосу", "12_17_1_2" },
        { "КоАП 12.18, не уступили пешеходу или велосипедисту", "12_18" },
        { "КоАП 12.19 ч.2, за остановку, стоянку на месте для инвалидов", "12_19_2" },
        { "КоАП 12.23 ч.3, нарушили правила перевозки людей", "12_23_3" },
        { "КоАП 12.29 ч.1, нарушили ПДД для пешеходов и пассажиров", "12_29_1" },
        { "КоАП 12.3 ч.2, за вождение без прав и полиса ОСАГО на руках", "12_03_2" },
        { "КоАП 12.6, не пристегнули ремень или не надели шлем", "12_06" },
        { "КоАП 12.9 ч.2, за превышение на 20-40 км/ч", "12_09_2" },
        { "КоАП 12.9 ч.3, за превышение на 40-60 км/ч", "12_09_3" },
        { "КоАП 12.9 ч.6, за повторное превышение на 40-60 км/ч", "12_09_6" },
        { "КоАП 12.9 ч.7, за повторное превышение на 60 км/ч и больше", "12_09_7" },
        { "Размещение транспортных средств на территории, занятой зелеными насаждениями", "8_25" },
    };

    NDrive::NFine::TFineArticleMatcherConfig GetArticleMatcherConfig() {
        NDrive::NFine::TFineArticleMatcherConfig config;
        UNIT_ASSERT(config.DeserializeFromJson(NJson::ReadJsonFastTree(articleMatcherConfigStr)));
        return config;
    }

    NDrive::NFine::TFineArticleMatcher GetArticleMatcher() {
        return NDrive::NFine::TFineArticleMatcher(GetArticleMatcherConfig());
    }
}

Y_UNIT_TEST_SUITE(ArticleMatcher) {
    Y_UNIT_TEST(DetermineArticleCode) {
        auto articleMatcher = GetArticleMatcher();

        for (auto&& [article, expectedArticleCode] : articleCodeMapping) {
            TString actualArticleCode;
            UNIT_ASSERT_C(articleMatcher.DetermineArticle(article, actualArticleCode), "Error processing article \"" << article << "\"");
            UNIT_ASSERT_STRINGS_EQUAL(expectedArticleCode, actualArticleCode);
        }
    }
}
