#include "fueling_manager.h"

#include <drive/backend/areas/areas.h>
#include <drive/backend/common/localization.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/mediator/global_notifications/system_status.h>
#include <library/cpp/string_utils/quote/quote.h>
#include <library/cpp/threading/named_lock/named_lock.h>

#include <rtline/library/unistat/cache.h>

#include <util/folder/path.h>
#include <util/generic/guid.h>
#include <util/stream/file.h>
#include <util/string/join.h>
#include <util/string/split.h>
#include <util/string/subst.h>

TString TFuelingManager::GetHRFuel(const EFuelType fType, const TString& prefix) {
    switch (fType) {
        case EFuelType::A100: return prefix + "АИ-100";
        case EFuelType::A100Premium: return prefix + "АИ-100 Премиум";
        case EFuelType::A80: return prefix + "АИ-80";
        case EFuelType::A92: return prefix + "АИ-92";
        case EFuelType::A92Premium: return prefix + "АИ-92 Премиум";
        case EFuelType::A95: return prefix + "АИ-95";
        case EFuelType::A95Premium: return prefix + "АИ-95 Премиум";
        case EFuelType::A98: return prefix + "АИ-98";
        case EFuelType::A98Premium: return prefix + "АИ-98 Премиум";
        case EFuelType::Diesel: return "дизель";
        case EFuelType::DieselPremium: return "дизель премиум";
        case EFuelType::DieselWinter: return "дизель зимний";
        case EFuelType::DieselDemiseason: return "дизель демисезонный";
        case EFuelType::Metan: return "газ метан";
        case EFuelType::Propane: return "газ пропан";
        case EFuelType::Undefined:
        default:
            return "";
    };
    return "";
}

NJson::TJsonValue TColumn::SerializeToJson(const TSet<EFuelType>* fTypes) const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    result.InsertValue("id", Id);
    NJson::TJsonValue& fTypesJson = result.InsertValue("ft", NJson::JSON_ARRAY);
    bool hasFuelTypes = false;
    for (auto&& i : Fuels) {
        if (!fTypes || fTypes->empty() || fTypes->contains(i)) {
            hasFuelTypes = true;
            fTypesJson.AppendValue(::ToString(i));
        }
    }
    return hasFuelTypes ? result : NJson::JSON_UNDEFINED;
}

bool TColumn::DeserializeFromJson(const NJson::TJsonValue& json) {
    const NJson::TJsonValue::TArray* arr;
    if (!json["fuels"].GetArrayPointer(&arr)) {
        return false;
    }
    for (auto&& i : *arr) {
        if (!i.IsString()) {
            return false;
        }
        EFuelType fType;
        if (!TryFromString(i.GetString(), fType)) {
            continue;
        }
        Fuels.emplace(fType);
    }
    return true;
}

NJson::TJsonValue TStation::GetJsonMapReport(const TSet<EFuelType>* fTypes) const {
    NJson::TJsonValue result;
    result.InsertValue("location", Location.ToString());
    result.InsertValue("name", Name);
    result.InsertValue("id", Id);
    result.InsertValue("address", Address);
    result.InsertValue("post_pay", PostPay);
    NJson::TJsonValue& columns = result.InsertValue("columns", NJson::JSON_ARRAY);
    TSet<EFuelType> fuels;
    bool hasColumns = false;
    for (auto&& column : Columns) {
        auto columnJson = column.SerializeToJson(fTypes);
        if (columnJson.IsDefined()) {
            columns.AppendValue(std::move(columnJson));
            hasColumns = true;
            for (auto type : column.GetFuels()) {
                if (!fTypes || fTypes->contains(type)) {
                    fuels.emplace(type);
                }
            }
        }
    }
    NJson::InsertField(result, "fuel_types", fuels);
    return hasColumns ? result : NJson::JSON_UNDEFINED;
}

bool TStation::DeserializeFromJson(const NJson::TJsonValue& json) {
    JREAD_STRING(json, "name", Name);
    JREAD_STRING(json, "id", Id);
    if (!json.Has("location")) {
        return false;
    }
    if (!Location.DeserializeLatLonFromJson(json["location"])) {
        return false;
    }
    if (!TGeoCoord::DeserializeLatLonVectorFromJson(json["polygon"], Area)) {
        return false;
    }
    if (Area.empty()) {
        return false;
    }
    if (Area.front().GetLengthTo(Area.back()) > 1e-5) {
        Area.emplace_back(Area.front());
    }
    if (json.Has("address")) {
        Address = json["address"].GetStringRobust();
    }
    if (json.Has("columns")) {
        const NJson::TJsonValue::TMapType* jsonMap;
        if (!json["columns"].GetMapPointer(&jsonMap)) {
            return false;
        }
        for (auto&& i : *jsonMap) {
            TColumn column(i.first);
            if (!column.DeserializeFromJson(i.second)) {
                return false;
            }
            Columns.emplace_back(std::move(column));
        }
        const auto predSort = [](const TColumn& l, const TColumn& r) {
            int lInt = FromStringWithDefault(l.GetId(), -1000000);
            int rInt = FromStringWithDefault(r.GetId(), -1000000);
            if (lInt == rInt) {
                return l.GetId() < r.GetId();
            } else {
                return lInt < rInt;
            }
        };
        std::sort(Columns.begin(), Columns.end(), predSort);
    }
    PostPay = json.Has("postPayPolling") && json["postPayPolling"].GetBooleanRobust();
    return true;
}

bool TStationWithRect::operator==(const TStationWithRect& item) const {
    return StationId == item.StationId;
}

bool TStationWithRect::operator<(const TStationWithRect& item) const {
    return StationId < item.StationId;
}

bool TFuelingManager::Process(IMessage* message) {
    const NDrive::TCacheRefreshMessage* dropCache = dynamic_cast<const NDrive::TCacheRefreshMessage*>(message);
    if (dropCache) {
        if (dropCache->GetComponents().empty() || dropCache->GetComponents().contains("fueling_manager")) {
            INFO_LOG << "Refresh Cache for fueling_manager" << Endl;
            Refresh();
            INFO_LOG << "Refresh Cache for fueling_manager OK" << Endl;
        }
        return true;
    }
    return false;
}

TString TFuelingManager::Name() const {
    return "fueling_manager_cache_" + ToString(Now().GetValue()) + "_" + ToString(RandomNumber<ui64>());
}

bool TFuelingManager::GetStartFailIsProblem() const {
    return false;
}

void TFuelingManager::DoWaitPeriod(const TInstant start) {
    bool isEmpty = false;
    {
        TReadGuard wg(Mutex);
        isEmpty = Polylines.empty();
    }
    if (!isEmpty) {
        IAutoActualization::DoWaitPeriod(start);
    } else {
        Sleep(TDuration::Seconds(5));
    }
}

TVector<TFuelingManager::TStationAddress> TFuelingManager::GetStationsAddresses() const {
    TReadGuard wg(Mutex);
    TVector<TStationAddress> result;
    for (auto&& stationId : NContainer::Keys(Polylines)) {
        TStation station;
        if (Stations.GetData(stationId, station) == TStations::ECachedValue::HasData) {
            result.push_back({stationId, station.GetAddress()});
        }
    }
    return result;
}

bool TFuelingManager::GetStationInfo(const TString& stationId, TStation& result) const {
    TStation station;
    if (Stations.GetData(stationId, station) == TStations::ECachedValue::HasData) {
        result = station;
        return true;
    }
    return false;
}

bool TFuelingManager::GetStationInfo(const TGeoCoord& c, TStation& result) const {
    TReadGuard wg(Mutex);
    TSet<TStationWithRect> objects;
    StationsRectHash.FindObjects(::TRect<TGeoCoord>(c), objects);
    for (auto&& i : objects) {
        auto itPolyline = Polylines.find(i.GetStationId());
        if (itPolyline == Polylines.end()) {
            continue;
        }
        if (itPolyline->second.IsPointInternal(c)) {
            TStation station;
            if (Stations.GetData(i.GetStationId(), station) == TStations::ECachedValue::HasData) {
                result = station;
                return true;
            }
        }
    }
    return false;
}

EStationSearchResult TFuelingManager::GetStationInfo(const TGeoRect& carRect, TStation& result) const {
    TReadGuard wg(Mutex);
    TSet<TStationWithRect> objects;
    StationsRectHash.FindObjects(carRect, objects);
    bool hasResult = false;
    for (auto&& i : objects) {
        auto itPolyline = Polylines.find(i.GetStationId());
        if (itPolyline == Polylines.end()) {
            continue;
        }
        TStation station;
        if (Stations.GetData(i.GetStationId(), station) == TStations::ECachedValue::HasData) {
            if (hasResult) {
                return EStationSearchResult::MoreThanOneStation;
            }
            result = station;
            hasResult = true;
        }
    }
    return hasResult ? EStationSearchResult::Ok : EStationSearchResult::NotInStation;
}

NJson::TJsonValue TFuelingManager::GetJsonMapReport(const TSet<EFuelType>* fTypes, const bool withPostPay, const TMaybe<TGeoRect>& geoFilter) const {
    NJson::TJsonValue result;
    NJson::TJsonValue& stationsJson = result.InsertValue("stations", NJson::JSON_ARRAY);
    for (auto&& i : Polylines) {
        if (geoFilter) {
            TGeoRect stationRect;
            if (!i.second.GetRect(stationRect) || !stationRect.Cross(*geoFilter)) {
                continue;
            }
        }
        TStation station;
        if (Stations.GetData(i.first, station) == TStations::ECachedValue::HasData) {
            if (!withPostPay && station.GetPostPay()) {
                continue;
            }
            NJson::TJsonValue jsonStationInfo = station.GetJsonMapReport(fTypes);
            if (jsonStationInfo.IsDefined()) {
                stationsJson.AppendValue(std::move(jsonStationInfo));
            }
        }
    }
    return result;
}

bool TFuelingManager::Refresh() {
    NNeh::THttpRequest request;
    request.SetUri("/api/station").SetCgiData("apikey=" + Config.GetApiKey() + "&withPostPayPolling=True");
    request.AddHeader("X-Robot", "Drive");
    TSignalGuardLast gSignal("fueling_manager", "refresh", 1, 0);
    NUtil::THttpReply reply = NehAgent->SendMessageSync(request, Config.GetReaskConfig().CalcRequestDeadline());
    TUnistatSignalsCache::SignalAdd("fueling-api-station", ::ToString(reply.Code()), 1);
    if (reply.Code() != 200) {
        TUnistatSignalsCache::SignalAdd("fueling_manager", "fails", 1);
        WARNING_LOG << reply.GetDebugReply() << Endl;
        return false;
    }
    NJson::TJsonValue json;
    if (!NJson::ReadJsonFastTree(reply.Content(), &json)) {
        TUnistatSignalsCache::SignalAdd("fueling_manager", "fails", 1);
        ALERT_LOG << reply.Content() << Endl;
        return false;
    }
    const NJson::TJsonValue::TArray* arr;
    if (!json.GetArrayPointer(&arr)) {
        TUnistatSignalsCache::SignalAdd("fueling_manager", "fails", 1);
        ALERT_LOG << reply.Content() << Endl;
        return false;
    }
    if (!Server) {
        TUnistatSignalsCache::SignalAdd("fueling_manager", "fails", 1);
        ERROR_LOG << "Server undefined" << Endl;
        return false;
    }
    for (auto&& i : *arr) {
        TStation station;
        if (!station.DeserializeFromJson(i)) {
            TUnistatSignalsCache::SignalAdd("fueling_manager", "failed_station_parsing", 1);
            WARNING_LOG << i << Endl;
            continue;
        }
        TStation stationCached;
        if (Stations.GetData(station.GetId(), stationCached) == TStations::ECachedValue::NoInfo) {
            if (Server->GetDriveDatabase().GetAreaManager().CheckGeoTags(station.GetLocation(), Config.GetAvailableAreaTags(), TInstant::Zero()).empty()) {
                continue;
            }
            TWriteGuard wg(Mutex);
            StationsRectHash.AddObject(TStationWithRect(station.GetId(), ::TRect<TGeoCoord>(station.GetLocation(), station.GetArea())));
            Polylines.emplace(station.GetId(), station.GetArea());
        }
        Stations.Refresh(station.GetId(), station);
    }
    TUnistatSignalsCache::SignalAdd("fueling_manager", "success", 1);
    return true;
}

TString TFuelingManager::GetFuelingTypeMessage(const EFuelType fType) const {
    return "Вам нужен " + GetHRFuel(fType);
}

TString TFuelingManager::GetFuelingTypeMessage(const EFuelType fType, const double litres) const {
    return "Мне " + NDrive::TLocalization::FuelTankVolumeValue(litres) + " " + GetHRFuel(fType, "бензин ") + ", оплата через приложение, пожалуйста";
}

TString TFuelingManager::GenerateOrderId() {
    TString guid = CreateGuidAsString();
    SubstGlobal(guid, "-", "");
    return guid;
}

bool TFuelingManager::CreateOrder(const TOrderInfo& context, TMessagesCollector& errors) const {
    if (!HasFuelClientKey(context.ClientType)) {
        errors.AddMessage(__LOCATION__, TStringBuilder()
            << "Incorrect fuel keys configuration '" << context.OrderId << "'"
            << " with client type '" << context.ClientType << "'");
        return false;
    }
    NNeh::THttpRequest request;
    request.SetUri("/api/order/create?").SetRequestType("POST");
    request.AddHeader("Content-Type", "application/x-www-form-urlencoded");
    request.AddHeader("Accept", "*/*");
    request.AddHeader("X-Robot", "Drive");
    TString postData = "apikey=" + Config.GetApiKey(context.ClientType)
        + "&stationId=" + context.StationId
        + "&userId=" + context.CarId
        + "&deviceId=drive&userAgent=drive-agent&orderType=Liters"
        + "&orderVolume=" + ToString(std::round(context.FuelInfo.Liters * 100) / 100)
        + "&columnId=" + context.ColumnId
        + "&fuelId=" + ToString(context.FuelInfo.FuelId)
        + "&orderId=" + context.OrderId;
    if (context.Coord) {
        postData += "&lat=" + ToString(context.Coord->Y) + "&lon=" + ToString(context.Coord->X);
    }
    request.SetPostData(postData);
    NUtil::THttpReply reply = NehAgent->SendMessageSync(request, Config.GetReaskConfig().CalcRequestDeadline());
    TUnistatSignalsCache::SignalAdd("fueling-api-create", ::ToString(reply.Code()), 1);
    if (reply.Code() != 200) {
        errors.AddMessage(__LOCATION__, TStringBuilder()
            << "Fail to create order '" << context.OrderId << "'"
            << " with code " << reply.Code()
            << " and error '" << reply.ErrorMessage() << "'");
        ERROR_LOG << reply.GetDebugReply() << Endl;
        return false;
    }
    NJson::TJsonValue json;
    if (!NJson::ReadJsonFastTree(reply.Content(), &json)) {
        errors.AddMessage(__LOCATION__, TStringBuilder()
            << "Fail to parse order creation '" << context.OrderId << "'"
            << " with reply '" << reply.Content() << "'");
        ERROR_LOG << reply.GetDebugReply() << Endl;
        return false;
    }
    if (!json["id"].IsString()) {
        errors.AddMessage(__LOCATION__, TStringBuilder()
            << "Fail to parse order creation '" << context.OrderId << "'"
            << " with reply '" << reply.Content() << "'");
        ERROR_LOG << reply.GetDebugReply() << Endl;
        return false;
    }
    if (context.OrderId != json["id"].GetString()) {
        errors.AddMessage(__LOCATION__, TStringBuilder()
            << "Got wrong order id '" << context.OrderId << "'"
            << " from reply '" << json["id"].GetString() << "'");
        return false;
    };
    DEBUG_LOG << "FUELING ORDER: " << context.OrderId << Endl;
    return true;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TFuelingManager::TStatusInfo& result) {
    return NJson::ParseField(value, "fuelId", NJson::Stringify(result.FuelId), true) && NJson::ParseField(value, "litre", result.Liters, true);
}

EFuelingStatus TFuelingManager::GetPostStatus(const TString& stationId, const TString& columnId, TMaybe<TFuelingManager::TStatusInfo>& statusInfo, const EFuelClientType clientType, TMessagesCollector& errors) const {
    NNeh::THttpRequest request;
    request.SetUri("/api/order/post/status").SetCgiData("apikey=" + Config.GetApiKey(clientType) + "&stationId=" + stationId + "&columnId=" + columnId);
    request.AddHeader("X-Robot", "Drive");
    NUtil::THttpReply reply = NehAgent->SendMessageSync(request, Config.GetReaskConfig().CalcRequestDeadline());
    TUnistatSignalsCache::SignalAdd("fueling-api-post-status", ::ToString(reply.Code()), 1);
    EFuelingStatus result = EFuelingStatus::ServerProblems;
    INFO_LOG << reply.Code() << " " << reply.Content() << Endl;
    if (reply.Code() != 200) {
        errors.AddMessage(__LOCATION__, TStringBuilder()
            << "Fail to get column status '" << columnId << "'"
            << " from station '" << stationId << "'"
            << " with code " << reply.Code()
            << " and error '" << reply.ErrorMessage() << "'");
        ERROR_LOG << reply.GetDebugReply() << Endl;
        return result;
    }
    NJson::TJsonValue replyJson;
    if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
        errors.AddMessage(__LOCATION__, TStringBuilder()
            << "Fail to parse column status '" << columnId << "'"
            << " from station '" << stationId << "'"
            << " with content '" << reply.Content() << "'");
        ERROR_LOG << reply.GetDebugReply() << Endl;
        return result;
    }
    if (!TryFromString(replyJson["status"].GetString(), result)) {
        errors.AddMessage(__LOCATION__, TStringBuilder()
            << "Fail to get parsed column status '" << columnId << "'"
            << " from station '" << stationId << "'"
            << " with content '" << reply.Content() << "'");
        ERROR_LOG << reply.GetDebugReply() << Endl;
        return result;
    }
    statusInfo = NJson::TryFromJson<TFuelingManager::TStatusInfo>(replyJson);
    if (result == EFuelingStatus::Completed) {
        return statusInfo ? EFuelingStatus::FuelingCompleted : EFuelingStatus::Fueling;
    }
    return result;
}

class TParsedFuelingResult {
    R_OPTIONAL(EFuelType, FuelType);
    R_OPTIONAL(double, Liters);
    R_OPTIONAL(double, Sum);

public:
    bool CheckData() {
        return FuelType || Liters || Sum;
    }

    bool SerializeFromJson(const NJson::TJsonValue& json) {
        return NJson::ParseField(json, "sumPaid", Sum, /* required = */ false)
            && NJson::ParseField(json, "litreCompleted", Liters, /* required = */ false)
            && (!json.Has("fuel") || NJson::ParseField(json, "marka", FuelType, /* required = */ false))
            && CheckData();
    }

    TString GetStringReport() {
        if (!CheckData()) {
            return "";
        }
        auto report = TStringBuilder() << "Заправлено:";
        if (FuelType) {
            report << " " << *FuelType;
        }
        if (Liters) {
            report << " " << std::round(*Liters * 100) / 100 << " л";
        }
        if (Sum) {
            report << " на сумму " << std::round(*Sum * 100) / 100;
        }
        return report;
    }

};

EFuelingStatus TFuelingManager::GetStatus(const TString& orderId, TString& comment, const EFuelClientType clientType, TMessagesCollector& errors) const {
    NNeh::THttpRequest request;
    request.SetUri("/api/order/status").SetCgiData("apikey=" + Config.GetApiKey(clientType) + "&orderId=" + orderId);
    request.AddHeader("X-Robot", "Drive");
    NUtil::THttpReply reply = NehAgent->SendMessageSync(request, Config.GetReaskConfig().CalcRequestDeadline());
    TUnistatSignalsCache::SignalAdd("fueling-api-status", ::ToString(reply.Code()), 1);
    EFuelingStatus result = EFuelingStatus::ServerProblems;
    if (reply.Code() == 400) {
        NJson::TJsonValue error;
        if (NJson::ReadJsonFastTree(reply.Content(), &error) && error.Has("error") && error["error"].Has("message")) {
            TString pattern = Config.GetFreeOrderMsgTemplate();
            if (Server) {
                pattern = Server->GetSettings().GetValueDef<TString>("fueling.free_order_msg_template", pattern);
            }
            SubstGlobal(pattern, "_OrderId_", orderId);
            if (pattern == error["error"]["message"].GetStringRobust()) {
                TUnistatSignalsCache::SignalAdd("fueling-api-status", "unknown", 1);
                return EFuelingStatus::Unknown;
            }
        }
    }
    if (reply.Code() != 200) {
        errors.AddMessage(__LOCATION__, TStringBuilder()
            << "Fail to get order status '" << orderId << "'"
            << " with code " << reply.Code()
            << " and error '" << reply.ErrorMessage() << "'");
        ERROR_LOG << reply.GetDebugReply() << Endl;
        return result;
    }
    NJson::TJsonValue replyJson;
    if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
        errors.AddMessage(__LOCATION__, TStringBuilder()
            << "Fail to parse order status '" << orderId << "'"
            << " with content '" << reply.Content() << "'");
        ERROR_LOG << reply.GetDebugReply() << Endl;
        return result;
    }
    if (!TryFromString(replyJson["status"].GetString(), result)) {
        errors.AddMessage(__LOCATION__, TStringBuilder()
            << "Fail to get parsed order status '" << orderId << "'"
            << " with content '" << reply.Content() << "'");
        ERROR_LOG << reply.GetDebugReply() << Endl;
        return result;
    }
    if (replyJson.Has("description")) {
        comment = replyJson["description"].GetString();
    } else if (result == EFuelingStatus::Completed && replyJson.Has("order")) {
        TParsedFuelingResult parsedComment;
        if (parsedComment.SerializeFromJson(replyJson["order"])) {
            comment = parsedComment.GetStringReport();
        }
    }
    return result;
}

bool TFuelingManager::CancelOrder(const TString& orderId, const TString& reason, const EFuelClientType clientType, TMessagesCollector& errors) const {
    if (!HasFuelClientKey(clientType)) {
        errors.AddMessage(__LOCATION__, TStringBuilder()
            << "Incorrect fuel keys configuration '" << orderId << "'"
            << " with client type '" << clientType << "'"
            << " and reason '" << reason << "'");
        return false;
    }
    NNeh::THttpRequest request;
    request.SetUri("/api/order/user/canceled").SetRequestType("POST");
    request.AddHeader("Content-Type", "application/x-www-form-urlencoded");
    request.AddHeader("Accept", "*/*");
    request.AddHeader("X-Robot", "Drive");
    request.SetPostData(
        "apikey=" + Config.GetApiKey(clientType)
        + "&orderId=" + orderId
        + "&reason=" + UrlEscapeRet(reason)
    );
    NUtil::THttpReply reply = NehAgent->SendMessageSync(request, Config.GetReaskConfig().CalcRequestDeadline());
    TUnistatSignalsCache::SignalAdd("fueling-api-cancel", ::ToString(reply.Code()), 1);
    if (reply.Code() != 200) {
        errors.AddMessage(__LOCATION__, TStringBuilder()
            << "Fail to cancel order '" << orderId << "'"
            << " with code " << reply.Code()
            << " and error '" << reply.ErrorMessage() << "'");
        ERROR_LOG << reply.GetDebugReply() << Endl;
    } else {
        INFO_LOG << "fueling order " << orderId << " canceled" << Endl;
    }
    return reply.Code() == 200;
}

TFuelingManager::~TFuelingManager() {
    UnregisterGlobalMessageProcessor(this);
    AsyncDelivery->Stop();
}

TFuelingManager::TFuelingManager(const TFuelingManagerConfig& config, const NDrive::IServer* server)
    : IAutoActualization("fueling_manager", config.GetFreshness())
    , Config(config)
    , StationsRectHash(::TRect<TGeoCoord>(-180, -180, 180, 180))
    , Server(server)
{
    AsyncDelivery.Reset(new TAsyncDelivery);
    AsyncDelivery->Start(1, 8);
    NehAgent.Reset(new NNeh::THttpClient(AsyncDelivery));
    NehAgent->RegisterSource("fueling", Config.GetHost(), Config.GetPort(), Config.GetReaskConfig(), Config.GetIsHttps());
    RegisterGlobalMessageProcessor(this);
}

bool TFuelingManager::HasFuelClientKey(const EFuelClientType clientType) const {
    return clientType == EFuelClientType::Default || Config.GetApiKeysKeeper().GetApiKey(clientType);
}

NThreading::TFuture<NJson::TJsonValue> TFuelingManager::GetColumnStatus(const TString& stationId, const TString& columnId) const {
    NNeh::THttpRequest request;
    request.SetUri("/api/order/post/status").SetCgiData("apikey=" + Config.GetApiKey(EFuelClientType::Default) + "&stationId=" + stationId + "&columnId=" + columnId);
    request.AddHeader("X-Robot", "Drive");
    return NehAgent->SendAsync(request, Config.GetReaskConfig().CalcRequestDeadline()).Apply([columnId](const NThreading::TFuture<NUtil::THttpReply>& r) {
        NJson::TJsonValue result;
        result["id"] = columnId;
        result["status"] = ToString(EFuelingStatus::ServerProblems);
        if (!r.HasValue()) {
            result["error"] = "Fail to get reply: " + NThreading::GetExceptionMessage(r);
            return result;
        }
        const auto& reply = r.GetValue();
        if (!reply.IsSuccessReply()) {
            result["error"] = reply.Serialize();
            return result;
        }
        NJson::TJsonValue replyJson;
        if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
            result["error"] = "Fail to parse reply: " + reply.Content();
            return result;
        }
        EFuelingStatus status = EFuelingStatus::ServerProblems;
        if (!TryFromString(replyJson["status"].GetString(), status)) {
            result["error"] = "Fail to parse reply json: " + replyJson.GetStringRobust();
            return result;
        }
        auto statusInfo = NJson::TryFromJson<TFuelingManager::TStatusInfo>(replyJson);
        result["status"] = ToString(status == EFuelingStatus::Completed && statusInfo ? EFuelingStatus::FuelingCompleted : status);
        if (statusInfo) {
            result["lites"] = statusInfo->Liters;
            result["fuel"] = ToString(statusInfo->FuelId);
        }
        return result;
    });
}

std::pair<TString, TString> TFuelingManagerConfig::TApiKeysKeeper::GetParamNames(const EFuelClientType type) {
    switch (type) {
        case EFuelClientType::Default: return {"ApiKey", "ApiKeyFilePath"};
        case EFuelClientType::UserApp: return {"UserAppApiKey", "UserAppApiKeyFilePath"};
        case EFuelClientType::ServiceApp: return {"ServiceAppApiKey", "ServiceAppApiKeyFilePath"};
    };
};

std::pair<TString, TString> GetApiKeyFromSection(const TYandexConfig::Section* section, const EFuelClientType apiKeyType) {
    TString apiKey;
    TString apiKeyFilePath;
    const auto& [apiKeyField, apiKeyFilePathField] = TFuelingManagerConfig::TApiKeysKeeper::GetParamNames(apiKeyType);
    apiKey = section->GetDirectives().Value(apiKeyField, apiKey);
    apiKeyFilePath = section->GetDirectives().Value(apiKeyFilePathField, apiKeyFilePath);
    if (apiKey || apiKeyFilePath || apiKeyType == EFuelClientType::Default) {
        AssertCorrectConfig(apiKey || apiKeyFilePath, "neither %s nor %s are defined", apiKeyField.c_str(), apiKeyFilePathField.c_str());
        if (!apiKey) {
            AssertCorrectConfig(TFsPath(apiKeyFilePath).Exists(), "Incorrect path for tanker's apikey in '" + apiKeyFilePathField + "'");
            TFileInput fi(apiKeyFilePath);
            apiKey = fi.ReadAll();
        }
    }
    return {apiKey, apiKeyFilePath};
}

void TFuelingManagerConfig::TApiKeysKeeper::Init(const TYandexConfig::Section* section) {
    for (auto&& item : GetEnumAllValues<EFuelClientType>()) {
        if (item == EFuelClientType::Default) {
            continue;
        }
        auto [key, path] = GetApiKeyFromSection(section, item);
        if (key) {
            Secrets[item] = {std::move(key), std::move(path)};
        }
    }
}

void TFuelingManagerConfig::TApiKeysKeeper::ToString(IOutputStream& os) const {
    for (const auto& [type, secret] : Secrets) {
        auto [keyField, pathField] = GetParamNames(type);
        const auto& [key, path] = secret;
        if (path) {
            os << pathField << ": " << path << Endl;
        } else {
            os << keyField << ": " << key << Endl;
        }
    }
}

TMaybe<TString> TFuelingManagerConfig::TApiKeysKeeper::GetApiKey(const EFuelClientType type) const {
    auto ptr = Secrets.FindPtr(type);
    if (!ptr) {
        return {};
    }
    return ptr->Key;
}

TString TFuelingManagerConfig::GetApiKey(const EFuelClientType type) const {
    auto key = ApiKeysKeeper.GetApiKey(type);
    return key ? *key : ApiKey;
}

void TFuelingManagerConfig::Init(const TYandexConfig::Section* section) {
    Host = section->GetDirectives().Value("Host", Host);
    Port = section->GetDirectives().Value("Port", Port);
    IsHttps = section->GetDirectives().Value("IsHttps", IsHttps);
    FreeOrderMsgTemplate = section->GetDirectives().Value("FreeOrderMsgTemplate", FreeOrderMsgTemplate);
    auto [apiKey, apiKeyFilePath] = GetApiKeyFromSection(section, EFuelClientType::Default);
    if (apiKey) {
        ApiKey = apiKey;
    }
    if (apiKeyFilePath) {
        ApiKeyFilePath = apiKeyFilePath;
    }
    Freshness = section->GetDirectives().Value("Freshness", Freshness);
    StringSplitter(section->GetDirectives().Value<TString>("AvailableAreaTags", "allow_riding")).SplitBySet(", ").SkipEmpty().Collect(&AvailableAreaTags);
    auto children = section->GetAllChildren();
    auto itPolicy = children.find("RequestPolicy");
    if (itPolicy != children.end()) {
        ReaskConfig.InitFromSection(itPolicy->second);
    }
    auto itApiKey = children.find("ApiKeys");
    if (itApiKey != children.end()) {
        ApiKeysKeeper.Init(itApiKey->second);
    }
}

void TFuelingManagerConfig::ToString(IOutputStream& os) const {
    os << "Host: " << Host << Endl;
    os << "Port: " << Port << Endl;
    os << "AvailableAreaTags: " << JoinSeq(", ", AvailableAreaTags) << Endl;
    os << "IsHttps: " << IsHttps << Endl;
    if (!!ApiKeyFilePath) {
        os << "ApiKeyFilePath: " << ApiKeyFilePath << Endl;
    } else {
        os << "ApiKey: " << ApiKey << Endl;
    }
    os << "Freshness: " << Freshness << Endl;
    os << "<RequestPolicy>" << Endl;
    ReaskConfig.ToString(os);
    os << "</RequestPolicy>" << Endl;
    os << "<ApiKeys>" << Endl;
    ApiKeysKeeper.ToString(os);
    os << "</ApiKeys>" << Endl;
}

TString GetFuelingStatusDefInfo(const EFuelingStatus status) {
    switch (status) {
        case EFuelingStatus::AcceptOrder:
            return "Ожидаем подтверждение получения заказа от АЗС";
        case EFuelingStatus::OrderCreated:
            return "Заказ создан и полностью оплачен";
        case EFuelingStatus::PaymentInProgress:
            return "Идет оплата заказа";
        case EFuelingStatus::WaitingRefueling:
            return "Ожидаем включения колонки";
        case EFuelingStatus::Fueling:
        case EFuelingStatus::WrongFuelType:
            return "Идет заправка";
        case EFuelingStatus::Expire:
            return "Статус от АЗС не поступил в течение 30 минут";
        case EFuelingStatus::Completed:
            return "Заказ завершен успешно";
        case EFuelingStatus::StationCanceled:
            return "Заказ отменен оператором АЗС или же интегрируемой системой";
        case EFuelingStatus::UserCanceled:
            return "Заказ отменен пользователем";
        case EFuelingStatus::ServerProblems:
            return "Ошибка на сервере";
        case EFuelingStatus::MoreThanOneStation:
        case EFuelingStatus::NotInStation:
            return "Заказа нет и позиция не на АЗС";
        case EFuelingStatus::TriedOrderCreate:
        case EFuelingStatus::ReadyForStart:
            return "Заказа нет, но позиция на АЗС";
        case EFuelingStatus::Unknown:
            return "Неизвестное состояние";
        case EFuelingStatus::ErrorPayment:
            return "Ошибка проведения платежа";
        case EFuelingStatus::Free:
            return "Скажите заправщику:";
        case EFuelingStatus::FuelingCompleted:
            return "Заправка завершена?";
        case EFuelingStatus::Unavailable:
            return "Колонка недоступна";
        case EFuelingStatus::WrongFuelLiters:
            return "Кажется, вы залили бензина больше, чем нужно";
        case EFuelingStatus::WaitCancel:
            return "Заправка отменяется";
    }
}

TString GetFuelingStatusInfo(const ILocalization* localization, const EFuelingStatus status) {
    if (!localization) {
        return GetFuelingStatusDefInfo(status);
    }
    const TString prefix = "FuelingInfo.";
    return localization->GetLocalString(ELocalization::Rus, prefix + ToString(status), GetFuelingStatusDefInfo(status));
}

TString GetFuelingStatusDefDescription(const EFuelingStatus status, const bool isPostPay) {
    switch (status) {
    case EFuelingStatus::Free:
        return "Если заправщика нет, то залейте полный бак самостоятельно";
    case EFuelingStatus::FuelingCompleted:
        return "Если всё ок, то сейчас начнётся оплата";
    case EFuelingStatus::WrongFuelType:
        return "Но кажется, вы заливаете что-то не то";
    case EFuelingStatus::WrongFuelLiters:
        return "Оплатите всю заправку на кассе сами";
    case EFuelingStatus::WaitCancel:
        return "Придётся подождать";
    case EFuelingStatus::Fueling:
        return isPostPay ? "Повесьте пистолет обратно, когда заправитесь" : "";
    default:
        return "";
    }
}

TString GetFuelingStatusDescription(const ILocalization* localization, const EFuelingStatus status, const TString& def, const bool isPostPay) {
    TString comment = GetFuelingStatusDefDescription(status, isPostPay);
    comment = comment.empty() ? def : comment;
    if (!localization) {
        return comment;
    }
    const TString prefix = "FuelingDescription.";
    return localization->GetLocalString(ELocalization::Rus, prefix + ToString(status), comment);
}

bool TFuelGroups::FuelTypeCmp(const EFuelType request, const EFuelType actual) const {
    if (request == actual) {
        return true;
    }
    if (!Map) {
        return false;
    }
    auto requestIt = Map->find(request);
    if (requestIt == Map->end()) {
        return false;
    }
    return requestIt->second.contains(actual);
}

TMaybe<TFuelGroups::TFuelMap> TFuelGroups::DeserializeFuelMap(const NJson::TJsonValue& json) {
    if (!json.IsArray()) {
        return {};
    }
    TFuelMap map;
    for (const auto& item : json.GetArray()) {
        if (!item.Has("request") || !item["request"].IsArray() || !item.Has("allow") || !item["allow"].IsArray()) {
            return {};
        }
        TSet<EFuelType> allow;
        for (const auto& i : item["allow"].GetArray()) {
            EFuelType type;
            if (i.IsString() && TryFromString(i.GetString(), type)) {
                allow.insert(type);
            } else {
                WARNING_LOG << "Fail to parse allow fuel type '" << i << "'" << Endl;
            }
        }
        for (const auto& i : item["request"].GetArray()) {
            EFuelType type;
            if (i.IsString() && TryFromString(i.GetString(), type)) {
                map[type].insert(allow.begin(), allow.end());
            } else {
                WARNING_LOG << "Fail to parse request fuel type '" << i << "'" << Endl;
            }
        }
    }
    return map;
}

void TFuelGroups::InitFromSettings(const ISettings& settings) {
    {
        const auto fuelGroups = settings.GetValueDef<TString>("fueling.fuel_groups", "[]");
        NJson::TJsonValue json;
        if (!ReadJsonTree(fuelGroups, &json)) {
            ERROR_LOG << "Fail to parse fuel proups settings" << Endl;
        }
        Map = DeserializeFuelMap(json);
        if (!Map) {
            ERROR_LOG << "Fail to parse fuel proups json" << Endl;
        }
    }
}

template <>
NJson::TJsonValue NJson::ToJson(const EFuelType& type) {
    return NJson::ToJson(NJson::Stringify(type));
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& json, EFuelType& type) {
    return NJson::TryFromJson(json, NJson::Stringify(type));
}
