#pragma once
#include <drive/backend/abstract/frontend.h>

#include <library/cpp/mediator/messenger.h>
#include <library/cpp/yconf/conf.h>

#include <rtline/library/async_proxy/async_delivery.h>
#include <rtline/library/geometry/coord.h>
#include <rtline/library/geometry/polyline.h>
#include <rtline/library/metasearch/simple/config.h>
#include <rtline/library/rect_hash/rect_hash.h>
#include <rtline/util/auto_actualization.h>
#include <rtline/util/json_processing.h>
#include <rtline/util/network/neh.h>
#include <rtline/util/types/accessor.h>
#include <rtline/util/types/cache_with_age.h>

#include <util/datetime/base.h>
#include <util/random/random.h>

enum class EFuelingStatus {
    OrderCreated = 0, // – заказ создан и полностью оплачен
    PaymentInProgress = 1, // – идет оплата заказа
    AcceptOrder = 2, // – ожидаем подтверждение получения заказа от АЗС
    WaitingRefueling = 3, // – ожидаем включения колонки
    Fueling = 4, // – идет заправка
    Expire = 5, // – статус от АЗС не поступил в течение 30 минут
    Completed = 6, // – заказа завершен успешно
    StationCanceled = 7, // – заказ отменен оператором АЗС или же интегрируемой системой
    UserCanceled = 8, // – заказ отменен пользователем
    ServerProblems = 9, // - ошибка на сервере
    NotInStation = 10, // - заказа нет и позиция не на АЗС
    ReadyForStart = 11, // - заказа нет, но позиция на АЗС
    Unknown = 12, // - неизвестное состояние
    ErrorPayment = 13, // - проблемы с платежом на стороне заправок

    Free = 14, // - ожидаем включения колонки без предоплаты
    FuelingCompleted = 15, // - подача топлива завершина
    Unavailable = 16, // - ошибка при получении информаци о колонке
    WrongFuelType = 17, // - ошибка при несоответствии типа топлив с ожидаемом
    WrongFuelLiters = 18, // - ошибка при перелиле топлива
    WaitCancel = 19, // - ожидание отмены заправки, при пост оплате ожидаем освобождения колонки
    MoreThanOneStation = 20, // - заказа нет, и нельзя определить АЗС по позиции

    TriedOrderCreate = 21, // - совершена попытка создания заказа
};

enum class EFuelType {
    Diesel = 0 /* "diesel" */,
    DieselPremium = 1 /* "diesel_premium" */,
    A80 = 2 /* "a80" */,
    A92 = 3 /* "a92" */,
    A92Premium = 4 /* "a92_premium" */,
    A95 = 5 /* "a95" */,
    A95Premium = 6 /* "a95_premium" */,
    A98 = 7 /* "a98" */,
    A98Premium = 8 /* "a98_premium" */,
    A100 = 9 /* "a100" */,
    A100Premium = 10 /* "a100_premium" */,
    Propane = 11 /* "propane" */,
    Metan = 12 /* "metan" */,

    Undefined = 13 /* "undefined" */,

    DieselWinter = 14 /* "diesel_winter" */,
    DieselDemiseason = 15 /* "diesel_demiseason" */,
};

enum class EFuelClientType {
    Default = 0 /* "default" */,
    UserApp = 1 /* "user_app" */,
    ServiceApp = 2 /* "service_app" */,
};

enum class EStationSearchResult {
    NotInStation,
    MoreThanOneStation,
    Ok,
};

TString GetFuelingStatusInfo(const ILocalization* localization, const EFuelingStatus status);
TString GetFuelingStatusDescription(const ILocalization* localization, const EFuelingStatus status, const TString& def, const bool isPostPay);

class TFuelGroups {
public:
    using TFuelMap = TMap<EFuelType, TSet<EFuelType>>;
    R_FIELD(TMaybe<TFuelMap>, Map);

public:
    static TMaybe<TFuelGroups::TFuelMap> DeserializeFuelMap(const NJson::TJsonValue& json);
    void InitFromSettings(const ISettings& settings);
    bool FuelTypeCmp(const EFuelType request, const EFuelType actual) const;
};

class TFuelingManagerConfig {
private:
    R_READONLY(TString, Host, "https://app.tanker.yandex.net");
    R_READONLY(ui32, Port, 443);
    R_READONLY(bool, IsHttps, true);
    R_READONLY(TString, ApiKey);
    R_READONLY(TString, ApiKeyFilePath);
    R_READONLY(NSimpleMeta::TConfig, ReaskConfig);
    R_READONLY(TDuration, Freshness, TDuration::Minutes(10));
    R_READONLY(TSet<TString>, AvailableAreaTags);
    R_READONLY(TString, FreeOrderMsgTemplate, "Статус заказа _OrderId_ не найден");

public:
    class TApiKeysKeeper {
        struct TKeySecret {
            TString Key;
            TString Path;
        };
        TMap<EFuelClientType, TKeySecret> Secrets;
    public:
        void Init(const TYandexConfig::Section* section);
        void ToString(IOutputStream& os) const;
        TMaybe<TString> GetApiKey(const EFuelClientType type) const;
        static std::pair<TString, TString> GetParamNames(const EFuelClientType type);
    };

private:
    R_READONLY(TApiKeysKeeper, ApiKeysKeeper);

public:
    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;
    TString GetApiKey(const EFuelClientType type) const;
};

class TColumn {
    R_READONLY(TString, Id);
    R_READONLY(TSet<EFuelType>, Fuels);

public:
    TColumn(const TString& id)
        : Id(id)
    {
    }

    NJson::TJsonValue SerializeToJson(const TSet<EFuelType>* fTypes) const;
    bool DeserializeFromJson(const NJson::TJsonValue& json);
};

class TStation {
private:
    R_READONLY(TString, Id);
    R_READONLY(TString, Name);
    R_READONLY(TString, Address);
    R_READONLY(TGeoCoord, Location);
    R_READONLY(TVector<TGeoCoord>, Area);
    R_READONLY(TVector<TColumn>, Columns);
    R_READONLY(bool, PostPay, false);

public:
    using TPtr = TAtomicSharedPtr<TStation>;
    TStation() = default;
    NJson::TJsonValue GetJsonMapReport(const TSet<EFuelType>* fTypes) const;
    bool DeserializeFromJson(const NJson::TJsonValue& json);
};

class TStationWithRect {
private:
    R_READONLY(TString, StationId);
    R_READONLY(TRect<TGeoCoord>, Rect);

public:
    TStationWithRect(const TString& stationId, const ::TRect<TGeoCoord>& rect)
        : StationId(stationId)
        , Rect(rect)
    {
    }

    bool operator==(const TStationWithRect& item) const;
    bool operator<(const TStationWithRect& item) const;
};

class TFuelingManager: public IAutoActualization, public IMessageProcessor {
public:
    struct TStationAddress {
        TString Id;
        TString Address;
    };

    struct TFuelInfo {
        EFuelType FuelId;
        double Liters;
    };

    struct TOrderInfo {
        TMaybe<TGeoCoord> Coord;
        TString OrderId;
        TString StationId;
        TString ColumnId;
        TString CarId;
        TFuelInfo FuelInfo;
        EFuelClientType ClientType;
    };

    using TStatusInfo = TFuelInfo;

private:
    THolder<NNeh::THttpClient> NehAgent;
    const TFuelingManagerConfig Config;
    TAsyncDelivery::TPtr AsyncDelivery;
    using TStations = TCacheWithAge<TString, TStation>;
    TStations Stations;
    TMap<TString, ::TPolyLine<TGeoCoord>> Polylines;
    TRWMutex Mutex;
    TRectHash<TGeoCoord, TStationWithRect> StationsRectHash;
    const NDrive::IServer* Server;

protected:
    virtual void DoWaitPeriod(const TInstant start) override;
    virtual bool Process(IMessage* message) override;
    virtual bool GetStartFailIsProblem() const override;
    virtual bool Refresh() override;
    virtual TString Name() const override;

public:
    TFuelingManager(const TFuelingManagerConfig& config, const NDrive::IServer* server);
    ~TFuelingManager();

    bool HasFuelClientKey(const EFuelClientType clientType) const;

    TVector<TStationAddress> GetStationsAddresses() const;
    bool GetStationInfo(const TString& stationId, TStation& result) const;
    bool GetStationInfo(const TGeoCoord& coord, TStation& result) const;
    EStationSearchResult GetStationInfo(const TGeoRect& carRect, TStation& result) const;

    bool CreateOrder(const TOrderInfo& context, TMessagesCollector& errors) const;
    bool CancelOrder(const TString& orderId, const TString& reason, const EFuelClientType clientType, TMessagesCollector& errors) const;
    EFuelingStatus GetStatus(const TString& orderId, TString& comment, const EFuelClientType clientType, TMessagesCollector& errors) const;
    EFuelingStatus GetPostStatus(const TString& stationId, const TString& columnId, TMaybe<TStatusInfo>& statusInfo, const EFuelClientType clientType, TMessagesCollector& errors) const;

    TString GetFuelingTypeMessage(const EFuelType fType) const;
    TString GetFuelingTypeMessage(const EFuelType fType, const double litres) const;
    NJson::TJsonValue GetJsonMapReport(const TSet<EFuelType>* fTypes = nullptr, const bool withPostPay = false, const TMaybe<TGeoRect>& geoFilter = {}) const;
    NThreading::TFuture<NJson::TJsonValue> GetColumnStatus(const TString& stationId, const TString& columnId) const;

    static TString GenerateOrderId();
    static TString GetHRFuel(const EFuelType fType, const TString& prefix = "");
};
