#include <drive/backend/fueling_manager/ut/library/constants.h>
#include <drive/backend/fueling_manager/ut/library/tanker_emulator.h>

#include <drive/backend/fueling_manager/fueling_manager.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/deprecated/async_impl/async_impl.h>

#include <util/system/env.h>

Y_UNIT_TEST_SUITE(FuelingManagerSuite) {
    Y_UNIT_TEST(FuelingManagerConfigInit) {
        const TString defKey = "test_def_key";
        TString configStr = TStringBuilder()
            << "Host: app.tst.tanker.yandex.net" << Endl
            << "Port: 80" << Endl
            << "ApiKey: " << defKey << Endl
            << "IsHttps: false" << Endl
            << "<RequestPolicy>" << Endl
            << "GlobalTimeout: 1000" << Endl
            << "</RequestPolicy>" << Endl;
        {
            INFO_LOG << configStr << Endl;
            TAnyYandexConfig yConf;
            UNIT_ASSERT(yConf.ParseMemory(configStr.data()));
            TFuelingManagerConfig config;
            config.Init(yConf.GetRootSection());
            UNIT_ASSERT_EQUAL(config.GetApiKey(), defKey);
            UNIT_ASSERT_EQUAL(config.GetApiKey(EFuelClientType::Default), defKey);
            UNIT_ASSERT_EQUAL(config.GetApiKey(EFuelClientType::ServiceApp), defKey);
            UNIT_ASSERT_EQUAL(config.GetApiKey(EFuelClientType::UserApp), defKey);
        }
        const TString serviceKey = "test_service_key";
        const TString userKey = "test_user_key";
        configStr += TStringBuilder()
            << "<ApiKeys>" << Endl
            << "ServiceAppApiKey: " << serviceKey << Endl
            << "UserAppApiKey: " << userKey << Endl
            << "</ApiKeys>" << Endl;
        {
            INFO_LOG << configStr << Endl;
            TAnyYandexConfig yConf;
            UNIT_ASSERT(yConf.ParseMemory(configStr.data()));
            TFuelingManagerConfig config;
            config.Init(yConf.GetRootSection());
            UNIT_ASSERT_EQUAL(config.GetApiKey(), defKey);
            UNIT_ASSERT_EQUAL(config.GetApiKey(EFuelClientType::Default), defKey);
            UNIT_ASSERT_EQUAL(config.GetApiKey(EFuelClientType::ServiceApp), serviceKey);
            UNIT_ASSERT_EQUAL(config.GetApiKey(EFuelClientType::UserApp), userKey);
        }
    }

    Y_UNIT_TEST(SimplerOperations) {
        const TString defKey = "test_def_key";
        const TString serviceKey = "test_service_key";
        const ui32 tankerPort = 12345;
        TString configStr = TStringBuilder()
            << "Host: localhost" << Endl
            << "Port: " << tankerPort << Endl
            << "ApiKey: " << defKey << Endl
            << "IsHttps: false" << Endl
            << "<RequestPolicy>" << Endl
            << "GlobalTimeout: 1000" << Endl
            << "</RequestPolicy>" << Endl
            << "<ApiKeys>" << Endl
            << "ServiceAppApiKey: " << serviceKey << Endl
            << "</ApiKeys>" << Endl;
        INFO_LOG << configStr << Endl;
        TAnyYandexConfig yConf;
        UNIT_ASSERT(yConf.ParseMemory(configStr.data()));
        TFuelingManagerConfig config;
        config.Init(yConf.GetRootSection());
        TFuelingManager mgr(config, nullptr);
        TTankerEmulator tanker;
        tanker.Run(tankerPort);

        TFuelingManager::TOrderInfo testOrder = {
            /* Coord = */ {},
            /* StationId = */ "station_id",
            /* OrderId = */ "order_id",
            /* ColumnId = */ "column_id",
            /* CarId = */ "car_id",
            /* FuelInfo = */ {
                /* FuelId = */ EFuelType::A92,
                /* Liters = */ 20.0
            },
            /* ClientType = */ EFuelClientType::Default
        };
        TMessagesCollector errors;

        {
            UNIT_ASSERT_C(mgr.CreateOrder(testOrder, errors), errors.GetStringReport());
            TString comment;
            UNIT_ASSERT_EQUAL_C(mgr.GetStatus(testOrder.OrderId, comment, EFuelClientType::Default, errors), EFuelingStatus::Completed, errors.GetStringReport());
            UNIT_ASSERT_EQUAL_C(mgr.GetStatus(testOrder.OrderId, comment, EFuelClientType::UserApp, errors), EFuelingStatus::Completed, errors.GetStringReport());
            {
                TMessagesCollector errorsWithMsg;
                UNIT_ASSERT_EQUAL(mgr.GetStatus(testOrder.OrderId, comment, EFuelClientType::ServiceApp, errorsWithMsg), EFuelingStatus::ServerProblems);
            }
            UNIT_ASSERT_C(mgr.CancelOrder(testOrder.OrderId, "reason", EFuelClientType::Default, errors), errors.GetStringReport());
            {
                TMessagesCollector errorsWithMsg;
                UNIT_ASSERT(!mgr.CancelOrder(testOrder.OrderId, "reason", EFuelClientType::UserApp, errorsWithMsg));
            }
            {
                TMessagesCollector errorsWithMsg;
                UNIT_ASSERT(!mgr.CancelOrder(testOrder.OrderId, "reason", EFuelClientType::ServiceApp, errorsWithMsg));
            }
            auto it = tanker.GetStates().find(testOrder.OrderId);
            UNIT_ASSERT(it != tanker.GetStates().end());
            UNIT_ASSERT_EQUAL(it->second.ApiKey, defKey);
        }
        {
            auto order = testOrder;
            order.ClientType = EFuelClientType::ServiceApp;
            order.OrderId = "new_order_id";
            UNIT_ASSERT_C(mgr.CreateOrder(order, errors), errors.GetStringReport());
            TString comment;
            UNIT_ASSERT_EQUAL_C(mgr.GetStatus(order.OrderId, comment, EFuelClientType::ServiceApp, errors), EFuelingStatus::Completed, errors.GetStringReport());
            {
                TMessagesCollector errorsWithMsg;
                UNIT_ASSERT_EQUAL(mgr.GetStatus(order.OrderId, comment, EFuelClientType::Default, errorsWithMsg), EFuelingStatus::ServerProblems);
            }
            UNIT_ASSERT_C(mgr.CancelOrder(order.OrderId, "reason", EFuelClientType::ServiceApp, errors), errors.GetStringReport());
            {
                TMessagesCollector errorsWithMsg;
                UNIT_ASSERT(!mgr.CancelOrder(order.OrderId, "reason", EFuelClientType::Default, errorsWithMsg));
            }
            auto it = tanker.GetStates().find(order.OrderId);
            UNIT_ASSERT(it != tanker.GetStates().end());
            UNIT_ASSERT_EQUAL(it->second.ApiKey, serviceKey);
        }
    }

    Y_UNIT_TEST(UnknownOrder) {
        const TString defKey = "test_def_key";
        const ui32 tankerPort = 12345;
        TString configStr = TStringBuilder()
            << "Host: localhost" << Endl
            << "Port: " << tankerPort << Endl
            << "ApiKey: " << defKey << Endl
            << "IsHttps: false" << Endl;
        INFO_LOG << configStr << Endl;
        TAnyYandexConfig yConf;
        UNIT_ASSERT(yConf.ParseMemory(configStr.data()));
        TFuelingManagerConfig config;
        config.Init(yConf.GetRootSection());
        TFuelingManager mgr(config, nullptr);
        TTankerEmulator tanker;
        tanker.Run(tankerPort);
        TString comment;
        TMessagesCollector errors;
        UNIT_ASSERT_EQUAL_C(mgr.GetStatus(NTestDriveFueling::WrongOrderId, comment, EFuelClientType::Default, errors), EFuelingStatus::Unknown, errors.GetStringReport());
        UNIT_ASSERT_EQUAL(mgr.GetStatus(NTestDriveFueling::WrongOrderIdWithError, comment, EFuelClientType::Default, errors), EFuelingStatus::ServerProblems);
    }
}
