#include "constants.h"
#include "tanker_emulator.h"

#include <library/cpp/http/misc/parsed_request.h>

#include <util/random/random.h>


bool TFuelingState::Parse(const TMap<TString, TString>& data) {
    auto parse = [&data] (const TString& key, auto& value) {
        auto it = data.find(key);
        return it != data.end() && TryFromString(it->second, value);
    };
    return
           parse("apikey", ApiKey)
        && parse("orderId", OrderId)
        && parse("fuelId", FuelId)
        && parse("orderVolume", Liters)
        && parse("stationId", StationId)
        && parse("columnId", ColumnId);
}


class TTankerReplier : public TRequestReplier {
public:
    using TParsedData = TMap<TString, TString>;
    using TOptionalParsedData = TMaybe<TParsedData>;

public:
    TTankerReplier(TTankerEmulator& owner)
        : Owner(owner) {}

    TOptionalParsedData ParseData(const TString& post, const TString& cgi) const;
    TString CreateOrder(const TParsedData& data);
    TString CancelOrder(const TParsedData& data);
    TString ReportOrder(const TParsedData& data) const;
    TString ReportColumn(const TParsedData& data) const;


    bool DoReply(const TReplyParams& params) override {
        TParsedHttpFull parsedRequest(params.Input.FirstLine());
        auto parsedData = ParseData(params.Input.ReadAll(), TString(parsedRequest.Cgi));
        if (!parsedData) {
            params.Output << NTestDriveFueling::BadRequestReply;
            ERROR_LOG << "No apikey" << Endl;
            return true;
        }
        if (parsedRequest.Path.StartsWith(NTestDriveFueling::CreatePath)) {
            params.Output << CreateOrder(*parsedData);
        } else if (parsedRequest.Path.StartsWith(NTestDriveFueling::InfoPath)) {
            params.Output << ReportOrder(*parsedData);
        } else if (parsedRequest.Path.StartsWith(NTestDriveFueling::InfoPostPath)) {
            params.Output << ReportColumn(*parsedData);
        } else if (parsedRequest.Path.StartsWith(NTestDriveFueling::CancelPath)) {
            params.Output << CancelOrder(*parsedData);
        } else if (parsedRequest.Path.StartsWith(NTestDriveFueling::MapPath)) {
            params.Output << NTestDriveFueling::OkReply << NTestDriveFueling::MapJson << Endl;
        } else {
            ERROR_LOG << parsedRequest.Path << " unimplemented" << Endl;
            params.Output << NTestDriveFueling::BadRequestReply;
        }
        return true;
    }

private:
    TTankerEmulator& Owner;
};

TTankerReplier::TOptionalParsedData TTankerReplier::ParseData(const TString& post, const TString& cgi) const {
    TMap<TString, TString> parsedData;
    auto parse = [&parsedData] (const TString& data) {
        TVector<TString> parts = SplitString(data, "&");
        for (auto&& part : parts) {
            TVector<TString> items = SplitString(part, "=");
            if (items.size() != 2) {
                return false;
            }
            parsedData[items.front()] = items.back();
        }
        return !parsedData["apikey"].empty();
    };
    if (!parse(post) && !parse(cgi)) {
        ERROR_LOG << "No apikey" << Endl;
        return {};
    }
    return parsedData;
}

TString TTankerReplier::CreateOrder(const TParsedData& data) {
    TFuelingState state;
    if (!state.Parse(data)) {
        return NTestDriveFueling::BadRequestReply;
    }
    TString reply;
    auto it = Owner.GetStates().find(state.OrderId);
    if (it != Owner.GetStates().end()) {
        reply = it->second.GetReport();
    } else {
        state.Status = state.ColumnId == "4" ? EFuelingStatus::StationCanceled : EFuelingStatus::Completed;
        reply = state.GetReport();
        Owner.GetStates().emplace(state.OrderId, state);
    }
    return TStringBuilder() << NTestDriveFueling::OkReply << reply << Endl;
}

TString TTankerReplier::CancelOrder(const TParsedData& data) {
    const TString& orderId = data.Value("orderId", "");
    const TString& apiKey = data.Value("apikey", "");
    if (orderId.empty()) {
        ERROR_LOG << "Wrong order" << Endl;
        return NTestDriveFueling::BadRequestReply;
    }
    auto it = Owner.GetStates().find(orderId);
    if (it == Owner.GetStates().end()) {
        return NTestDriveFueling::NotFoundReply;
    }
    if (it->second.ApiKey != apiKey) {
        ERROR_LOG << "Wrong apikey" << Endl;
        return NTestDriveFueling::BadRequestReply;
    }
    return TStringBuilder() << NTestDriveFueling::OkReply << Endl;
}

TString TTankerReplier::ReportOrder(const TParsedData& data) const {
    const TString& orderId = data.Value("orderId", "");
    const TString& apiKey = data.Value("apikey", "");
    if (orderId.empty()) {
        return NTestDriveFueling::BadRequestReply;
    }
    if (orderId == NTestDriveFueling::WrongOrderId) {
        return TStringBuilder()
            << NTestDriveFueling::BadRequestReply
            << "{\"error\":{\"message\":\"Статус заказа "
            << NTestDriveFueling::WrongOrderId
            << " не найден\"}}"
            << Endl;
    }
    if (orderId == NTestDriveFueling::WrongOrderIdWithError) {
        return TStringBuilder()
            << NTestDriveFueling::BadRequestReply
            << "{\"error\":{\"message\":\"Статус заказа "
            << NTestDriveFueling::WrongOrderId
            << " не найден\"}}"
            << Endl;
    }
    auto it = Owner.GetStates().find(orderId);
    if (it == Owner.GetStates().end()) {
        return NTestDriveFueling::NotFoundReply;
    }
    if (it->second.ApiKey != apiKey) {
        return NTestDriveFueling::BadRequestReply;
    }
    return TStringBuilder() << NTestDriveFueling::OkReply << it->second.GetReport() << Endl;
}

TString GetColumnState(const TString& columnId, const bool completed) {
    if (columnId == NTestDriveFueling::PetrolColumnFor100Premium) {
        return completed
            ? NTestDriveFueling::Completed100PremiumColumnState
            : NTestDriveFueling::Fueling100PremiumColumnState;
    }
    return completed
        ? NTestDriveFueling::DefaultCompletedColumnState
        : NTestDriveFueling::DefaultFuelingColumnState;
}

TString TTankerReplier::ReportColumn(const TParsedData& data) const {
    TString columnId;
    if (auto ptr = data.FindPtr("columnId")) {
        columnId = *ptr;
    }
    switch (RandomNumber<ui8>(3)) {
    case 1:
        return TStringBuilder() << NTestDriveFueling::OkReply << GetColumnState(columnId, true) << Endl;
    case 2:
        return TStringBuilder() << NTestDriveFueling::OkReply << GetColumnState(columnId, false) << Endl;
    default:
        return TStringBuilder() << NTestDriveFueling::OkReply << NTestDriveFueling::FreeColumnState << Endl;
    }
}

void TTankerEmulator::Run(ui16 httpPort) {
    THttpServerOptions httpOptions(httpPort);
    Server = MakeHolder<THttpServer>(this, httpOptions);
    try {
        Server->Start();
    } catch (const std::exception& e) {
        ERROR_LOG << FormatExc(e);
        FAIL_LOG("Can't create emulator");
    }
}

TClientRequest* TTankerEmulator::CreateClient() {
    return new TTankerReplier(*this);
}

TTankerEmulator::~TTankerEmulator() {
    if (Server) {
        Server->Stop();
    }
}
