#include "logic.h"

#include <drive/backend/compiled_riding/compiled_riding.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_manager.h>

TDBTag TGame::GetUserGameTag(const TString& userId) const {
    const TUserTagsManager& userTagsManager = DriveApi.GetTagsManager().GetUserTags();
    TDBTag gameStat;

    TVector<TDBTag> userTags;
    {
        auto session = userTagsManager.BuildSession(true);
        auto optionalObject = userTagsManager.RestoreObject(userId, session);
        userTags = Yensured(optionalObject)->DetachTags();
    }

    for (auto&& tag : userTags) {
        if (tag.Is<TGameTag>() && tag->GetName() == Config.GetGameTag()) {
            return tag;
        }
    }

    auto defaults = MakeHolder<TGameTag>();
    defaults->SetName(Config.GetGameTag());
    gameStat.SetObjectId(userId);
    gameStat.SetData(defaults.Release());
    return gameStat;
}

std::pair<bool, TString> TGame::CheckSessionDeep(const TFullCompiledRiding& fullRiding) const {
    if (!fullRiding.GetOffer()) {
        return { false, "no_offer_data" };
    }

    if (!Config.GetOfferBuildersBlackList().empty() && Config.GetOfferBuildersBlackList().contains(fullRiding.GetOffer()->GetBehaviourConstructorId())) {
        return { false, "blacklisted_offer" };
    }

    auto action = DriveApi.GetRolesManager()->GetAction(fullRiding.GetOffer()->GetBehaviourConstructorId());
    if (!action || !action->GetAs<IOfferBuilderAction>()) {
        return { false, "internal_error" };
    }

    if (Config.GetOfferBuilderAttrsFilter()) {
        TVector<TString> groupingTags((*action)->GetGrouppingTags().begin(), (*action)->GetGrouppingTags().end());
        if (!Config.GetTagsFilter().IsMatching(groupingTags)) {
            return { false, "service_offer" };
        }
    }

    if (Config.GetFilterCorp()) {
        if (fullRiding.GetOffer()->GetSelectedCharge() != "card") {
            return { false, "corp_offer" };
        }
    }
    return { true, "" };
}

void TGame::TConfig::GetScheme(const IServerBase& server, NDrive::TScheme& scheme) {
    scheme.Add<TFSString>("attrs_filter", "Атриубуты тарифов для фильтрации");
    scheme.Add<TFSVariants>("offers_blacklist", "Игнорировать тарифы").SetVariants(IOfferBuilderAction::GetNames(server.GetAsPtrSafe<NDrive::IServer>())).SetMultiSelect(true);

    scheme.Add<TFSBoolean>("filter_corp", "Фильтровать корпоративные поездки").SetDefault(true);
    scheme.Add<TFSNumeric>("first_instant", "Старт акции").SetVisual(TFSNumeric::EVisualType::DateTime);
    scheme.Add<TFSNumeric>("last_instant", "Финиш акции").SetVisual(TFSNumeric::EVisualType::DateTime);
    scheme.Add<TFSNumeric>("min_mileage", "Минимальный пробег").SetDefault(1);
    scheme.Add<TFSNumeric>("local_limit", "Минимальная граница для участия в розыгрыше").SetDefault(1);
    scheme.Add<TFSString>("game_tag", "Тег для хранения статистики");
    scheme.Add<TFSString>("home_work_tag", "Тег для бонуса Дом-Работа");
    scheme.Add<TFSString>("disable_game_tag", "Тег для отключения игры");
}


TMaybe<TGame::TConfig> TGame::TConfig::BuildFromString(const TString& configString) {
    NJson::TJsonValue jsonData;
    if (!NJson::ReadJsonFastTree(configString, &jsonData)) {
        return Nothing();
    }
    TConfig result;
    if (!result.DeserializeFromJson(jsonData)) {
        return Nothing();
    }
    return result;
}


TMaybe<TGame::TConfig> TGame::TConfig::BuildFromSettings(const ISettings& settings) {
    TString configString;
    if (!settings.GetValue("ny_game.game_config", configString)) {
        return Nothing();
    }
    return BuildFromString(configString);
}

bool TGame::TConfig::DeserializeFromJson(const NJson::TJsonValue& result) {
    TJsonProcessor::ReadContainer(result, "offers_blacklist", OfferBuildersBlackList);
    JREAD_STRING_OPT(result, "attrs_filter", OfferBuilderAttrsFilter);
    JREAD_BOOL_OPT(result, "filter_corp", FilterCorp);
    JREAD_STRING(result, "game_tag", GameTag);
    JREAD_STRING_OPT(result, "disable_game_tag", DisableGameTag);
    JREAD_STRING_OPT(result, "home_work_tag", HomeWorkTag);
    JREAD_UINT(result, "min_mileage", MinMileage);
    JREAD_UINT_OPT(result, "local_limit", LocalLimit);
    JREAD_INSTANT(result, "first_instant", FirstSessionInstant);
    JREAD_INSTANT_OPT(result, "last_instant", LastSessionInstant);
    TagsFilter = TTagsFilter::BuildFromString(OfferBuilderAttrsFilter);
    return true;
}

void TGame::TConfig::SerializeToJson(NJson::TJsonValue& result) const {
    TJsonProcessor::WriteContainerArray(result, "offers_blacklist", OfferBuildersBlackList);
    JWRITE(result, "attrs_filter", OfferBuilderAttrsFilter);
    JWRITE(result, "filter_corp", FilterCorp);
    JWRITE(result, "game_tag", GameTag);
    JWRITE(result, "home_work_tag", HomeWorkTag);
    JWRITE(result, "disable_game_tag", DisableGameTag);
    JWRITE(result, "min_mileage", MinMileage);
    JWRITE(result, "local_limit", LocalLimit);
    JWRITE_INSTANT(result, "first_instant", FirstSessionInstant);
    JWRITE_INSTANT(result, "last_instant", LastSessionInstant);
}
