#include "tag.h"



TMaybe<TInstant> TGameTag::GetIntervalStart(const TGameTagDescription& description, const TInstant timestamp) {
    TMaybe<TInstant> startInstant;
    for (auto&& ts : description.GetDeadlines()) {
        if (timestamp >= ts) {
            startInstant = ts;
        } else {
            break;
        }
    }
    return startInstant;
}

ui32 TGameTag::AddPoints(const TGameTagDescription& description, const ui32 points, const TInstant timestamp) {
    ui32 corrected = Min(points, description.GetDeltaLimit());
    auto intervalStart = GetIntervalStart(description, timestamp);
    if (intervalStart.Defined()) {
        Balance += corrected;
        Balance2Week[*intervalStart] += corrected;
        return corrected;
    }
    return 0;
}

bool TGameTag::SetGameId(const TGameTagDescription& description, const ui32 gameId, const TInstant timestamp) {
    auto intervalStart = GetIntervalStart(description, timestamp);
    if (intervalStart.Defined()) {
        auto it = GameIds.find(*intervalStart);
        if (it == GameIds.end()) {
            GameIds[*intervalStart] = gameId;
            return true;
        }
    }
    return false;
}


ui32 TGameTag::GetFullBalance(const TDBTag& tag) {
    const TGameTag* gameTag = tag.GetTagAs<TGameTag>();
    if (!gameTag) {
        return 0;
    }
    return gameTag->GetBalance();
}

ui32 TGameTag::GetLocalBalance(const TGameTagDescription& description, const TGameTag& gameTag, const TInstant timestamp) {
    auto startInstant = GetIntervalStart(description, timestamp);
    if (!startInstant) {
        return 0;
    }
    auto it = gameTag.GetBalance2Week().find(*startInstant);
    if (it == gameTag.GetBalance2Week().end()) {
        return 0;
    }
    return it->second;
}

ui32 TGameTag::GetLocalBalance(const TGameTagDescription& description, const TDBTag& tag, const TInstant timestamp) {
    const TGameTag* gameTag = tag.GetTagAs<TGameTag>();
    if (!gameTag) {
        return 0;
    }
    return GetLocalBalance(description, *gameTag, timestamp);
}

TMaybe<ui32> TGameTag::GetGameId(const TGameTagDescription& description, const TGameTag& gameTag, const TInstant timestamp) {
    auto startInstant = GetIntervalStart(description, timestamp);
    if (!startInstant) {
        return Nothing();
    }
    auto it = gameTag.GetGameIds().find(*startInstant);
    if (it == gameTag.GetGameIds().end()) {
        return Nothing();
    }
    return it->second;;
}

TMaybe<ui32> TGameTag::GetGameId(const TGameTagDescription& description, const TDBTag& tag, const TInstant timestamp) {
    const TGameTag* gameTag = tag.GetTagAs<TGameTag>();
    if (!gameTag) {
        return 0;
    }
    return GetGameId(description, *gameTag, timestamp);
}

const TString TGameTag::TypeName = "game_tag";
ITag::TFactory::TRegistrator<TGameTag> TGameTag::Registrator(TGameTag::TypeName);
TTagDescription::TFactory::TRegistrator<TGameTagDescription> TGameTag::RegistratorDescription(TGameTag::TypeName);

TGameTag::TProto TGameTag::DoSerializeSpecialDataToProto() const {
    TGameTag::TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.MutableGameData()->SetBalance(Balance);

    for (auto&& gId : GameIds) {
        NDrive::NProto::TGameTagData::TTimedItem* item = proto.MutableGameData()->AddGameId();
        item->SetDeadline(gId.first.Seconds());
        item->SetValue(gId.second);
    }
    for (auto&& balance : Balance2Week) {
        NDrive::NProto::TGameTagData::TTimedItem* item = proto.MutableGameData()->AddDeadline();
        item->SetDeadline(balance.first.Seconds());
        item->SetValue(balance.second);
    }
    return proto;
}


NDrive::TScheme TGameTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("balance", "balance");
    return result;
}


void TGameTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    json["session_id"] = GetSessionId();
    json["balance"] = Balance;
    NJson::TJsonValue& idsJson = json.InsertValue("ids", NJson::JSON_ARRAY);
    for (auto[ts, id] : GameIds) {
        NJson::TJsonValue tmp;
        tmp["deadline"] = ts.Seconds();
        tmp["id"] = id;
        idsJson.AppendValue(tmp);
    }
    NJson::TJsonValue& balanceJson = json.InsertValue("point", NJson::JSON_ARRAY);
    for (auto[ts, balance] : Balance2Week) {
        NJson::TJsonValue tmp;
        tmp["deadline"] = ts.Seconds();
        tmp["count"] = balance;
        balanceJson.AppendValue(tmp);
    }
}


bool TGameTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* /*errors*/) {
    Balance = json["balance"].GetUInteger();

    for (auto&& item : json["ids"].GetArray()) {
        GameIds[TInstant::Seconds(item["deadline"].GetUInteger())] = item["id"].GetUInteger();
    }

    for (auto&& item : json["point"].GetArray()) {
        GameIds[TInstant::Seconds(item["deadline"].GetUInteger())] = item["count"].GetUInteger();
    }
    return true;
}


bool TGameTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    for (auto&& item : proto.GetGameData().GetGameId()) {
        GameIds[TInstant::Seconds(item.GetDeadline())] = item.GetValue();
    }
    for (auto&& item : proto.GetGameData().GetDeadline()) {
        Balance2Week[TInstant::Seconds(item.GetDeadline())] = item.GetValue();
    }
    Balance = proto.GetGameData().GetBalance();
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}


NJson::TJsonValue TGameTagDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result;
    result["point_limit"] = DeltaLimit;
    auto& datesJson = result.InsertValue("dates", NJson::JSON_ARRAY);
    for (auto&& instant : Deadlines) {
        NJson::TJsonValue d;
        d["date"] = instant.Seconds();
        datesJson.AppendValue(d);
    }
    return result;
}


bool TGameTagDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    JREAD_UINT_OPT(jsonMeta, "point_limit", DeltaLimit);
    const NJson::TJsonValue& dates = jsonMeta["dates"];
    if (!dates.IsArray()) {
        return false;
    }
    for (auto&& d : dates.GetArray()) {
        TInstant deadline;
        JREAD_INSTANT(d, "date", deadline);
        if (!Deadlines.empty() && Deadlines.back() >= deadline) {
            return false;
        }
        Deadlines.push_back(deadline);
    }
    return true;
}
