#include "head_account.h"

#include <drive/backend/cars/hardware.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/logging/evlog.h>
#include <drive/backend/roles/permissions.h>
#include <drive/backend/tags/tags.h>

#include <util/digest/fnv.h>
#include <util/string/type.h>

TSet<TString> AvailableAuthStatuses = { "new", "authorized", "not_authorized" };

namespace {
    TNamedSignalSimple HeadAppLaunchApp("head-app-launch-app");
    TNamedSignalSimple HeadAppSetPassportKey("head-app-set-pkey");
    TNamedSignalSimple HeadAppGetPassportKey("head-app-get-key");
    TNamedSignalSimple PassportSessionStart("head-passport-session-start");
    TNamedSignalSimple PassportSessionStop("head-passport-session-stop");
}

TString TShortSessionsManager::GetPublicKey(const TString& userId, const TInstant lastInstant) const {
    auto pKey = PublicKeys.GetCustomObject(userId, lastInstant);
    if (!pKey) {
        auto evlog = NDrive::GetThreadEventLogger();
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "GetPublicKeyError")
                ("user_id", userId)
            );
        }
        return {};
    }
    return pKey->GetPublicKey();
}

bool TShortSessionsManager::StartSession(const TString& userId, NDrive::TEntitySession& session) const {
    auto table = HistoryManager->GetDatabase().GetTable(TSessionInfo::GetTableName());
    NStorage::TRecordBuilder record;
    record("user_id", userId)("session_key", CreateGuidAsString());
    record("expire_at", (TInstant::Now() + SessionKeyLifetime).Seconds());

    NStorage::TObjectRecordsSet<TSessionInfo> affected;
    if (!table->AddRow(record, session.GetTransaction(), "", &affected)) {
        return false;
    }
    for (auto&& historyRec : affected) {
        if (!GetHistoryManager().AddHistory(historyRec, userId, EObjectHistoryAction::Add, session)) {
            return false;
        }
    }
    return true;
}

bool TShortSessionsManager::ResetSessions(NDrive::TEntitySession& session, const TString& robotId, const TDuration delta) const {
    auto table = HistoryManager->GetDatabase().GetTable(TSessionInfo::GetTableName());

    NStorage::TObjectRecordsSet<TSessionInfo> affected;
    TInstant currentTime = Now() + delta;
    if (!table->UpdateRow("expire_at < " + ::ToString(currentTime.Seconds()),
        "(expire_at, session_key)=(" + ToString((currentTime + SessionKeyLifetime).Seconds())+ ", uuid_generate_v4())",
        session.GetTransaction(),
        &affected))
    {
        return false;
    }

    for (auto&& historyRec : affected) {
        if (!GetHistoryManager().AddHistory(historyRec, robotId, EObjectHistoryAction::UpdateData, session)) {
            return false;
        }
    }
    return true;
}

bool TShortSessionsManager::FinishSession(const TString& userId, NDrive::TEntitySession& session) const {
    auto table = HistoryManager->GetDatabase().GetTable(TSessionInfo::GetTableName());
    NStorage::TObjectRecordsSet<TSessionInfo> affected;
    if (!table->RemoveRow(NStorage::TRecordBuilder("user_id", userId), session.GetTransaction(), &affected)) {
        return false;
    }
    for (auto&& historyRec : affected) {
        if (!GetHistoryManager().AddHistory(historyRec, userId, EObjectHistoryAction::Remove, session)) {
            return false;
        }
    }
    return true;
}

bool TShortSessionsManager::RegisterPublicKey(const TString& userId, const TString& publicKey, const TString& author, NDrive::TEntitySession& session) const {
    auto table = HistoryManager->GetDatabase().GetTable("public_keys");
    NStorage::TObjectRecordsSet<TPublicKeyInfo> affected;
    if (!table->AddRow(NStorage::TRecordBuilder("owner_id", userId)("public_key", publicKey), session.GetTransaction(), "", &affected)) {
        return false;
    }

    for (auto&& historyRec : affected) {
        if (!PublicKeys.GetHistoryManager().AddHistory(historyRec, author, EObjectHistoryAction::Add, session)) {
            return false;
        }
    }

    return true;
}

bool TShortSessionsManager::DropPublicKey(const TString& userId, const TString& author, NDrive::TEntitySession& session) const {
    auto table = HistoryManager->GetDatabase().GetTable(TPublicKeyInfo::GetTableName());
    NStorage::TObjectRecordsSet<TPublicKeyInfo> affected;
    auto result = table->RemoveRow(NStorage::TRecordBuilder("owner_id", userId), session.GetTransaction(), &affected);
    for (auto&& historyRec : affected) {
        if (!PublicKeys.GetHistoryManager().AddHistory(historyRec, author, EObjectHistoryAction::Remove, session)) {
            return false;
        }
    }

    return true;
}

void THeadAccountManager::StartPassportSession(const TUserPermissions& permissions, const TString& sessionId, const TString& carId) const {
    if (!Passport) {
        return;
    }
    const TString headId = GetHeadIdByCar(carId);
    if (!headId) {
        return;
    }
    NDrive::TEventLog::Log("StartingPassportSession", NJson::TMapBuilder
        ("car_id", carId)
        ("head_id", headId)
        ("session_id", sessionId)
    );
    PassportSessionStart.Signal(1);
    auto eventLogState = NDrive::TEventLog::CaptureState();
    auto started = Passport->StartSessionAsync(permissions.GetUserFeatures().GetTVMTicket(), permissions.GetUserFeatures().GetClientIP(), sessionId, headId);
    started.Subscribe([carId, headId, sessionId, eventLogState = std::move(eventLogState)](const NThreading::TFuture<void>& started) {
        NDrive::TEventLog::TStateGuard stateGuard(eventLogState);
        NJson::TJsonValue ev = NJson::TMapBuilder
            ("car_id", carId)
            ("head_id", headId)
            ("session_id", sessionId)
        ;
        if (started.HasValue()) {
            NDrive::TEventLog::Log("StartedPassportSession", ev);
        } else {
            ev["exception"] = NJson::ToJson(NJson::JsonString(NThreading::GetExceptionMessage(started)));
            NDrive::TEventLog::Log("StartPassportSessionError", ev);
        }
    });
}

void THeadAccountManager::StopPassportSession(const TUserPermissions& permissions, const TString& sessionId, const TString& carId) const {
    if (!Passport) {
        return;
    }
    const TString headId = GetHeadIdByCar(carId);
    if (!headId) {
        return;
    }
    NDrive::TEventLog::Log("StoppingPassportSession", NJson::TMapBuilder
        ("car_id", carId)
        ("head_id", headId)
        ("session_id", sessionId)
    );
    PassportSessionStop.Signal(1);
    auto eventLogState = NDrive::TEventLog::CaptureState();
    auto stopped = Passport->StopSessionAsync(permissions.GetUserFeatures().GetTVMTicket(), permissions.GetUserFeatures().GetClientIP(), sessionId, headId);
    stopped.Subscribe([carId, headId, sessionId, eventLogState = std::move(eventLogState)](const NThreading::TFuture<void>& stopped) {
        NDrive::TEventLog::TStateGuard stateGuard(eventLogState);
        NJson::TJsonValue ev = NJson::TMapBuilder
            ("car_id", carId)
            ("head_id", headId)
            ("session_id", sessionId)
        ;
        if (stopped.HasValue()) {
            NDrive::TEventLog::Log("StoppedPassportSession", ev);
        } else {
            ev["exception"] = NJson::ToJson(NJson::JsonString(NThreading::GetExceptionMessage(stopped)));
            NDrive::TEventLog::Log("StopPassportSessionError", ev);
        }
    });
}

TString THeadAccountManager::GetDriveUserId(const TString& headId, NDrive::TEntitySession& session) const {
    auto table = Database->GetTable("head_app_sessions");
    NStorage::TObjectRecordsSet<THeadSession> records;
    table->GetRows(NStorage::TRecordBuilder("head_id", headId), records, session.GetTransaction());
    if (records.size() != 1) {
        return Default<TString>();
    }
    return records.front().GetUserId();
}

bool THeadAccountManager::CreateSession(const TString& userId, const TString& carId, bool launchAppFlag, NDrive::TEntitySession& session) const {
    if (!FinishSession(carId, session)) {
        return false;
    }

    auto table = Database->GetTable("head_app_sessions");
    NStorage::TTableRecord record;
    record.Set("car_id", carId);
    record.Set("user_id", userId);
    record.Set("dummy_id", "");
    record.Set("passport_key", "");
    record.Set("launch_app", launchAppFlag);

    const TString headId = GetHeadIdByCar(carId);
    if (headId.empty()) {
        // For cars without associated heads
        return true;
    }
    record.Set("head_id", headId);

    const TString headDeviceId = GetHeadDeviceIdByCar(carId);
    record.Set("head_device_id", headDeviceId);

    NStorage::TObjectRecordsSet<THeadSession> affected;
    auto result = table->AddRow(record, session.GetTransaction(), "", &affected);

    if (result->IsSucceed() && affected.size() == 1) {
        if (!HistoryManager.AddHistory(affected.front(), userId, EObjectHistoryAction::Add, session)) {
            return false;
        }
    } else {
        session.SetErrorInfo("head_manager", "Can't add session record", EDriveSessionResult::InternalError);
        return false;
    }

    return AuthManager.StartSession(headId, session);
}

bool THeadAccountManager::FinishSession(const TString& carId, NDrive::TEntitySession& session) const {
    auto table = Database->GetTable("head_app_sessions");
    NStorage::TObjectRecordsSet<THeadSession> affected;
    auto result = table->RemoveRow(NStorage::TRecordBuilder("car_id", carId), session.GetTransaction(), &affected);
    if (!result || !result->IsSucceed()) {
        ERROR_LOG << session.GetStringReport() << Endl;
        return false;
    }
    if (affected.size() == 1) {
        if (!HistoryManager.AddHistory(affected.front(), affected.front().GetUserId(), EObjectHistoryAction::Remove, session)) {
            return false;
        }
    } else if (affected.size() == 0) {
        // For cars without associated heads
        return true;
    } else {
        session.SetResult(EDriveSessionResult::InconsistencyUser);
        return false;
    }
    return AuthManager.FinishSession(affected.front().GetHeadId(), session);
}


bool THeadAccountManager::RegisterHead(const TString& headId, const TString& publicKey, const TString& author, NDrive::TEntitySession& session) const {
    const TString linkedCar = AssignmentsDB.GetAttachmentOwnerByServiceAppSlug(headId);
    if (!linkedCar) {
        session.SetErrorInfo("register_head", "incorrect linked car", EDriveSessionResult::IncorrectHeadId);
        return false;
    }
    return AuthManager.RegisterPublicKey(headId, publicKey, author, session);
}

bool THeadAccountManager::UpdateCarHeadId(const TString& carId, const TString& headId, const TString& userId, NDrive::TEntitySession& session, const NDrive::IServer* server) const {
    TCarHardwareHead* impl = new TCarHardwareHead(headId);
    TCarGenericAttachment headAttachment(impl);
    return AssignmentsDB.Attach(headAttachment, carId, userId, session, server);
}

TString THeadAccountManager::GetHeadIdByCar(const TString& carId, const TInstant actuality) const {
    TCarGenericAttachment attachment;
    if (AssignmentsDB.TryGetAttachmentOfType(carId, EDocumentAttachmentType::CarHardwareHead, attachment, actuality)) {
        auto headImpl = dynamic_cast<const TCarHardwareHead*>(attachment.Get());
        if (headImpl) {
            return headImpl->GetHeadId();
        }
    }
    return "";
}

TString THeadAccountManager::GetHeadDeviceIdByCar(const TString& carId, const TInstant actuality) const {
    TCarGenericAttachment attachment;
    if (AssignmentsDB.TryGetAttachmentOfType(carId, EDocumentAttachmentType::CarHardwareHeadNew, attachment, actuality)) {
        auto headImpl = dynamic_cast<const TCarHardwareHeadNew*>(attachment.Get());
        if (headImpl) {
            return headImpl->GetHeadId();
        }
    }
    return "";
}

bool THeadAccountManager::GetHeadIds(const TVector<TString>& ids, TMap<TString, TString>& result) const {
    result.clear();
    auto mapToAttachments = AssignmentsDB.GetAssignmentsOfType(ids, EDocumentAttachmentType::CarHardwareHead, TInstant::Zero());
    for (auto it : mapToAttachments) {
        auto headImpl = dynamic_cast<const TCarHardwareHead*>(it.second.Get());
        if (headImpl) {
            result[it.first] = headImpl->GetHeadId();
        }
    }
    return true;
}

bool THeadAccountManager::GetHeadDeviceIds(const TVector<TString>& ids, TMap<TString, TString>& result) const {
    result.clear();
    auto mapToAttachments = AssignmentsDB.GetAssignmentsOfType(ids, EDocumentAttachmentType::CarHardwareHeadNew, TInstant::Zero());
    for (auto it : mapToAttachments) {
        auto headImpl = dynamic_cast<const TCarHardwareHead*>(it.second.Get());
        if (headImpl) {
            result[it.first] = headImpl->GetHeadId();
        }
    }
    return true;
}
