#pragma once

#include "session.h"

#include <drive/backend/roles/permissions.h>

#include <drive/library/cpp/passport/client.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/types/accessor.h>

#include <util/generic/guid.h>
#include <util/generic/string.h>
#include <util/string/type.h>
#include <util/string/vector.h>

class TUserPermissions;
class TCarAttachmentAssignmentDB;

class TShortSessionsManager : public TDBEntitiesCache<TSessionInfo> {
private:
    using TBase = TDBEntitiesCache<TSessionInfo>;

protected:
    virtual bool DoStart() override {
        return
            TBase::DoStart() &&
            PublicKeys.Start();
    }

    virtual bool DoStop() override {
        return
            PublicKeys.Stop() &&
            TBase::DoStop();
    }

public:
    TShortSessionsManager(const IHistoryContext& context, const THistoryConfig& hConfig)
        : TBase(context, hConfig)
        , PublicKeys(context, hConfig)
    {}

    TMaybe<TSessionInfo> GetUserSession(const TString& userId, const TInstant lastInstant = TInstant::Zero()) const {
        return GetCustomObject(userId, lastInstant);
    }

    bool IsRegistered(const TString& userId, TInstant lastInstant = TInstant::Zero()) const {
        return PublicKeys.GetCustomObject(userId, lastInstant).Defined();
    }

    TString GetPublicKey(const TString& userId, const TInstant lastInstant = TInstant::Zero()) const;

    bool StartSession(const TString& userId, NDrive::TEntitySession& session) const;
    bool ResetSessions(NDrive::TEntitySession& session, const TString& robotId, const TDuration delta) const;
    bool FinishSession(const TString& userId, NDrive::TEntitySession& session) const;

    bool RegisterPublicKey(const TString& userId, const TString& publicKey, const TString& author, NDrive::TEntitySession& session) const;
    bool DropPublicKey(const TString& userId, const TString& author, NDrive::TEntitySession& session) const;

private:
    TDuration SessionKeyLifetime = TDuration::Hours(2);

    mutable TPublicKeysCache PublicKeys;
};

enum class EHeadAppMode {
    Disable = 0 /* "disable" */,
    Yandex = 1 /* "yandex" */,
    Enable = 2 /* "all" */,
};

class THeadAccountManager {
public:
    THeadAccountManager(TDatabasePtr database, const TCarAttachmentAssignmentDB& assignmentsDB, const TShortSessionsManager& authManager,
                        const ITagsHistoryContext& context, const TPassportClient* passportClient, const TDuration& standartFreshness = TDuration::Seconds(10))
        : Database(database)
        , AssignmentsDB(assignmentsDB)
        , AuthManager(authManager)
        , HistoryManager(context, TDuration::Hours(1))
        , StandartFreshness(standartFreshness)
        , Passport(passportClient)
    {}

    void StartPassportSession(const TUserPermissions& permissions, const TString& sessionId, const TString& carId) const;
    void StopPassportSession(const TUserPermissions& permissions, const TString& sessionId, const TString& carId) const;

    TString GetDriveUserId(const TString& headId, NDrive::TEntitySession& session) const;

    bool CreateSession(const TString& userId, const TString& carId, bool launchAppFlag, NDrive::TEntitySession& session) const;
    bool FinishSession(const TString& carId, NDrive::TEntitySession& session) const;

    bool RegisterHead(const TString& headId, const TString& publicKey, const TString& author, NDrive::TEntitySession& session) const;

    bool IsRegistered(const TString& userId) const {
        return AuthManager.IsRegistered(userId);
    }

    bool UpdateCarHeadId(const TString& carId, const TString& headId, const TString& userId, NDrive::TEntitySession& session, const NDrive::IServer* server) const;

    TString GetHeadIdByCar(const TString& carId, const TInstant actuality = TInstant::Zero()) const;
    TString GetHeadDeviceIdByCar(const TString& carId, const TInstant actuality = TInstant::Zero()) const;

    bool GetHeadIds(const TVector<TString>& ids, TMap<TString, TString>& result) const;
    bool GetHeadDeviceIds(const TVector<TString>& ids, TMap<TString, TString>& result) const;

private:
    TDatabasePtr Database;
    const TCarAttachmentAssignmentDB& AssignmentsDB;
    const TShortSessionsManager& AuthManager;
    THeadHistoryManager HistoryManager;
    TDuration StandartFreshness;
    const TPassportClient* Passport;
};
