#pragma once

#include "public_key.h"

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/cache.h>
#include <drive/backend/database/history/db_entities.h>
#include <drive/backend/database/history/event.h>
#include <drive/backend/database/history/manager.h>

#include <rtline/util/types/accessor.h>

class ITagsHistoryContext;

class TSessionInfo {
public:
    using TId = TString;

    class TDecoder : public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, UserId, -1);
        R_FIELD(i32, SessionKey, -1);
        R_FIELD(i32, ExpireAt, -1);

    public:
        TDecoder() = default;

        TDecoder(const TMap<TString, ui32>& decoderBase) {
            Id = GetFieldDecodeIndex("id", decoderBase);
            UserId = GetFieldDecodeIndex("user_id", decoderBase);
            SessionKey = GetFieldDecodeIndex("session_key", decoderBase);
            ExpireAt = GetFieldDecodeIndex("expire_at", decoderBase);
        }
    };

    TSessionInfo() = default;

    NStorage::TTableRecord SerializeToTableRecord() const {
        NStorage::TTableRecord result;
        result.Set("id", Id);
        result.Set("user_id", UserId);
        result.Set("session_key", SessionKey);
        result.Set("expire_at", ExpireAt.Seconds());
        return result;
    }

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
        READ_DECODER_VALUE(decoder, values, Id);
        READ_DECODER_VALUE(decoder, values, UserId);
        READ_DECODER_VALUE(decoder, values, SessionKey);
        READ_DECODER_VALUE_INSTANT(decoder, values, ExpireAt);
        return true;
    }

    bool CheckExpired(const TInstant& ts) const {
        return ts >= ExpireAt;
    }

    static TString GetHistoryTableName() {
        return "short_sessions_history";
    }

    static TString GetTableName() {
        return "short_sessions";
    }

    const TString& GetInternalId() const {
        return UserId;
    }

    bool operator!() const {
        return true;
    }

private:
    R_READONLY(ui32, Id, 0);
    R_READONLY(TString, UserId);
    R_READONLY(TString, SessionKey);
    R_READONLY(TInstant, ExpireAt, TInstant::Max());
};


class THeadSession {
    R_READONLY(TString, UserId);
    R_READONLY(TString, CarId);
    R_READONLY(TString, DummyId);
    R_READONLY(TString, HeadId);
    R_READONLY(TString, HeadDeviceId);
    R_READONLY(TString, PassportKey);
    R_READONLY(TString, PassportKeyStatus);
    R_READONLY(bool, LaunchAppFlag, false);

public:
    class TDecoder : public TBaseDecoder {
        R_FIELD(i32, UserId, -1);
        R_FIELD(i32, CarId, -1);
        R_FIELD(i32, DummyId, -1);
        R_FIELD(i32, HeadId, -1);
        R_FIELD(i32, HeadDeviceId, -1);
        R_FIELD(i32, PassportKey, -1);
        R_FIELD(i32, PassportKeyStatus, -1);
        R_FIELD(i32, LaunchAppFlag, -1);

    public:
        TDecoder() = default;

        TDecoder(const TMap<TString, ui32>& decoderBase) {
            UserId = GetFieldDecodeIndex("user_id", decoderBase);
            CarId = GetFieldDecodeIndex("car_id", decoderBase);
            DummyId = GetFieldDecodeIndex("dummy_id", decoderBase);
            HeadId = GetFieldDecodeIndex("head_id", decoderBase);
            HeadDeviceId = GetFieldDecodeIndex("head_device_id", decoderBase);
            PassportKey = GetFieldDecodeIndex("passport_key", decoderBase);
            PassportKeyStatus = GetFieldDecodeIndex("passport_key_status", decoderBase);
            LaunchAppFlag = GetFieldDecodeIndex("launch_app", decoderBase);
        }
    };

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    bool Parse(const NStorage::TTableRecord& record);
    NStorage::TTableRecord SerializeToTableRecord() const;
    const TStringBuf GetHistoryObjectId(const THeadSession& e) const {
        return e.CarId;
    }
};

class THeadHistoryManager : public TDatabaseHistoryManager<THeadSession> {
private:
    using TBase = TDatabaseHistoryManager<THeadSession>;
public:
    THeadHistoryManager(const IHistoryContext& context, const TDuration /*historyDeep*/)
        : TBase(context, "head_app_sessions_history")
    {
    }
};
