#include "history_iterator.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/offer.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/device_snapshot/snapshot.h>
#include <drive/backend/sessions/manager/billing.h>

#include <iterator>
#include <rtline/library/json/merge.h>

#include <library/cpp/protobuf/json/proto2json.h>

inline const TString RidingCompilationType = "riding_compilation";
inline const TString EventsSessionType = "events_session";
inline const TString UniversalType = "universal";

void TShortSessionReportBuilder::DoBuildReportItem(const TConstDBTag* ev, NJson::TJsonValue& item) {
    auto chargableTag = ev ? ev->GetTagAs<TChargableTag>() : nullptr;
    if (chargableTag) {
        item.InsertValue("tag_name", chargableTag->GetName());
        if (chargableTag->GetTransformationSkippedByExternalCommand()) {
            item.InsertValue("transformation_skipped_by_external_command", true);
        }
    }
}

THistoryRidesIterator::THistoryRidesIterator(
    const TVector<TAtomicSharedPtr<TObjectEvent<TMinimalCompiledRiding>>>& rides,
    const TVector<IEventsSession<TCarTagHistoryEvent>::TConstPtr>& sessions,
    const TMap<TString, TPaymentsData>& payments,
    const TTaggedObjectsSnapshot& traceTags,
    const TMap<TString, TVector<NDrive::NFine::TAutocodeFineEntry>>& fines,
    const NDrive::IServer* server
)
    : CIt(rides.rbegin())
    , CEnd(rides.rend())
    , SIt(sessions.rbegin())
    , SEnd(sessions.rend())
    , Server(server)
    , PaymentsData(payments)
    , TraceTags(traceTags)
    , Fines(fines)
{
}

bool THistoryRidesIterator::MoveCompiledIterator(THistoryRideObject& result) {
    while (CIt != CEnd && result.GetSessionId() == (*CIt)->GetSessionId()) {
        if (result.GetSumPrice() < (*CIt)->GetSumPrice() || (!result.IsUserConstruction() && THistoryRideObject(*CIt, {}, {}, {}, Server).IsUserConstruction())) {
            result.SetRiding(*CIt);
        }
        ++CIt;
    }
    return true;
}

bool THistoryRidesIterator::GetAndNext(THistoryRideObject& result) {
    while (CIt != CEnd || SIt != SEnd) {
        TMaybe<TPaymentsData> sPaymentsData;
        TMaybe<TPaymentsData> cPaymentsData;
        TMaybe<TTaggedObject> sTraceTags;
        TMaybe<TTaggedObject> cTraceTags;
        TVector<NDrive::NFine::TAutocodeFineEntry> sFines;
        TVector<NDrive::NFine::TAutocodeFineEntry> cFines;
        if (SIt != SEnd) {
            auto taskIt = PaymentsData.find((*SIt)->GetSessionId());
            sPaymentsData = (taskIt == PaymentsData.end()) ? TMaybe<TPaymentsData>() : taskIt->second;
            auto tagsPtr = TraceTags.Get((*SIt)->GetSessionId());
            sTraceTags = tagsPtr ? *tagsPtr : TMaybe<TTaggedObject>();
            auto finesPtr = Fines.find((*SIt)->GetSessionId());
            sFines = (finesPtr != Fines.end()) ? finesPtr->second : TVector<NDrive::NFine::TAutocodeFineEntry>();
        }
        if (CIt != CEnd) {
            auto taskIt = PaymentsData.find((*CIt)->GetSessionId());
            cPaymentsData = (taskIt == PaymentsData.end()) ? TMaybe<TPaymentsData>() : taskIt->second;
            auto tagsPtr = TraceTags.Get((*CIt)->GetSessionId());
            cTraceTags = tagsPtr ? *tagsPtr : TMaybe<TTaggedObject>();
            auto finesPtr = Fines.find((*CIt)->GetSessionId());
            cFines = (finesPtr != Fines.end()) ? finesPtr->second : TVector<NDrive::NFine::TAutocodeFineEntry>();
        }
        if (CIt == CEnd) {
            result = THistoryRideObject(*SIt, std::move(sPaymentsData), std::move(sTraceTags), std::move(sFines), Server);
            ++SIt;
        } else if (SIt == SEnd) {
            result = THistoryRideObject(*CIt, std::move(cPaymentsData), std::move(cTraceTags), std::move(cFines), Server);
            MoveCompiledIterator(result);
        } else if ((*SIt)->GetStartTS() < (*CIt)->GetStartInstant().Get()) {
            result = THistoryRideObject(*CIt, std::move(cPaymentsData), std::move(cTraceTags), std::move(cFines), Server);
            MoveCompiledIterator(result);
        } else if ((*SIt)->GetStartTS() > (*CIt)->GetStartInstant().Get()) {
            result = THistoryRideObject(*SIt, std::move(sPaymentsData), std::move(sTraceTags), std::move(sFines), Server);
            ++SIt;
        } else {
            result = THistoryRideObject(*CIt, *SIt, std::move(sPaymentsData), std::move(sTraceTags), std::move(sFines), Server);
            ++SIt;
            MoveCompiledIterator(result);
        }
        if (ReadyObjects.emplace(result.GetSessionId()).second) {
            return true;
        }
    }
    return false;
}

TMaybe<THistoryRideObject> THistoryRidesIterator::GetAndNext() {
    THistoryRideObject result;
    if (GetAndNext(result)) {
        return std::move(result);
    } else {
        return {};
    }
}

TString THistoryRideObject::GetObjectModel() const {
    auto offer = GetOffer();
    if (offer) {
        return offer->GetObjectModel();
    } else {
        return {};
    }
}

IOffer::TPtr THistoryRideObject::GetOffer() const {
    if (FullRiding) {
        return FullRiding->GetOffer();
    }
    if (Riding) {
        auto fullRiding = GetFullCompiledRiding();
        if (fullRiding) {
            return fullRiding->GetOffer();
        }
    }
    auto session = GetBillingSession();
    if (session) {
        return session->GetCurrentOffer();
    }
    return nullptr;
}

TMaybe<double> THistoryRideObject::GetDistance() const {
    if (FullRiding) {
        if (FullRiding->HasSnapshotsDiff()) {
            return FullRiding->GetSnapshotsDiffRef().OptionalMileage();
        } else {
            return {};
        }
    }
    auto startEvent = EvSession ? EvSession->GetFirstEvent() : Nothing();
    auto startSnapshot = startEvent ? startEvent.GetRef()->GetObjectSnapshotAs<THistoryDeviceSnapshot>() : nullptr;
    auto startMileage = startSnapshot ? startSnapshot->GetMileage() : Nothing();
    auto lastEvent = EvSession ? EvSession->GetLastEvent() : Nothing();
    auto lastSnapshot = lastEvent ? lastEvent.GetRef()->GetObjectSnapshotAs<THistoryDeviceSnapshot>() : nullptr;
    auto lastMileage = lastSnapshot ? lastSnapshot->GetMileage() : Nothing();
    if (startMileage && lastMileage && lastMileage >= startMileage) {
        return *lastMileage - *startMileage;
    }
    return {};
}

TMaybe<NDrive::TLocation> THistoryRideObject::GetStartLocation() const {
    if (FullRiding) {
        if (FullRiding->HasSnapshotsDiff() && FullRiding->GetSnapshotsDiffUnsafe().HasStart()) {
            return FullRiding->GetSnapshotsDiffUnsafe().GetStartUnsafe();
        } else {
            return {};
        }
    }
    auto startEvent = EvSession ? EvSession->GetFirstEvent() : Nothing();
    if (startEvent) {
        auto snapshot = (*startEvent)->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
        if (snapshot) {
            return snapshot->GetHistoryLocation();
        }
    }
    return {};
}

TMaybe<NDrive::TLocation> THistoryRideObject::GetLastLocation() const {
    if (FullRiding) {
        if (FullRiding->HasSnapshotsDiff() && FullRiding->GetSnapshotsDiffUnsafe().HasLast()) {
            return FullRiding->GetSnapshotsDiffUnsafe().GetLastUnsafe();
        } else {
            return {};
        }
    }
    auto lastEvent = EvSession ? EvSession->GetLastEvent() : Nothing();
    if (lastEvent.Defined()) {
        const THistoryDeviceSnapshot* snapshotCurrent = (*lastEvent)->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
        NDrive::TLocation location;
        if (snapshotCurrent && snapshotCurrent->GetHistoryLocation(location)) {
            return location;
        }
    }
    return {};
}

void THistoryRideObject::FetchFullRiding(const NDrive::IServer* server, TArrayRef<THistoryRideObject> sessions, bool useYDB /*= false*/) {
    TVector<ui64> ridingHistoryIds;
    for (auto&& i : sessions) {
        if (i.GetRiding()) {
            ridingHistoryIds.emplace_back(i.GetRiding()->GetHistoryEventId());
        }
    }
    TMap<ui64, TObjectEvent<TFullCompiledRiding>> fullRidingsIndex;
    const auto& compiledRides = server->GetDriveAPI()->GetMinimalCompiledRides();
    {
        auto tx = compiledRides.BuildSession(true);
        auto optionalCompiledSessions = compiledRides.GetEvents(ridingHistoryIds, tx);
        if (!optionalCompiledSessions) {
            tx.Check();
        }
        for (auto&& session : *optionalCompiledSessions) {
            auto eventId = session.GetHistoryEventId();
            fullRidingsIndex.emplace(eventId, std::move(session));
        }
    }
    if (auto ydb = server->GetYDB(); useYDB && ydb && fullRidingsIndex.size() != sessions.size()) {
        TVector<ui64> requestedIds;
        CopyIf(ridingHistoryIds.begin(), ridingHistoryIds.end(), std::back_inserter(requestedIds), [&fullRidingsIndex](const ui64 id) { return !fullRidingsIndex.contains(id); });
        auto tx = ydb->BuildSession(true);
        auto optionalCompiledSessions = compiledRides.GetEvents(requestedIds, tx);
        if (!optionalCompiledSessions) {
            tx.Check();
        }
        for (auto&& session : *optionalCompiledSessions) {
            auto eventId = session.GetHistoryEventId();
            fullRidingsIndex.emplace(eventId, std::move(session));
        }
    }
    for (auto&& i : sessions) {
        i.FetchFullRiding(fullRidingsIndex);
    }
}

THistoryRideObject& THistoryRideObject::FetchFullRiding(const TMap<ui64, TObjectEvent<TFullCompiledRiding>>& fullRidings) {
    if (Riding) {
        auto it = fullRidings.find(Riding->GetHistoryEventId());
        if (it != fullRidings.end()) {
            FullRiding = new TFullCompiledRiding(it->second);
        }
    }
    return *this;
}

TAtomicSharedPtr<const TBillingSession> TBaseHistoryRideObject::GetBillingSession() const {
    if (EvSession) {
        return std::dynamic_pointer_cast<const TBillingSession>(EvSession);
    } else if (Riding) {
        TMessagesCollector errors;
        return TBillingSession::BuildFromCompiled(Server, *Riding, errors);
    } else {
        return nullptr;
    }
}

TAtomicSharedPtr<TFullCompiledRiding> THistoryRideObject::GetFullCompiledRiding(bool useYDB /*= false*/) const {
    if (FullRiding) {
        return FullRiding;
    } else if (Riding) {
        const auto& compiledRides = Server->GetDriveAPI()->GetMinimalCompiledRides();
        auto tx = compiledRides.BuildSession(true);
        return GetFullCompiledRiding(tx, useYDB);
    } else {
        return Compile();
    }
}

TAtomicSharedPtr<TFullCompiledRiding> THistoryRideObject::GetFullCompiledRiding(NDrive::TEntitySession& tx, bool useYDB /*= false*/) const {
    if (FullRiding) {
        return FullRiding;
    } else if (Riding) {
        const auto& compiledRides = Server->GetDriveAPI()->GetMinimalCompiledRides();
        auto optionalEvent = compiledRides.GetEvent<TFullCompiledRiding>(Riding->GetHistoryEventId(), tx);
        R_ENSURE(optionalEvent, {}, "cannot GetFullCompiledRiding", tx);
        if (auto ydb = Server->GetYDB(); useYDB && ydb && !*optionalEvent) {
            auto tx = ydb->BuildSession(true);
            auto optionalEvent = compiledRides.GetEvent<TFullCompiledRiding>(Riding->GetHistoryEventId(), tx);
            R_ENSURE(optionalEvent, {}, "cannot GetFullCompiledRiding from YDB", tx);
            return *optionalEvent;
        }
        return *optionalEvent;
    } else {
        return Compile();
    }
}

TAtomicSharedPtr<TMinimalCompiledRiding> TBaseHistoryRideObject::GetMinimalCompiledRiding() const {
    if (Riding) {
        return Riding;
    } else {
        return Compile();
    }
}

TAtomicSharedPtr<TFullCompiledRiding> TBaseHistoryRideObject::Compile() const {
    auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(EvSession);
    if (!billingSession) {
        return nullptr;
    }
    return billingSession->BuildCompiledRiding(Server, PaymentsData.Get());
}

TString TBaseHistoryRideObject::GetStage() const {
    auto lastEvent = IsActive() ? EvSession->GetLastEvent() : Nothing();
    if (!lastEvent) {
        return {};
    }
    if (!*lastEvent) {
        return {};
    }

    auto offerContainer = lastEvent->GetTagAs<IOfferContainer>();
    if (!offerContainer) {
        return lastEvent.GetRef()->GetName();
    }

    const auto& objectId = GetObjectId();
    const auto taggedObject = Server->GetDriveDatabase().GetTagsManager().GetDeviceTags().GetObject(objectId);
    return offerContainer->GetStage(taggedObject.Get());
}

NJson::TJsonValue THistoryRideObject::GetOfferProtoReport() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    auto offer = GetOffer();
    if (offer) {
        NProtobufJson::Proto2Json(offer->SerializeToProto(), result);
    }
    return result;
}

double THistoryRideObject::GetSumPrice() const {
    if (!!Riding) {
        return Riding->GetSumPrice();
    } else {
        TMaybe<TBillingSession::TBillingCompilation> bc = EvSession->GetCompilationAs<TBillingSession::TBillingCompilation>();
        if (!bc) {
            return 0;
        }
        return bc->GetReportSumPrice();
    }
}

NJson::TJsonValue THistoryRideObject::GetDiffReport(ELocalization locale, const NDrive::IServer* server) const {
    NJson::TJsonValue result = NJson::JSON_NULL;
    if (!!EvSession) {
        TSnapshotsDiffCompilation snapshotsDiff;
        if (EvSession->FillCompilation(snapshotsDiff)) {
            result = snapshotsDiff.GetReport(locale, server, nullptr);
        }
    } else if (FullRiding && FullRiding->HasSnapshotsDiff()) {
        result = FullRiding->GetSnapshotsDiffUnsafe().GetReport(locale, *server);
    }
    return result;
}

NJson::TJsonValue THistoryRideObject::GetReport(ELocalization locale, ISessionReportCustomization::TPtr customization, TStringBuf type) const {
    auto billingCustomization = std::dynamic_pointer_cast<TBillingReportCustomization>(customization);
    if (billingCustomization) {
        billingCustomization->OptionalPaymentsData() = PaymentsData;
    }
    if (type == EventsSessionType) {
        auto session = GetBillingSession();
        return GetReport(session, locale, customization);
    }
    if (type == RidingCompilationType) {
        auto riding = GetFullCompiledRiding();
        return GetReport(riding, locale);
    }
    if (type == UniversalType) {
        auto riding = GetFullCompiledRiding();
        auto session = GetBillingSession();
        auto ridingReport = GetReport(riding, locale);
        auto sessionReport = GetReport(session, locale, customization);
        return NJson::MergeJson(sessionReport, ridingReport);
    }
    Y_ENSURE_BT(!type, "unknown type: " << type);
    if (!!FullRiding) {
        return GetReport(FullRiding, locale);
    } else if (!!EvSession) {
        return GetReport(EvSession, locale, customization);
    } else {
        return NJson::JSON_NULL;
    }
}

NJson::TJsonValue THistoryRideObject::GetReport(TAtomicSharedPtr<TFullCompiledRiding> riding, ELocalization locale) const {
    if (riding) {
        auto result = riding->GetReport(locale, NDriveSession::ReportAll, *Server);
        result.InsertValue("__type", RidingCompilationType);
        return result;
    } else {
        return NJson::JSON_NULL;
    }
}

NJson::TJsonValue THistoryRideObject::GetReport(IEventsSession<TCarTagHistoryEvent>::TConstPtr session, ELocalization locale, ISessionReportCustomization::TPtr customization) const {
    if (session) {
        auto result = session->GetReport(locale, Server, customization);
        result.InsertValue("events", session->GetTimelineReport<TShortSessionReportBuilder>());
        result.InsertValue("__type", EventsSessionType);
        return result;
    } else {
        return NJson::JSON_NULL;
    }
}

bool CollectFines(
    const NDrive::NFine::TFinesManager& manager,
    const NDrive::NFine::TFineFilterGroup& filters,
    THistoryRidesContext::TFinesMap& finesMap,
    NDrive::TEntitySession& tx
) {
    if (auto fines = manager.GetFines(filters, tx)) {
        finesMap.clear();
        for (auto&& fine : *fines) {
            finesMap[fine.GetSessionId()].push_back(fine);
        }
        return true;
    }
    return false;
}

bool THistoryRidesContext::InitializeSessions(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx) {
    TSet<TString> sessionIdsSet;
    for (const auto& sessionId : sessionIds) {
        if (!sessionId) {
            continue;
        }
        sessionIdsSet.emplace(sessionId);
        if (!SimpleInitializeSession(sessionId, tx, ydbTx)) {
            return false;
        }
    }
    std::sort(ReportSessions.begin(), ReportSessions.end(), [](const auto& left, const auto& right) {
        if (!left) {
            return true;
        }
        if (!right) {
            return false;
        }
        return std::make_tuple(left->GetLastTS(), left->GetStartTS()) < std::make_tuple(right->GetLastTS(), right->GetStartTS());
    });
    std::sort(CompiledRides.begin(), CompiledRides.end(), [](const auto& left, const auto& right) {
        if (!left) {
            return true;
        }
        if (!right) {
            return false;
        }
        return std::make_tuple(left->GetFinishInstant().Get(), left->GetStartInstant().Get()) < std::make_tuple(right->GetFinishInstant().Get(), right->GetStartInstant().Get());
    });

    auto paymentsData = DriveApi.GetBillingManager().GetPaymentsManager().GetFinishedPayments(sessionIds, tx);
    if (!paymentsData) {
        return false;
    }
    for (auto&& payment : *paymentsData) {
        PaymentsData.emplace(std::move(payment));
    }

    if (FetchTraceTags) {
        if (!NDrive::GetFullEntityTags(DriveApi.GetTagsManager().GetTraceTags(), sessionIdsSet, TraceTags, tx, ydbTx)){
            return false;
        }
    }
    if (FetchFines) {
        NDrive::NFine::TFineFilterGroup filters = {
            MakeAtomicShared<NDrive::NFine::TFineChargeableFilter>(),
            MakeAtomicShared<NDrive::NFine::TFineMultipleSessionsFilter>(sessionIdsSet),
        };
        if (!CollectFines(DriveApi.GetFinesManager(), filters, Fines, tx)) {
            return false;
        }
    }
    return true;
}

bool THistoryRidesContext::InitializeSession(const TString& sessionIdStart, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx) {
    auto builder = DriveApi.GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
    if (!builder) {
        tx.SetErrorInfo("THistoryRidesContext::InitializeSession", "cannot get session builder");
        return false;
    }
    TString sessionId = sessionIdStart;
    TSet<TString> sessionIds;
    TVector<IEventsSession<TCarTagHistoryEvent>::TConstPtr> sessions;
    if (sessionId) {
        sessionIds.emplace(sessionId);
    }
    while (sessionId) {
        auto optionalSession = DriveApi.GetSessionManager().GetSession(sessionId, tx);
        if (!optionalSession) {
            return false;
        }
        auto session = std::move(*optionalSession);
        if (!session) {
            session = builder->GetSession(sessionId);
        }
        if (!session) {
            break;
        }
        sessions.emplace_back(session);

        const TBillingSession* bSession = dynamic_cast<const TBillingSession*>(session.Get());
        sessionId = (bSession && bSession->GetCurrentOffer()) ? bSession->GetCurrentOffer()->GetParentId() : "";
        if (sessionId && !sessionIds.emplace(sessionId).second) {
            sessionId = "";
        }
    }
    std::reverse(sessions.begin(), sessions.end());
    if (ReportSessions.empty()) {
        ReportSessions = std::move(sessions);
    } else {
        ReportSessions.insert(ReportSessions.end(), sessions.begin(), sessions.end());
    }

    auto paymentsData = DriveApi.GetBillingManager().GetPaymentsManager().GetFinishedPayments(MakeVector(sessionIds), tx);
    if (!paymentsData) {
        return false;
    }
    for (auto&& payment : *paymentsData) {
        PaymentsData.emplace(std::move(payment));
    }

    if (FetchTraceTags) {
        if (!NDrive::GetFullEntityTags(DriveApi.GetTagsManager().GetTraceTags(), sessionIds, TraceTags, tx, ydbTx)){
            return false;
        }
    }
    if (FetchFines) {
        NDrive::NFine::TFineFilterGroup filters = {
            MakeAtomicShared<NDrive::NFine::TFineChargeableFilter>(),
            MakeAtomicShared<NDrive::NFine::TFineMultipleSessionsFilter>(sessionIds),
        };
        if (!CollectFines(DriveApi.GetFinesManager(), filters, Fines, tx)) {
            return false;
        }
    }

    TVector<TAtomicSharedPtr<TObjectEvent<TMinimalCompiledRiding>>> rides;
    if (!!sessionId) {
        const auto& compiledRides = DriveApi.GetMinimalCompiledRides();
        auto optionalSessions =  compiledRides.Get<TMinimalCompiledRiding>({ sessionId }, tx, ydbTx, Since);
        if (!optionalSessions) {
            return false;
        }
        for (auto&& session : *optionalSessions) {
            rides.push_back(MakeAtomicShared<TObjectEvent<TMinimalCompiledRiding>>(std::move(session)));
        }
    }
    if (CompiledRides.empty()) {
        CompiledRides = std::move(rides);
    } else {
        CompiledRides.insert(CompiledRides.end(), rides.begin(), rides.end());
    }
    return true;
}

bool THistoryRidesContext::SimpleInitializeSession(const TString& sessionId, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx) {
    if (!sessionId) {
        tx.SetErrorInfo("THistoryRidesContext::InitializeSession", "incorrect session id");
        return false;
    }

    auto optionalSession = DriveApi.GetSessionManager().GetSession(sessionId, tx);
    if (!optionalSession) {
        return false;
    }
    auto session = *optionalSession;
    if (session) {
        ReportSessions.emplace_back(session);
    } else {
        const auto& compiledRides = DriveApi.GetMinimalCompiledRides();
        auto optionalSessions =  compiledRides.Get<TMinimalCompiledRiding>({ sessionId }, tx, ydbTx, Since);
        if (!optionalSessions) {
            return false;
        }
        for (auto&& session : *optionalSessions) {
            CompiledRides.emplace_back(MakeAtomicShared<TObjectEvent<TMinimalCompiledRiding>>(std::move(session)));
        }
    }
    return true;
}

void THistoryRidesContext::FilterReportSessions(TMaybe<TInstant> until, TMaybe<ui64> last) {
    if (until) {
        ReportSessions.erase(std::remove_if(ReportSessions.begin(), ReportSessions.end(), [until](IEventsSession<TCarTagHistoryEvent>::TConstPtr session){ return session->GetStartTS() >= *until; }), ReportSessions.end());
    }
    if (last && ReportSessions.size() > *last) {
        ReportSessions.erase(ReportSessions.begin(), ReportSessions.begin() + (ReportSessions.size() - *last));
    }
}

bool THistoryRidesContext::InitializeCars(const TVector<TString>& carsIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TMaybe<TInstant> until, TMaybe<ui32> numdoc) {
    auto optionalSessions = DriveApi.GetSessionManager().GetObjectsSessions(carsIds, tx);
    if (!optionalSessions) {
        return false;
    }
    ReportSessions = std::move(*optionalSessions);
    FilterReportSessions(until, numdoc);

    {
        const auto& compiledRidesManager = DriveApi.GetMinimalCompiledRides();
        auto optionalSessions = compiledRidesManager.GetObject<TMinimalCompiledRiding>(carsIds, tx, ydbTx, { Since, until }, numdoc);
        if (!optionalSessions) {
            return false;
        }
        for (auto&& session : *optionalSessions) {
            CompiledRides.push_back(MakeAtomicShared<TObjectEvent<TMinimalCompiledRiding>>(std::move(session)));
        }
    }

    TVector<TString> sessionIds;
    Transform(ReportSessions.begin(), ReportSessions.end(), std::back_inserter(sessionIds), [](IEventsSession<TCarTagHistoryEvent>::TConstPtr item) { return item->GetSessionId(); });
    Transform(CompiledRides.begin(), CompiledRides.end(), std::back_inserter(sessionIds), [](TAtomicSharedPtr<TObjectEvent<TMinimalCompiledRiding>> item) { return item->GetSessionId(); });
    auto paymentsData = DriveApi.GetBillingManager().GetPaymentsManager().GetFinishedPayments(sessionIds, tx);
    if (!paymentsData) {
        return false;
    }
    PaymentsData = std::move(paymentsData.GetRef());
    if (FetchTraceTags) {
        TSet<TString> ids(sessionIds.begin(), sessionIds.end());
        if (!NDrive::GetFullEntityTags(DriveApi.GetTagsManager().GetTraceTags(), ids, TraceTags, tx, ydbTx)){
            return false;
        }
    }
    if (FetchFines) {
        NDrive::NFine::TFineFilterGroup filters = {
            MakeAtomicShared<NDrive::NFine::TFineChargeableFilter>(),
            MakeAtomicShared<NDrive::NFine::TFineMultipleCarsFilter>(MakeSet(carsIds)),
        };
        if (!CollectFines(DriveApi.GetFinesManager(), filters, Fines, tx)) {
            return false;
        }
    }
    return true;
}

bool THistoryRidesContext::InitializeUser(const TString& userId, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TMaybe<TInstant> until, TMaybe<ui32> numdoc) {
    auto optionalSessions = DriveApi.GetSessionManager().GetUserSessions(userId, tx);
    if (!optionalSessions) {
        return false;
    }
    ReportSessions = std::move(*optionalSessions);
    FilterReportSessions(until, numdoc);

    {
        const auto& compiledRidesManager = DriveApi.GetMinimalCompiledRides();
        auto optionalSessions =  compiledRidesManager.GetUser<TMinimalCompiledRiding>(userId, tx, ydbTx, { Since, until }, numdoc);
        if (!optionalSessions) {
            return false;
        }
        for (auto&& session : *optionalSessions) {
            CompiledRides.push_back(MakeAtomicShared<TObjectEvent<TMinimalCompiledRiding>>(std::move(session)));
        }
    }

    TVector<TString> sessionIds;
    Transform(ReportSessions.begin(), ReportSessions.end(), std::back_inserter(sessionIds), [](IEventsSession<TCarTagHistoryEvent>::TConstPtr item) { return item->GetSessionId(); });
    Transform(CompiledRides.begin(), CompiledRides.end(), std::back_inserter(sessionIds), [](TAtomicSharedPtr<TObjectEvent<TMinimalCompiledRiding>> item) { return item->GetSessionId(); });
    auto payments = DriveApi.GetBillingManager().GetPaymentsManager().GetFinishedPayments(sessionIds, tx);
    if (!payments) {
        return false;
    }

    PaymentsData = std::move(payments.GetRef());
    if (FetchTraceTags) {
        TSet<TString> ids(sessionIds.begin(), sessionIds.end());
        if (!NDrive::GetFullEntityTags(DriveApi.GetTagsManager().GetTraceTags(), ids, TraceTags, tx, ydbTx)){
            return false;
        }
    }
    if (FetchFines) {
        NDrive::NFine::TFineFilterGroup filters = {
            MakeAtomicShared<NDrive::NFine::TFineChargeableFilter>(),
            MakeAtomicShared<NDrive::NFine::TFineUserFilter>(userId),
        };
        if (!CollectFines(DriveApi.GetFinesManager(), filters, Fines, tx)) {
            return false;
        }
    }
    return true;
}

// FixNumdoc increments numdoc for the has_more flag to work correctly.
void FixNumdoc(TMaybe<ui32>& numdoc) {
    if (numdoc && *numdoc < Max<ui32>() - 5) {
        // TODO(iudovin@): Revert this kostyl to 1, when there are no duplicate compiled rides.
        *numdoc += 5;
    }
}

bool THistoryRidesContext::Initialize(const TString& sessionId, const TString& userId, const TString& carId, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TMaybe<TInstant> until, TMaybe<ui32> numdoc) {
    TVector<TString> carsIds;
    if (!carId.empty()) {
        carsIds.push_back(carId);
    }
    return Initialize(sessionId, userId, std::move(carsIds), tx, ydbTx, until, numdoc);
}

bool THistoryRidesContext::Initialize(const TString& sessionId, const TString& userId, const TVector<TString>& carsIds, NDrive::TEntitySession& tx, NDrive::TEntitySession &ydbTx, TMaybe<TInstant> until, TMaybe<ui32> numdoc) {
    FixNumdoc(numdoc);
    if (sessionId) {
        return InitializeSession(sessionId, tx, ydbTx);
    }
    if (!carsIds.empty()) {
        return InitializeCars(carsIds, tx, ydbTx, until, numdoc);
    }
    if (userId) {
        return InitializeUser(userId, tx, ydbTx, until, numdoc);
    }
    tx.SetErrorInfo("THistoryRidesContext::Initialize", "no initialize id provided");
    return false;
}

bool THistoryRidesContext::Initialize(NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TMaybe<TInstant> until, TMaybe<ui32> numdoc) {
    FixNumdoc(numdoc);
    auto builder = DriveApi.GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
    if (!builder) {
        tx.SetErrorInfo("THistoryRidesContext::Initialize", "cannot get session builder");
        return false;
    }
    auto sessions = builder->GetSessionsActual();
    ReportSessions = { sessions.begin(), sessions.end() };
    FilterReportSessions(until, numdoc);

    auto queryOptions = NSQL::TQueryOptions(numdoc ? *numdoc + 1 : 0, true);
    if (until) {
        auto startRange = MakeRange<ui64>(Nothing(), until->Seconds());
        queryOptions.SetGenericCondition("start", std::move(startRange));
    }

    {
        const auto& compiledRidesManager = DriveApi.GetMinimalCompiledRides();
        auto optionalSessions = compiledRidesManager.GetByTimestamp<TMinimalCompiledRiding>(tx, queryOptions, ydbTx, { Since }, numdoc);
        if (!optionalSessions) {
            return false;
        }
        for (auto&& session : *optionalSessions) {
            CompiledRides.push_back(MakeAtomicShared<TObjectEvent<TMinimalCompiledRiding>>(std::move(session)));
        }
    }

    return true;
}

THistoryRidesContext::THistoryRidesContext(
    const NDrive::IServer& server,
    const TInstant since, /*= TInstant::Zero()*/
    bool fetchTraceTags, /*= false*/
    bool fetchFines /*= false*/
)
    : Server(server)
    , DriveApi(*Server.GetDriveAPI())
    , Since(since)
    , FetchTraceTags(fetchTraceTags)
    , FetchFines(fetchFines)
{
}

TMaybe<THistoryRideObject> THistoryRidesContext::GetSession(const TString& sessionId) const {
    auto iterator = GetIterator();
    THistoryRideObject object;
    while (iterator.GetAndNext(object)) {
        if (object.GetSessionId() == sessionId) {
            return object;
        }
    }
    return {};
}

TVector<THistoryRideObject> THistoryRidesContext::GetSessions(
    const TInstant until,
    const ui32 numdoc,
    bool* hasMore /*= nullptr*/,
    const bool skipEmpty /*= false*/,
    std::function<bool(const THistoryRideObject& ride)> filter /*= nullptr*/
) {
    TVector<THistoryRideObject> result;

    THistoryRidesIterator hrIt = GetIterator();
    if (hasMore) {
        *hasMore = false;
    }
    THistoryRideObject hrObject;
    for (; hrIt.GetAndNext(hrObject);) {
        if (filter && !filter(hrObject)) {
            continue;
        }
        if (hrObject.GetStartTS() >= until) {
            continue;
        }
        if (hrObject.GetLastTS() < Since) {
            break;
        }
        if (skipEmpty && hrObject.GetSumPrice() <= 0) {
            continue;
        }
        if (result.size() >= numdoc) {
            if (hasMore) {
                *hasMore = true;
            }
            break;
        }
        result.emplace_back(std::move(hrObject));
    }
    return result;
}

bool NDrive::GetFullEntityTags(const IEntityTagsManager& tagsManager, const TSet<TString>& sessionIds, TTaggedObjectsSnapshot& taggedObject, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx) {
    if (!tagsManager.RestoreObjects(sessionIds, taggedObject, tx)) {
        return false;
    }
    TSet<TString> additionalIds;
    for (const auto& [id, object] : taggedObject) {
        if (object.GetTags().empty()) {
            additionalIds.insert(id);
        }
    }
    if (!additionalIds.empty() && ydbTx) {
        TMap<TString, TTaggedObject> objects;
        if (!tagsManager.RestoreObjects(additionalIds, objects, ydbTx)) {
            return false;
        }
        for (const auto& [id, object] : taggedObject) {
            if (!object.GetTags().empty()) {
                objects[id] = object;
            }
        }
        taggedObject.Reset(std::move(objects));
    }
    return true;
}
