#pragma once

#include <drive/backend/billing/interfaces/payments.h>
#include <drive/backend/compiled_riding/compiled_riding.h>
#include <drive/backend/database/history/session.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/fines/manager.h>

class TBillingSession;

class TShortSessionReportBuilder {
public:
    static void DoBuildReportItem(const TConstDBTag* ev, NJson::TJsonValue& item);
};

class TBaseHistoryRideObject {
protected:
    IEventsSession<TCarTagHistoryEvent>::TConstPtr EvSession;
    TAtomicSharedPtr<TObjectEvent<TMinimalCompiledRiding>> Riding;
    const NDrive::IServer* Server = nullptr;
    TMaybe<TPaymentsData> PaymentsData;
    TMaybe<TTaggedObject> TraceTags;
    TVector<NDrive::NFine::TAutocodeFineEntry> Fines;

public:
    TBaseHistoryRideObject() = default;

    TBaseHistoryRideObject(TAtomicSharedPtr<TObjectEvent<TMinimalCompiledRiding>> riding, TMaybe<TPaymentsData>&& task, TMaybe<TTaggedObject>&& tags, TVector<NDrive::NFine::TAutocodeFineEntry>&& fines, const NDrive::IServer* server)
        : Riding(riding)
        , Server(server)
        , PaymentsData(std::move(task))
        , TraceTags(std::move(tags))
        , Fines(std::move(fines))
    {
    }

    TBaseHistoryRideObject(TAtomicSharedPtr<TObjectEvent<TMinimalCompiledRiding>> riding, IEventsSession<TCarTagHistoryEvent>::TConstPtr session, TMaybe<TPaymentsData>&& task, TMaybe<TTaggedObject>&& tags, TVector<NDrive::NFine::TAutocodeFineEntry>&& fines, const NDrive::IServer* server)
        : EvSession(session)
        , Riding(riding)
        , Server(server)
        , PaymentsData(std::move(task))
        , TraceTags(std::move(tags))
        , Fines(std::move(fines))
    {
    }

    TBaseHistoryRideObject(IEventsSession<TCarTagHistoryEvent>::TConstPtr session, TMaybe<TPaymentsData>&& task, TMaybe<TTaggedObject>&& tags, TVector<NDrive::NFine::TAutocodeFineEntry>&& fines, const NDrive::IServer* server)
        : EvSession(session)
        , Server(server)
        , PaymentsData(std::move(task))
        , TraceTags(std::move(tags))
        , Fines(std::move(fines))
    {
    }

    const TString& GetObjectId() const {
        return Riding ? Riding->TCompiledRiding::GetObjectId() : EvSession->GetObjectId();
    }

    const TString& GetUserId() const {
        return Riding ? Riding->GetHistoryUserId() : EvSession->GetUserId();
    }

    const TString& GetSessionId() const {
        return Riding ? Riding->GetSessionId() : EvSession->GetSessionId();
    }

    TInstant GetStartTS() const {
        return Riding ? Riding->GetStartInstant().Get() : EvSession->GetStartTS();
    }

    TInstant GetLastTS() const {
        return Riding ? Riding->GetFinishInstant().Get() : EvSession->GetLastTS();
    }

    TMaybe<TTaggedObject> GetTraceTags() const {
        return TraceTags;
    }

    const TVector<NDrive::NFine::TAutocodeFineEntry>& GetFines() const {
        return Fines;
    }

    bool IsActive() const {
        return EvSession ? !EvSession->GetClosed() : false;
    }

    bool IsUserConstruction() const {
        if (!Riding) {
            return true;
        } else {
            return !Riding->GetHistoryOriginatorId();
        }
    }

    TAtomicSharedPtr<const TBillingSession> GetBillingSession() const;
    TAtomicSharedPtr<TMinimalCompiledRiding> GetMinimalCompiledRiding() const;
    TAtomicSharedPtr<TObjectEvent<TMinimalCompiledRiding>> GetRiding() const {
        return Riding;
    }
    TString GetStage() const;

    void SetRiding(TAtomicSharedPtr<TObjectEvent<TMinimalCompiledRiding>> obj) {
        Riding = obj;
    }

protected:
    TAtomicSharedPtr<TFullCompiledRiding> Compile() const;
};

class THistoryRideObject: public TBaseHistoryRideObject {
private:
    using TBase = TBaseHistoryRideObject;

private:
    TAtomicSharedPtr<TFullCompiledRiding> FullRiding;

private:
    THistoryRideObject& FetchFullRiding(const TMap<ui64, TObjectEvent<TFullCompiledRiding>>& fullRidings);

public:
    static void FetchFullRiding(const NDrive::IServer* server, TArrayRef<THistoryRideObject> sessions, bool useYDB = false);

public:
    using TBase::TBase;

    TString GetObjectModel() const;
    IOffer::TPtr GetOffer() const;
    TMaybe<double> GetDistance() const;
    TMaybe<NDrive::TLocation> GetStartLocation() const;
    TMaybe<NDrive::TLocation> GetLastLocation() const;

    double GetSumPrice() const;

    TAtomicSharedPtr<TFullCompiledRiding> GetFullCompiledRiding(bool useYDB = false) const;
    TAtomicSharedPtr<TFullCompiledRiding> GetFullCompiledRiding(NDrive::TEntitySession& tx, bool useYDB = false) const;

    NJson::TJsonValue GetOfferProtoReport() const;
    NJson::TJsonValue GetDiffReport(ELocalization locale, const NDrive::IServer* server) const;
    NJson::TJsonValue GetReport(ELocalization locale, ISessionReportCustomization::TPtr customization, TStringBuf type = {}) const;

private:
    NJson::TJsonValue GetReport(TAtomicSharedPtr<TFullCompiledRiding> riding, ELocalization locale) const;
    NJson::TJsonValue GetReport(IEventsSession<TCarTagHistoryEvent>::TConstPtr session, ELocalization locale, ISessionReportCustomization::TPtr customization) const;
};

class THistoryRidesIterator {
private:
    TVector<TAtomicSharedPtr<TObjectEvent<TMinimalCompiledRiding>>>::const_reverse_iterator CIt;
    TVector<TAtomicSharedPtr<TObjectEvent<TMinimalCompiledRiding>>>::const_reverse_iterator CEnd;
    TVector<IEventsSession<TCarTagHistoryEvent>::TConstPtr>::const_reverse_iterator SIt;
    TVector<IEventsSession<TCarTagHistoryEvent>::TConstPtr>::const_reverse_iterator SEnd;
    const NDrive::IServer* Server = nullptr;
    TSet<TString> ReadyObjects;
    const TMap<TString, TPaymentsData>& PaymentsData;
    const TTaggedObjectsSnapshot& TraceTags;
    const TMap<TString, TVector<NDrive::NFine::TAutocodeFineEntry>>& Fines;

private:
    bool MoveCompiledIterator(THistoryRideObject& result);

public:
    THistoryRidesIterator(
        const TVector<TAtomicSharedPtr<TObjectEvent<TMinimalCompiledRiding>>>& rides,
        const TVector< ::IEventsSession<TCarTagHistoryEvent>::TConstPtr>& sessions,
        const TMap<TString, TPaymentsData>& payments,
        const TTaggedObjectsSnapshot& traceTags,
        const TMap<TString, TVector<NDrive::NFine::TAutocodeFineEntry>>& fines,
        const NDrive::IServer* server
    );

    TMaybe<THistoryRideObject> GetAndNext();
    bool GetAndNext(THistoryRideObject& result);
};

class TDriveAPI;

class THistoryRidesContext {
public:
    using TFinesMap = TMap<TString, TVector<NDrive::NFine::TAutocodeFineEntry>>;

private:
    const NDrive::IServer& Server;
    const TDriveAPI& DriveApi;
    const TInstant Since;
    const bool FetchTraceTags;
    const bool FetchFines;

    TVector<IEventsSession<TCarTagHistoryEvent>::TConstPtr> ReportSessions;
    TVector<TAtomicSharedPtr<TObjectEvent<TMinimalCompiledRiding>>> CompiledRides;
    TMap<TString, TPaymentsData> PaymentsData;
    TTaggedObjectsSnapshot TraceTags;
    TFinesMap Fines;

public:
    THistoryRidesContext(const NDrive::IServer& server, const TInstant since = TInstant::Zero(), bool fetchTraceTags = false, bool fetchFines = false);

    [[nodiscard]] bool InitializeSessions(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx);
    [[nodiscard]] bool InitializeSession(const TString& sessionId, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx);
    [[nodiscard]] bool InitializeCars(const TVector<TString>& carsIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TMaybe<TInstant> until = {}, TMaybe<ui32> numdoc = {});
    [[nodiscard]] bool InitializeUser(const TString& userId, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TMaybe<TInstant> until = {}, TMaybe<ui32> numdoc = {});
    [[nodiscard]] bool Initialize(const TString& sessionId, const TString& userId, const TString& carId, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TMaybe<TInstant> until = {}, TMaybe<ui32> numdoc = {});
    [[nodiscard]] bool Initialize(const TString& sessionId, const TString& userId, const TVector<TString>& carsIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TMaybe<TInstant> until = {}, TMaybe<ui32> numdoc = {});
    [[nodiscard]] bool Initialize(NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TMaybe<TInstant> until = {}, TMaybe<ui32> numdoc = {});

    THistoryRidesIterator GetIterator() const {
        return THistoryRidesIterator(CompiledRides, ReportSessions, PaymentsData, TraceTags, Fines, &Server);
    }

    TMaybe<THistoryRideObject> GetSession(const TString& sessionId) const;
    TVector<THistoryRideObject> GetSessions(
        const TInstant until,
        const ui32 numdoc,
        bool* hasMore = nullptr,
        const bool skipEmpty = false,
        std::function<bool(const THistoryRideObject& ride)> filter = nullptr
    );

private:
    void FilterReportSessions(TMaybe<TInstant> until, TMaybe<ui64> last);
    [[nodiscard]] bool SimpleInitializeSession(const TString& sessionId, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx);
};

namespace NDrive {
    [[nodiscard]] bool GetFullEntityTags(const IEntityTagsManager& tagsManager, const TSet<TString>& sessionIds, TTaggedObjectsSnapshot& taggedObject, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx);
}
