#include "damage.h"

#include <drive/backend/images/damage.h_serialized.h>
#include <rtline/util/algorithm/container.h>

NJson::TJsonValue TCarDamage::BuildReport() const {
    NJson::TJsonValue report;
    report["description"] = Description;
    report["damage_level"] = ::ToString(Level);
    report["area"] = ::ToString(Area);
    report["discarded"] = Discarded;
    report["is_the_best"] = IsTheBest;
    if (Element) {
        report["element"] = Element;
    }
    return report;
}

bool TCarDamage::ParseFromJson(const NJson::TJsonValue& report, TMessagesCollector& errors) {
    Description = report["description"].GetString();
    Element = report["element"].GetString();
    if (report.Has("is_the_best")) {
        IsTheBest = report["is_the_best"].GetBoolean();
    }

    if (report.Has("damage_level")) {
        if (!TryFromString(report["damage_level"].GetString(), Level)) {
            errors.AddMessage(__LOCATION__, "unknown \"damage_level\"");
            return false;
        }
    }

    if (!TryFromString(report["area"].GetString(), Area)) {
        errors.AddMessage(__LOCATION__, "unknown \"area\"");
        return false;
    }

    return true;
}

bool TCarDamage::ParseFromProto(const NDrive::NProto::TCommonImageMeta::TDamageInfo& proto) {
    Description = proto.GetDescription();
    Discarded = proto.GetDiscarded();
    Element = proto.GetElement();
    IsTheBest = proto.GetTheBest();
    {
        ui32 area = proto.GetArea();
        if (area >= GetEnumItemsCount<EDamageArea>()) {
            ERROR_LOG << "Incorrect EDamageArea " << area << Endl;
            return false;
        }
        Area = static_cast<EDamageArea>(area);
    }
    if (proto.HasLevel()) {
        Level = static_cast<EDamageLevel>(proto.GetLevel());
        auto allLevels = MakeSet(GetEnumAllValues<EDamageLevel>());
        if (!allLevels.contains(Level)) {
            Level = EDamageLevel::Undefined;
        }
    }
    return true;
}

void TCarDamage::SerializeToProto(NDrive::NProto::TCommonImageMeta::TDamageInfo& proto) const {
    proto.SetDiscarded(Discarded);
    proto.SetElement(Element);
    proto.SetTheBest(IsTheBest);
    proto.SetArea(static_cast<ui32>(Area));
    if (Level != EDamageLevel::Undefined) {
        proto.SetLevel(static_cast<ui32>(Level));
    }
    proto.SetDescription(Description);
}
