#include "database.h"

#include <drive/backend/logging/evlog.h>
#include <drive/backend/tags/tags.h>

#include <rtline/library/json/merge.h>

#include <util/generic/guid.h>
#include <util/string/vector.h>

namespace {
    TUnistatSignal<double> CacheHit { { "images-cache-hit" }, false };
    TUnistatSignal<double> CacheMiss { { "images-cache-miss" }, false };
    TUnistatSignal<double> CacheExpired { { "images-cache-expired" }, false };
    TUnistatSignal<double> CacheInvalidated { { "images-cache-invalidated" }, false };
}

TImagesStorage::TImagesStorage(const IHistoryContext& context)
    : TBase(context.GetDatabase())
    , IAutoActualization("images-cache", TDuration::Seconds(10))
    , HistoryManager(context)
    , DefaultLifetime(TDuration::Seconds(1000))
    , ValidatedImagesCache(16 * 1024)
{
    Y_ENSURE_BT(Start());
}

TImagesStorage::~TImagesStorage() {
    if (!Stop()) {
        ERROR_LOG << GetName() << ": cannot stop" << Endl;
    }
}

NDrive::IObjectSnapshot::TPtr TImagesStorage::BuildSnapshot(
    const NJson::TJsonValue& jsonValue,
    const TDBTag& tag,
    NEntityTagsManager::EEntityType objectType,
    const TString& source,
    const TString& originator,
    bool generatePreviewPath,
    NDrive::TEntitySession& session,
    TMaybe<TString> userSessionId,
    const NJson::TJsonValue& extraMetaData
) const {
    return BuildSnapshot(jsonValue, tag.GetObjectId(), objectType, source, originator, generatePreviewPath, session, userSessionId, extraMetaData);
}

NDrive::IObjectSnapshot::TPtr TImagesStorage::BuildSnapshot(
    const NJson::TJsonValue& jsonValue,
    const TString& objectId,
    NEntityTagsManager::EEntityType objectType,
    const TString& source,
    const TString& originator,
    bool generatePreviewPath,
    NDrive::TEntitySession& session,
    TMaybe<TString> userSessionId,
    const NJson::TJsonValue& extraMetaData
) const  {
    const NJson::TJsonValue& photos = jsonValue["photos"];

    auto images = MakeAtomicShared<TImagesSnapshot>();
    {
        TMaybe<TString> comment;
        if (!NJson::ParseField(jsonValue["comment"], comment, session.MutableMessages())) {
            return nullptr;
        }
        images->SetComment(std::move(comment));
    }
    if (photos.IsArray()) {
        for (auto&& photo : photos.GetArray()) {
            auto image = BuildSinglePhoto(photo, objectId, objectType, source ? source : "default", originator, generatePreviewPath, userSessionId, extraMetaData, session);
            if (!image) {
                return nullptr;
            }
            images->AddImage(std::move(*image));
        }
    } else {
        auto image = BuildSinglePhoto(jsonValue, objectId, objectType, source ? source : "default", originator, generatePreviewPath, userSessionId, extraMetaData, session);
        if (!image) {
            return nullptr;
        }
        images->AddImage(std::move(*image));
    }
    return images;
}

TImagesStorage::TOptionalRecords TImagesStorage::Get(TConstArrayRef<TString> objectIds, TRange<TInstant> timestampRange, NDrive::TEntitySession& session) const {
    auto queryOptions = TQueryOptions().SetOrderBy({ "image_id" });
    if (!objectIds.empty()) {
        queryOptions.SetGenericCondition("object_id", MakeSet<TString>(objectIds));
    }
    if (timestampRange) {
        TRange<ui64> seconds;
        seconds.From = timestampRange.From ? MakeMaybe<ui32>(timestampRange.From->Seconds()) : Nothing();
        seconds.To = timestampRange.To ? MakeMaybe<ui32>(timestampRange.To->Seconds()) : Nothing();
        queryOptions.SetGenericCondition("created_at", seconds);
    }
    return Fetch(session, queryOptions);
}

TImagesStorage::TOptionalRecords TImagesStorage::Get(const TString& objectId, TRange<TInstant> timestampRange, NDrive::TEntitySession& session) const {
    return Get(NContainer::Scalar(objectId), std::move(timestampRange), session);
}

TExpectedValidatedImages TImagesStorage::GetCachedValidatedImages(const TString& objectId, TInstant statementDeadline, TMaybe<TDuration> cacheLifetime) const {
    auto impl = [&] {
        auto eventLogger = NDrive::GetThreadEventLogger();
        auto lifetimeKey = "images.cache_lifetime";
        auto lifetime = cacheLifetime;
        if (!lifetime) {
            lifetime = NDrive::HasServer()
                ? NDrive::GetServer().GetSettings().GetValue<TDuration>(lifetimeKey)
                : Nothing();
        }
        auto now = Now();
        auto threshold = now - lifetime.GetOrElse(DefaultLifetime);
        auto optionalObject = ValidatedImagesCache.find(objectId);
        if (optionalObject && optionalObject->GetTimestamp() > threshold) {
            if (eventLogger) {
                eventLogger->AddEvent(NJson::TMapBuilder
                    ("event", "ValidatedImagesCacheHit")
                    ("object_id", objectId)
                    ("timestamp", NJson::ToJson(optionalObject->GetTimestamp()))
                );
            }
            CacheHit.Signal(1);
            return std::move(*optionalObject);
        }
        if (optionalObject) {
            CacheExpired.Signal(1);
        }

        if (eventLogger) {
            eventLogger->AddEvent(NJson::TMapBuilder
                ("event", "ValidatedImagesCacheMiss")
                ("object_id", objectId)
            );
        }

        auto lockTimeout = TDuration::Zero();
        auto statementTimeout = statementDeadline - now;
        auto session = BuildSession(true, false, lockTimeout, statementTimeout);
        auto restoredObject = RestoreValidatedImages(objectId, session);
        if (!restoredObject) {
            session.Check();
        }

        Y_ASSERT(objectId == restoredObject->GetObjectId());
        ValidatedImagesCache.update(objectId, *restoredObject);
        CacheMiss.Signal(1);
        return std::move(*restoredObject);
    };
    return WrapUnexpected<TCodedException>(impl);
}

TOptionalValidatedImages TImagesStorage::RestoreValidatedImages(const TString& objectId, NDrive::TEntitySession& session) const {
    const auto& settings = NDrive::GetServer().GetSettings();
    auto hasActiveMarkup = settings.GetValue<bool>("images.has_active_markup").GetOrElse(false);
    auto timestamp = Now();
    auto allImages = TOptionalRecords();
    if (hasActiveMarkup) {
        allImages = Fetch(session, NSQL::TQueryOptions()
            .AddGenericCondition("object_id", objectId)
            .SetGenericCondition("active_markup", NSQL::Not(false))
        );
    } else {
        allImages = Get(objectId, {}, session);
    }

    if (!allImages) {
        return {};
    }
    std::sort(allImages->begin(), allImages->end(), [](const TCommonImageData& left, const TCommonImageData& right) {
        return left.GetImageId() < right.GetImageId();
    });

    TValidatedImages result;
    result.SetObjectId(objectId);
    result.SetTimestamp(timestamp);
    TMap<TString, ui64> descriptionFrequencies;
    TSet<TString> uniqueElements;
    const auto skipElements = settings.GetValue<bool>("images.skip_element_dups").GetOrElse(false);
    for (auto&& image : *allImages) {
        TSet<TString> supportVerdicts;
        for (const auto& json : image.GetMetaData()["support_verdicts"].GetArray()) {
            if (!json["verdict"].GetString().empty()) {
                supportVerdicts.emplace(json["verdict"].GetString());
            }
        }
        for (auto&& markup : image.GetMarkUpList()) {
            if (markup.IsDiscarded()) {
                continue;
            }
            const auto& description = markup.GetDescription();
            descriptionFrequencies[description] += 1;
            result.UpdateFechedSupportVerdicts(description, supportVerdicts);
            if (!markup.GetIsTheBest()) {
                continue;
            }
            if (skipElements && !markup.GetElement().empty() && !uniqueElements.insert(markup.GetElement()).second) {
                continue;
            }
            result.MutableImages().push_back(std::move(image));
            break;
        }
    }
    for (auto&& image : result.MutableImages()) {
        std::sort(image.MutableMarkUpList().begin(), image.MutableMarkUpList().end(), [&](const TCarDamage& left, const TCarDamage& right) {
            return descriptionFrequencies[left.GetDescription()] > descriptionFrequencies[right.GetDescription()];
        });
    }
    return result;
}

bool TImagesStorage::GetMarkUp(const ui64 imageId, TCommonImageData& markUp, NDrive::TEntitySession& session) const {
    auto record = GetRecord(imageId, session);
    if (!record) {
        return false;
    }
    markUp = std::move(*record);
    return true;
}

bool TImagesStorage::UpdateMarkUp(const ui64 imageId, const TVector<TCarDamage>& markUp, const TString& originator, bool replace, NDrive::TEntitySession& session) const {
    auto record = GetRecord(imageId, session);
    if (!record) {
        return false;
    }
    auto& elements = record->MutableMarkUpList();
    if (replace) {
        elements.clear();
    }
    elements.insert(elements.end(), markUp.begin(), markUp.end());
    return UpdateImage(*record, originator, session);
}

bool TImagesStorage::GetMetaData(const ui64 imageId, NJson::TJsonValue& metaData, NDrive::TEntitySession& session) const {
    auto record = GetRecord(imageId, session);
    if (!record) {
        return false;
    }
    metaData = std::move(record->MutableMetaData());
    return true;
}

bool TImagesStorage::UpdateMetaData(const ui64 imageId, const NJson::TJsonValue& metaData, const TString& originator, NDrive::TEntitySession& session) const {
    auto record = GetRecord(imageId, session);
    if (!record) {
        return false;
    }
    record->SetMetaData(metaData);
    return UpdateImage(*record, originator, session);
}

TMaybe<TImagesSnapshot::TImage> TImagesStorage::BuildSinglePhoto(
    const NJson::TJsonValue& jsonPhoto,
    const TString& objectId,
    NEntityTagsManager::EEntityType objectType,
    const TString& source,
    const TString& originator,
    bool generatePreviewPath,
    TMaybe<TString> userSessionId,
    const NJson::TJsonValue& extraMetaData,
    NDrive::TEntitySession& session
) const {
    if (!jsonPhoto.Has("marker") || !jsonPhoto.Has("uuid") || !jsonPhoto.Has("md5")) {
        session.SetErrorInfo("parse_image", "incorrect image data", EDriveSessionResult::DataCorrupted);
        return Nothing();
    }
    const TString& marker = jsonPhoto["marker"].GetString();

    TImagesSnapshot::TImage image;
    image.ExternalId = jsonPhoto["uuid"].GetString();
    image.Marker = marker;
    image.MD5 = jsonPhoto["md5"].GetString();
    if (userSessionId.Defined()) {
        image.Path = TFsPath(*userSessionId) / marker / CreateGuidAsString();
    } else {
        image.Path = TFsPath(objectId) / marker / CreateGuidAsString();
    }
    image.Origin = jsonPhoto["origin"].GetString();
    if (generatePreviewPath) {
        image.PreviewPath = TCommonImageData::BuildDefaultPreviewPath(image.Path);
    }

    TCommonImageData record;
    record.SetMarker(image.Marker);
    record.SetPath(image.Path);
    record.SetPreviewPath(image.PreviewPath);
    record.SetSource(source);
    record.SetObjectType(objectType);
    record.SetObjectId(objectId);
    record.SetOrigin(image.Origin);
    if (userSessionId.Defined()) {
        record.SetSessionId(*userSessionId);
    }
    if (jsonPhoto.Has("meta_data")) {
        record.SetMetaData(jsonPhoto["meta_data"]);
    }
    if (extraMetaData.IsDefined()) {
        NJson::MergeJson(extraMetaData, record.MutableMetaData().SetType(NJson::JSON_MAP));
    }

    auto imageId = SaveImage(record, originator, session);
    if (!imageId) {
        return Nothing();
    }
    image.ImageId = imageId.GetRef();
    return image;
}

TImagesStorage::TOptionalRecord TImagesStorage::GetRecord(const ui64 imageId, NDrive::TEntitySession& session) const {
    auto records = GetRecords({ imageId }, session);
    if (!records) {
        return {};
    }
    if (records->empty()) {
        session.SetErrorInfo("ImagesStorage::GetRecord", TStringBuilder() << "no record with id " << imageId);
        return {};
    }
    return records->front();
}

TImagesStorage::TOptionalRecords TImagesStorage::GetRecords(TConstArrayRef<ui64> imageIds, NDrive::TEntitySession& session) const {
    if (!imageIds) {
        return TRecords{};
    }
    auto queryOptions = TQueryOptions().SetGenericCondition("image_id", MakeSet<ui64>(imageIds));
    return Fetch(session, queryOptions);
}

bool TImagesStorage::GetStartFailIsProblem() const {
    return false;
}

bool TImagesStorage::Refresh() {
    auto session = BuildTx<NSQL::ReadOnly>();
    if (!LastEventId) {
        LastEventId = HistoryManager.GetMaxEventIdOrThrow(session);
    }

    auto since = LastEventId ? *LastEventId + 1 : 0;
    auto optionalEvents = HistoryManager.GetEvents(since, session, 1000);
    if (!optionalEvents) {
        ERROR_LOG << GetName() << ": cannot GetEventsSince " << since << ": " << session.GetStringReport() << Endl;
        return false;
    }
    for (auto&& ev : *optionalEvents) {
        LastEventId = std::max(LastEventId.GetOrElse(0), ev.GetHistoryEventId());
        if (ev.GetMarkUpList().empty()) {
            DEBUG_LOG << GetName() << ": skip photo with empty markup list: " << ev.GetHistoryEventId() << Endl;
            continue;
        }

        bool erased = ValidatedImagesCache.erase(ev.GetObjectId());
        if (erased) {
            CacheInvalidated.Signal(1);
            INFO_LOG << GetName() << ": invalidate " << ev.GetObjectId() << Endl;
        }
    }
    return true;
}

TMaybe<ui64> TImagesStorage::UpdateImpl(const TRecord& data, const TString& orginator, EObjectHistoryAction action, NDrive::TEntitySession& session) const {
    NStorage::TObjectRecordsSet<TRecord> affected;
    if (action == EObjectHistoryAction::Add) {
        HistoryManager.GetDatabase().GetTable(TRecord::GetTableName())->AddRow(data.SerializeToTableRecord(), session.GetTransaction(), "", &affected);
        if (affected.size() != 1) {
            return Nothing();
        }
    } else if (action == EObjectHistoryAction::UpdateData) {
        NStorage::TRecordBuilder condition("image_id", data.GetImageId());
        // Check version
        HistoryManager.GetDatabase().GetTable(TRecord::GetTableName())->UpdateRow(condition, data.SerializeToTableRecord(), session.GetTransaction(), &affected);
        if (affected.size() != 1) {
            return Nothing();
        }
    } else {
        return Nothing();
    }
    if (!HistoryManager.AddHistory(affected.front(), orginator, action, session)) {
        return Nothing();
    }
    return affected.front().GetImageId();
}
