#pragma once

#include "image.h"

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/manager.h>
#include <drive/backend/device_snapshot/image.h>

#include <drive/library/cpp/threading/concurrent_cache.h>

class TDBTag;

class TImageHistoryManager : public TDatabaseHistoryManager<TCommonImageData> {
private:
    using TBase = TDatabaseHistoryManager<TCommonImageData>;

public:
    TImageHistoryManager(const IHistoryContext& context)
        : TBase(context, TCommonImageData::GetHistoryTableName())
        , HistoryContext(context)
    {
    }

    const IHistoryContext& GetContext() const {
        return HistoryContext;
    }

private:
    const IHistoryContext& HistoryContext;
};

class TImagesStorage
    : public TBaseEntityManager<TCommonImageData>
    , public IAutoActualization
{
private:
    using TBase = TBaseEntityManager<TCommonImageData>;

public:
    using TRecord = TCommonImageData;
    using TOptionalRecord = TMaybe<TRecord>;
    using TRecords = TVector<TRecord>;
    using TOptionalRecords = TMaybe<TRecords>;
    using THistoryManager = TImageHistoryManager;

public:
    TImagesStorage(const IHistoryContext& context);
    ~TImagesStorage();

    NDrive::IObjectSnapshot::TPtr BuildSnapshot(
        const NJson::TJsonValue& jsonValue,
        const TString& objectId,
        NEntityTagsManager::EEntityType objectType,
        const TString& source,
        const TString& originator,
        bool generatePreviewPath,
        NDrive::TEntitySession& session,
        TMaybe<TString> userSessionId,
        const NJson::TJsonValue& extraMetaData = {}
    ) const;

    NDrive::IObjectSnapshot::TPtr BuildSnapshot(
        const NJson::TJsonValue& jsonValue,
        const TDBTag& tag,
        NEntityTagsManager::EEntityType objectType,
        const TString& source,
        const TString& originator,
        bool generatePreviewPath,
        NDrive::TEntitySession& session,
        TMaybe<TString> userSessionId = Nothing(),
        const NJson::TJsonValue& extraMetaData = {}
    ) const;

    TString GetTableName() const override {
        return "drive_images";
    }

    TOptionalRecords Get(TConstArrayRef<TString> objectIds, TRange<TInstant> timestampRange, NDrive::TEntitySession& session) const;
    TOptionalRecords Get(const TString& objectId, TRange<TInstant> timestampRange, NDrive::TEntitySession& session) const;

    TExpectedValidatedImages GetCachedValidatedImages(const TString& objectId, TInstant statementDeadline = TInstant::Zero(), TMaybe<TDuration> cacheLifetime = {}) const;
    TOptionalValidatedImages RestoreValidatedImages(const TString& objectId, NDrive::TEntitySession& session) const;

    TMaybe<ui64> SaveImage(const TRecord& data, const TString& orginator, NDrive::TEntitySession& session) const {
        return UpdateImpl(data, orginator, EObjectHistoryAction::Add, session);
    }

    bool UpdateImage(const TRecord& data, const TString& orginator, NDrive::TEntitySession& session) const {
        return UpdateImpl(data, orginator, EObjectHistoryAction::UpdateData, session).Defined();
    }

    TOptionalRecord GetRecord(const ui64 imageId, NDrive::TEntitySession& session) const;
    TOptionalRecords GetRecords(TConstArrayRef<ui64> imageIds, NDrive::TEntitySession& session) const;

    bool GetMarkUp(const ui64 imageId, TCommonImageData& markUp, NDrive::TEntitySession& session) const;
    bool UpdateMarkUp(const ui64 imageId, const TVector<TCarDamage>& markUp, const TString& originator, bool replace, NDrive::TEntitySession& session) const;

    bool GetMetaData(const ui64 imageId, NJson::TJsonValue& metaData, NDrive::TEntitySession& session) const;
    bool UpdateMetaData(const ui64 imageId, const NJson::TJsonValue& metaData, const TString& originator, NDrive::TEntitySession& session) const;

protected:
    TMaybe<TImagesSnapshot::TImage> BuildSinglePhoto(
        const NJson::TJsonValue& jsonPhoto,
        const TString& objectId,
        NEntityTagsManager::EEntityType objectType,
        const TString& source,
        const TString& originator,
        bool generatePreviewPath,
        TMaybe<TString> userSessionId,
        const NJson::TJsonValue& extraMetaData,
        NDrive::TEntitySession& session
    ) const;

    bool GetStartFailIsProblem() const override;
    bool Refresh() override;

    TMaybe<ui64> UpdateImpl(const TRecord& data, const TString& orginator, EObjectHistoryAction action, NDrive::TEntitySession& session) const;

private:
    THistoryManager HistoryManager;
    TDuration DefaultLifetime;

    TOptionalObjectEventId<TRecord> LastEventId;

    mutable NUtil::TConcurrentCache<TString, TValidatedImages> ValidatedImagesCache;
};
