#include "image.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/history_iterator/history_iterator.h>

#include <library/cpp/string_utils/base64/base64.h>

#include <rtline/library/json/parse.h>

#include <util/string/join.h>
#include <util/string/split.h>


TCommonImageData::TCommonImageData()
    : CreatedAt(TInstant::Now())
{
}

TCommonImageData::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    ImageId = GetFieldDecodeIndex("image_id", decoderBase);
    CreatedAt = GetFieldDecodeIndex("created_at", decoderBase);
    Path = GetFieldDecodeIndex("path", decoderBase);
    ObjectId = GetFieldDecodeIndex("object_id", decoderBase);
    ObjectType = GetFieldDecodeIndex("object_type", decoderBase);
    Source = GetFieldDecodeIndex("source", decoderBase);
    Meta = GetFieldDecodeIndex("meta", decoderBase);
}

TMap<TString, TString> TCommonImageData::GetBucketByImageSourceMapping() {
    TMap<TString, TString> mapping;
    const auto mappingJson = NDrive::GetServer().GetSettings().GetJsonValue("photos.bucket_by_image_source_mapping");
    if (!NJson::TryFromJson(mappingJson,  NJson::Dictionary(mapping))) {
        // NB. Default bucket will be used as a fallback always, therefore service tag sources haven't be handled separately
        // Current list of service tag sources:
        //   srv_cleaning, srv_driving_to, srv_evacuation, srv_fueling, srv_long_term, srv_none, srv_other, srv_pasting, srv_tech, srv_ya_auto
        mapping = {
            {"default", "carsharing-tags"},
            {"feedback", "carsharing-acceptances"},
        };
    }
    return mapping;
}

TString TCommonImageData::BuildDefaultPreviewPath(const TCommonImageData& image) {
    return BuildDefaultPreviewPath(image.GetPath());
}

TString TCommonImageData::BuildDefaultPreviewPath(const TString& path) {
    TVector<TString> pathParts = StringSplitter(path).Split('/');
    if (pathParts.back().Contains('.')) {
        pathParts.back().prepend("preview_");
    } else {
        pathParts.push_back("preview");
    }
    return JoinSeq("/", pathParts);
}

TString TCommonImageData::BuildHost(const THostByImageSourceMapping& hostBySourceMapping) const {
    auto host = hostBySourceMapping.Value(Source, hostBySourceMapping.Value("default", ""));
    if (host && !host.EndsWith('/')) {
        host.append('/');
    }
    return host;
}

TString TCommonImageData::BuildUrl(const THostByImageSourceMapping& hostBySourceMapping, const TString& path) const {
    auto host = BuildHost(hostBySourceMapping);

    if (host) {
        return host + path;
    }
    return path;
}

NJson::TJsonValue TCommonImageData::BuildReport(const TMap<TString, TString>& hostBySourceMapping) const {
    NJson::TJsonValue report;

    report["image_id"] = ImageId;
    report["created_at"] = CreatedAt.Seconds();

    report["marker"] = Marker;
    report["path"] = Path;
    report["preview_path"] = NJson::ToJson(NJson::Nullable(PreviewPath));

    auto needleHost = BuildHost(hostBySourceMapping);
    if (needleHost) {
        report["url"] = needleHost + Path;
        report["preview_url"] = (PreviewPath) ? NJson::TJsonValue(needleHost + PreviewPath) : NJson::TJsonValue(NJson::JSON_NULL);
    } else {
        report["url"] = NJson::JSON_NULL;
        report["preview_url"] = NJson::JSON_NULL;
    }

    report["object_id"] = ObjectId;
    report["object_type"] = ::ToString(ObjectType);

    report["user_session_id"] = SessionId;

    report["source"] = Source;
    report["origin"] = NJson::ToJson(NJson::Nullable(Origin));

    NJson::TJsonValue markUpJson(NJson::JSON_ARRAY);
    for (auto&& markUp : MarkUpList) {
        markUpJson.AppendValue(markUp.BuildReport());
    }
    report["mark_up_list"] = markUpJson;

    report["meta_data"] = MetaData;

    return report;
}

bool TCommonImageData::HasActiveMarkup() const {
    for (auto&& markup : MarkUpList) {
        if (!markup.IsDiscarded()) {
            return true;
        }
    }
    return false;
}

bool TCommonImageData::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, ImageId);
    READ_DECODER_VALUE_INSTANT(decoder, values, CreatedAt);
    READ_DECODER_VALUE(decoder, values, Path);
    READ_DECODER_VALUE_DEF(decoder, values, ObjectId, Default<TString>());
    READ_DECODER_VALUE_DEF(decoder, values, ObjectType, NEntityTagsManager::EEntityType::Car);
    READ_DECODER_VALUE_DEF(decoder, values, Source, "default");

    TString meta;
    READ_DECODER_VALUE_TEMP(decoder, values, meta, Meta);

    NDrive::NProto::TCommonImageMeta metaProto;
    TString decoded = Base64Decode(meta);
    if (!metaProto.ParseFromArray(decoded.Data(), decoded.Size())) {
        return false;
    }

    Marker = metaProto.GetMarker();
    Origin = metaProto.GetOrigin();
    SessionId = metaProto.GetSessionId();

    for (auto&& markupProto : metaProto.GetMarkUp()) {
        TCarDamage damage;
        if (!damage.ParseFromProto(markupProto)) {
            return false;
        }
        MarkUpList.push_back(std::move(damage));
    }

    if (metaProto.HasMetaData() && !NJson::ReadJsonFastTree(metaProto.GetMetaData(), &MetaData)) {
        return false;
    }

    if (metaProto.HasCustomPreviewPath()) {
        PreviewPath = metaProto.GetCustomPreviewPath();
    } else if (metaProto.HasHasPreview() && metaProto.GetHasPreview()) {
        PreviewPath = BuildDefaultPreviewPath(Path);
    }

    return true;
}

NStorage::TTableRecord TCommonImageData::SerializeToTableRecord() const {
    const auto& settings = NDrive::GetServer().GetSettings();
    NStorage::TTableRecord record;
    if (ImageId) {
        record.Set("image_id", ImageId);
    }
    auto serializeActiveMarkup = settings.GetValue<bool>("images.serialize_active_markup").GetOrElse(true);
    if (serializeActiveMarkup) {
        record.Set("active_markup", HasActiveMarkup());
    }
    auto serializeSessionId = settings.GetValue<bool>("images.serialize_session_id").GetOrElse(true);
    if (serializeSessionId && SessionId) {
        record.Set("session_id", SessionId);
    }
    record.Set("path", Path);
    record.Set("created_at", CreatedAt.Seconds());
    record.Set("object_id", ObjectId);
    record.Set("object_type", ObjectType);
    record.Set("source", Source);

    {
        NDrive::NProto::TCommonImageMeta metaProto;

        metaProto.SetMarker(Marker);

        if (Origin) {
            metaProto.SetOrigin(Origin);
        }

        if (SessionId) {
            metaProto.SetSessionId(SessionId);
        }

        for (auto&& markUp : MarkUpList) {
            auto markupProto = metaProto.AddMarkUp();
            markUp.SerializeToProto(*markupProto);
        }

        if (MetaData.IsDefined()) {
            metaProto.SetMetaData(MetaData.GetStringRobust());
        }

        if (PreviewPath) {
            metaProto.SetHasPreview(true);
            if (PreviewPath != BuildDefaultPreviewPath(Path)) {
                metaProto.SetCustomPreviewPath(PreviewPath);
            }
        }

        record.Set("meta", Base64Encode(metaProto.SerializeAsString()));
    }

    return record;
}

namespace {
    struct TPublicImageInfo {
        ui64 ImageId = 0;
        TString Path;
        TString PreviewPath;
        TString Source;
        TCarDamage Markup;
        TVector<TCarDamage> Markups;

        i64 Priority = 0;
        TInstant CreatedAt;
        TString SessionId;
    };
    using TPublicImages = TVector<TPublicImageInfo>;

    struct TPublicGroup {
        TString Name;
        TPublicImages Images;

        i64 Priority = 0;
    };
}

bool TValidatedImages::TReportContext::NeedToReport(const TCarDamage& markup) const {
    return SupportAllowedVerdicts.empty()
        || AnyOf(SupportVerdicts.Value(markup.GetDescription(), TSet<TString>()), [&allowed = SupportAllowedVerdicts] (const TString& item) { return allowed.contains(item); });
}

void TValidatedImages::UpdateFechedSupportVerdicts(const TString& description, const TSet<TString>& verdicts) {
    if (!verdicts.empty()) {
        ReportContext.MutableSupportVerdicts()[description].insert(verdicts.begin(), verdicts.end());
    }
}

NJson::TJsonValue TValidatedImages::GetReport(ELocalization locale, const NDrive::IServer& server, const TMap<TString, TString>& hostBySourceMapping) const {
    const auto& settings = server.GetSettings();
    const auto localization = server.GetLocalization();
    TMap<TString, TString> elementToGroup;
    auto elementToGroupString = settings.GetValue<TString>("car.damage.element_to_group");
    if (elementToGroupString) {
        auto value = NJson::ToJson(NJson::JsonString(*elementToGroupString));
        bool parsed = NJson::TryFromJson(value, NJson::Dictionary(elementToGroup));
        if (!parsed) {
            ALERT_LOG << "cannot parse car.damage.element_to_group from " << value.GetStringRobust() << Endl;
        }
    }

    TMap<TString, TString> elementIcons;
    auto elementIconsString = settings.GetValue<TString>("car.damage.element_icons");
    if (elementIconsString) {
        auto value = NJson::ToJson(NJson::JsonString(*elementIconsString));
        bool parsed = NJson::TryFromJson(value, NJson::Dictionary(elementIcons));
        if (!parsed) {
            ALERT_LOG << "cannot parse car.damage.element_icons from " << value.GetStringRobust() << Endl;
        }
    }

    const bool skipDups = ReportContext.GetSkipIdenticalDupsDef(settings.GetValue<bool>("car.damage.skip_dups").GetOrElse(true));

    TMap<TString, TPublicImages> grps;
    TSet<ui64> imageIds;
    TSet<TString> uniqueElements;
    TSet<TString> uniqueMarkups;
    for (auto&& image : Images) {
        for (auto&& markup : image.GetMarkUpList()) {
            if (!markup.GetIsTheBest() || markup.GetDiscarded()) {
                continue;
            }
            if (skipDups && !imageIds.insert(image.GetImageId()).second) {
                continue;
            }
            if (ReportContext.IsSkipElementDups() && !uniqueElements.insert(markup.GetElement()).second) {
                continue;
            }
            if (!ReportContext.NeedToReport(markup) || !uniqueMarkups.insert(markup.GetDescription()).second) {
                continue;
            }

            auto group = TString("default");
            auto gp = elementToGroup.find(markup.GetElement());
            if (gp != elementToGroup.end()) {
                group = gp->second;
            }

            TPublicImageInfo info;
            info.ImageId = image.GetImageId();
            info.Path = image.GetPath();
            info.PreviewPath = image.GetPreviewPath();
            info.Source = image.GetSource();
            info.Markup = markup;
            if (ReportContext.GetReportValidatedDetails()) {
                info.CreatedAt = image.GetCreatedAt();
                info.SessionId = image.GetSessionId();
            }
            if (ReportContext.IsReportMarkups()) {
                TSet<TString> uniqueDescriptions;
                auto& markups = image.GetMarkUpList();
                CopyIf(markups.begin(), markups.end(), std::back_inserter(info.Markups), [&uniqueDescriptions](const auto& item) {
                    return item.GetIsTheBest()
                        && !item.GetDiscarded()
                        && uniqueDescriptions.insert(item.GetDescription()).second;
                });
            }
            info.Priority = settings.GetValue<i64>(TStringBuilder() << "car.damage.element." << markup.GetElement() << ".priority").GetOrElse(0);
            grps[group].push_back(std::move(info));
        }
    }

    TVector<TPublicGroup> groups;
    for (auto&& [name, images] : grps) {
        TPublicGroup group;
        group.Name = name;
        group.Images = std::move(images);
        group.Priority = settings.GetValue<i64>(TStringBuilder() << "car.damage.group." << group.Name << ".priority").GetOrElse(0);
        groups.push_back(std::move(group));
    }
    for (auto&& group : groups) {
        std::stable_sort(group.Images.begin(), group.Images.end(), [](const TPublicImageInfo& left, const TPublicImageInfo& right) {
            return left.Priority > right.Priority;
        });
    }
    {
        std::stable_sort(groups.begin(), groups.end(), [](const TPublicGroup& left, const TPublicGroup& right) {
            return left.Priority > right.Priority;
        });
    }

    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (auto&& group : groups) {
        auto titleKey = TStringBuilder() << "car.damage.group." << group.Name << ".title";
        NJson::TJsonValue grp;
        grp["group"] = group.Name;
        grp["title"] = localization ? localization->GetLocalString(locale, titleKey, titleKey) : group.Name;
        NJson::TJsonValue& elements = grp.InsertValue("elements", NJson::JSON_ARRAY);
        for (auto&& info : group.Images) {
            auto elementIcon = elementIcons[info.Markup.GetElement()];
            auto elementTitleKey = TStringBuilder() << "car.damage.element." << info.Markup.GetElement() << ".title";
            auto host = hostBySourceMapping.Value(info.Source, hostBySourceMapping.Value("default", ""));
            if (host && !host.EndsWith('/')) {
                host.append('/');
            }
            NJson::TJsonValue element;
            element["image_id"] = info.ImageId;
            element["id"] = info.Markup.GetDescription();
            element["element"] = info.Markup.GetElement();
            element["title"] = localization ? localization->GetLocalString(locale, elementTitleKey, elementTitleKey) : info.Markup.GetElement();
            if (ReportContext.GetReportValidatedDetails()) {
                NJson::TJsonValue jDetails;
                auto elementLevelKey = TStringBuilder() << "car.damage.level." << info.Markup.GetLevel() << ".title";
                jDetails["level"] = localization ? localization->GetLocalString(locale, elementLevelKey, elementLevelKey) : elementLevelKey;
                jDetails["session_id"] = info.SessionId;
                jDetails["created_at"] = info.CreatedAt.Seconds();
                if (const auto detailsIt = ImagesDetails.find(info.SessionId); detailsIt != ImagesDetails.end()) {
                    const auto& details = detailsIt->second;
                    jDetails["user_id"] = details.UserId;
                    jDetails["first_name"] = details.UserFirstName;
                    jDetails["last_name"] = details.UserLastName;
                }
                element["details"] = jDetails;
            }
            if (host) {
                element["url"] = host + info.Path;
                element["preview_url"] = info.PreviewPath ? NJson::TJsonValue(host + info.PreviewPath) : NJson::JSON_NULL;
            } else {
                element["url"] = info.Path;
                element["preview_url"] = info.PreviewPath;
            }
            if (elementIcon) {
                element["icon"] = std::move(elementIcon);
            }
            if (ReportContext.IsReportMarkups()) {
                auto& markups = element.InsertValue("markups", NJson::TJsonArray());
                for (auto&& markup : info.Markups) {
                    auto markupReport = markup.BuildReport();
                    const TString titleKey = TStringBuilder() << "car.damage.element." << markup.GetElement() << ".title";
                    markupReport["title"] = localization ? localization->GetLocalString(locale, titleKey, titleKey) : markup.GetElement();
                    const TString levelKey = TStringBuilder() << "car.damage.level." << markup.GetLevel() << ".title";
                    markupReport["level"] = localization ? localization->GetLocalString(locale, levelKey, levelKey) : levelKey;
                    markups.AppendValue(std::move(markupReport));
                }
            }
            elements.AppendValue(std::move(element));
        }
        result.AppendValue(std::move(grp));
    }
    return result;
}

void TValidatedImages::FetchImagesDetails(
    NDrive::TEntitySession& tx
) {
    ImagesDetails.clear();
    auto server = NDrive::GetServer().GetAsPtrSafe<NDrive::IServer>();
    Y_ENSURE(server);
    THistoryRidesContext ridesContext(*server);
    TVector<TString> sessionIds;
    for (auto&& image : Images) {
        sessionIds.push_back(image.GetSessionId());
    }
    auto driveApi = server->GetDriveAPI();
    Y_ENSURE(driveApi);

    auto ydbTx = driveApi->BuildYdbTx<NSQL::ReadOnly>("fetch_images_details", server);
    Y_ENSURE(ridesContext.InitializeSessions(sessionIds, tx, ydbTx));
    auto ridesIterator = ridesContext.GetIterator();

    THistoryRideObject sessionInfo;
    TSet<TString> userIds;
    while (ridesIterator.GetAndNext(sessionInfo)) {
        auto offer = sessionInfo.GetOffer();
        Y_ENSURE(offer);
        ImagesDetails[sessionInfo.GetSessionId()] = { offer->GetUserId(), "", "" };
        userIds.insert(offer->GetUserId());
    }

    if (!userIds.empty()) {
        auto usersFetchData = driveApi->GetUsersData()->FetchInfo(userIds, tx);
        for (auto& [_, details]: ImagesDetails) {
            const TDriveUserData* userData = usersFetchData.GetResultPtr(details.UserId);
            if (userData) {
                details.UserFirstName = userData->GetFirstName();
                details.UserLastName = userData->GetLastName();
            }
        }
    }
}
