#pragma once

#include "damage.h"

#include <drive/backend/database/history/manager.h>
#include <drive/backend/roles/permissions.h>
#include <drive/backend/tags/abstract.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/types/accessor.h>

using THostByImageSourceMapping = TMap<TString, TString>;

class TCommonImageData {
    R_READONLY(ui64, ImageId, 0);
    R_READONLY(TInstant, CreatedAt);

    R_FIELD(TString, Marker);  // basically it's a part of path; can be used to group images related to incident or so

    R_FIELD(TString, Path);
    R_FIELD(TString, PreviewPath);

    R_FIELD(TString, ObjectId);
    R_FIELD(NEntityTagsManager::EEntityType, ObjectType);

    R_FIELD(TString, SessionId);

    R_FIELD(TString, Origin);  // some origin keyword
    R_FIELD(TString, Source, "default");  // basically grouping tags of a holder tag

    R_FIELD(TVector<TCarDamage>, MarkUpList);

    R_FIELD(NJson::TJsonValue, MetaData, NJson::JSON_NULL);


public:
    TCommonImageData();

    static TString GetTableName() {
        return "drive_images";
    }

    static TString GetHistoryTableName() {
        return GetTableName() + "_history";
    }

    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, ImageId, -1);
        R_FIELD(i32, CreatedAt, -1);
        R_FIELD(i32, Path, -1);
        R_FIELD(i32, ObjectId, -1);
        R_FIELD(i32, ObjectType, -1);
        R_FIELD(i32, Source, -1);
        R_FIELD(i32, Meta, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);
    };

    static TMap<TString, TString> GetBucketByImageSourceMapping();

    static TString BuildDefaultPreviewPath(const TCommonImageData& image);
    static TString BuildDefaultPreviewPath(const TString& path);

    template <class TMDSClient>
    static THostByImageSourceMapping GetHostByImageSourceMapping(const TMDSClient& mdsClient) {
        THostByImageSourceMapping result;
        for (auto&& [origin, bucket]: GetBucketByImageSourceMapping()) {
            result.emplace(origin, mdsClient.GetTmpFilePath(bucket, ""));
        }
        return result;
    }

    template <class TMDSClient>
    static TString BuildUrl(const TString& imagePath, const TMDSClient& mdsClient, const TString& source = "default") {
        const auto& bucketByImageSourceMapping = GetBucketByImageSourceMapping();
        auto bucketIt = bucketByImageSourceMapping.find(source);
        if (bucketIt != bucketByImageSourceMapping.end()) {
            return mdsClient.GetTmpFilePath(bucketIt->second, imagePath);
        }
        return {};
    }

    TString BuildHost(const THostByImageSourceMapping& hostBySourceMapping) const;
    TString BuildUrl(const THostByImageSourceMapping& hostBySourceMapping, const TString& path) const;
    NJson::TJsonValue BuildReport(const THostByImageSourceMapping& hostBySourceMapping = {}) const;
    bool HasActiveMarkup() const;

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext);
    NStorage::TTableRecord SerializeToTableRecord() const;
};

class TValidatedImages {
private:
    using TSuportVerdicts = TMap<TString, TSet<TString>>;

    class TReportContext {
        R_FIELD(TSuportVerdicts, SupportVerdicts);
        R_FIELD(TSet<TString>, SupportAllowedVerdicts);
        R_FIELD(bool, ReportMarkups, false);
        R_FIELD(bool, SkipElementDups, true);
        R_OPTIONAL(bool, SkipIdenticalDups);
        R_FIELD(bool, ReportValidatedDetails, false);

    public:
        bool NeedToReport(const TCarDamage& markup) const;
    };

    struct TSessionDetails {
        TString UserId;
        TString UserFirstName;
        TString UserLastName;
    };

    using TMapImagesDetails = TMap<TString, TSessionDetails>;

public:
    R_FIELD(TString, ObjectId);
    R_FIELD(TInstant, Timestamp);
    R_FIELD(TVector<TCommonImageData>, Images);
    R_FIELD(TReportContext, ReportContext);
    R_FIELD(TMapImagesDetails, ImagesDetails);

public:
    size_t Count() const {
        return Images.size();
    }

    void UpdateFechedSupportVerdicts(const TString& description, const TSet<TString>& verdicts);
    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server, const THostByImageSourceMapping& hostBySourceMapping) const;
    void FetchImagesDetails(NDrive::TEntitySession& tx);
};
using TOptionalValidatedImages = TMaybe<TValidatedImages>;
using TExpectedValidatedImages = TExpected<TValidatedImages, TCodedException>;
