#include "evacuation_ticket.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/common/scheme_adapters.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/library/json/merge.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/algorithm/type_traits.h>

namespace {
    using TContextPtr = TAtomicSharedPtr<NDrive::TEvacuationTicketIncidentContext>;

    bool ApplyDefault(NDrive::IBaseDefaultSchemeElement& item, TContextPtr ctx, const TString& fieldName = {}) {
        return ctx && item.SetDefaultValueView(ctx->GetData()[Coalesce(fieldName, item.GetFieldName())]);
    }

    template <typename T, typename... TArgs>
    T& AddWithDefault(TContextPtr ctx, NDrive::TScheme& scheme, TArgs&&... args) {
        T& item = scheme.Add<T>(args...);
        ApplyDefault(item, ctx);
        return item;
    }

    template <typename... TArgs>
    TFSVariable& AddVariableWithDefault(TContextPtr ctx, NDrive::TScheme& scheme, const TString& fieldName, TArgs&&... args) {
        auto& item = scheme.Add<TFSVariable>(fieldName, args...);
        ApplyDefault(item.MutableCondition(), ctx, fieldName);
        return item;
    }
}

namespace {
    struct TInspectionDepartment {
        NDrive::TScheme GetPublicScheme() const {
            NDrive::TScheme scheme;
            scheme.Add<TFSString>("evacuation_inspection_department_title", "Подразделение ГИБДД/МАДИ").SetDefault(Title);
            scheme.Add<TFSString>("evacuation_inspection_department_address", "Адрес ГИБДД/МАДИ").SetDefault(Address);
            scheme.Add<TFSString>("evacuation_inspection_department_phone", "Телефон ГИБДД/МАДИ").SetDefault(Phone);
            return scheme;
        }

        TString Id;
        TString Title;
        TString Address;
        TString Phone;
    };
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& data, TInspectionDepartment& instance) {
    return NJson::ParseField(data["id"], instance.Id, true) &&
           NJson::ParseField(data["title"], instance.Title, true) &&
           NJson::ParseField(data["address"], instance.Address) &&
           NJson::ParseField(data["phone"], instance.Phone);
}

namespace NDrive {
    TEvacuationTicketIncidentContext::TRegistrator TEvacuationTicketIncidentContext::Registrator;

    NDrive::TScheme TEvacuationTicketIncidentContext::GetScheme(const IServerBase& server, const TMaybe<EIncidentType>& /* incidentType */, TPtr existingContextPtr) {
        auto ctx = std::dynamic_pointer_cast<TEvacuationTicketIncidentContext>(existingContextPtr);

        NDrive::TScheme scheme;

        TString districtLocalizationPrefix = server.GetSettings().GetValueDef<TString>(EvacuationSettingPrefix + ".district_localization_prefix", "EEvacuationDistrict");
        AddWithDefault<TFSVariants>(ctx, scheme, "evacuation_district", "Город / Область")
            .SetCompoundVariants(TLocalizedSettingVariantsAdapter::Cast(EvacuationSettingPrefix + ".districts", districtLocalizationPrefix));

        AddWithDefault<TFSString>(ctx, scheme, "evacuation_violation_paragraph", "Статья задержания ТС");
        AddWithDefault<TFSNumeric>(ctx, scheme, "evacuation_instant", "Дата и время эвакуации").SetVisual(TFSNumeric::EVisualType::DateTime);
        AddWithDefault<TFSString>(ctx, scheme, "evacuation_address", "Адрес эвакуации");
        AddWithDefault<TFSString>(ctx, scheme, "evacuation_inspection_parking_address", "Адрес штрафстоянки");

        auto& inspectionDepartmentsScheme = AddVariableWithDefault(ctx, scheme, "evacuation_inspection_department_id", "Подразделение ГИБДД/МАДИ");
        {
            NJson::TJsonValue rawInspectionDepartments = server.GetSettings().GetJsonValue(EvacuationSettingPrefix + ".inspection_departments");
            TVector<TInspectionDepartment> inspectionDepartments;
            if (NJson::TryFromJson(rawInspectionDepartments, inspectionDepartments)) {
                for (auto&& department: inspectionDepartments) {
                    TFSVariable::TCompoundVariant value(department.Id, department.Title);
                    inspectionDepartmentsScheme.AddVariant(value, department.GetPublicScheme());
                }
            }
        }

        scheme.Add<TFSSeparator>("fields_to_suggest", "Автозаполняемые поля");

        return scheme;
    }

    void TEvacuationTicketIncidentContext::InitializeLocalizedVariants(const IServerBase& server, TDynamicContext& dynamicContext, const ELocalization locale) {
        TString districtLocalizationPrefix = server.GetSettings().GetValueDef<TString>(EvacuationSettingPrefix + ".district_localization_prefix", "EEvacuationDistrict");

        TVector<TFSVariants::TCompoundVariants> allLocalizedVariants = {
            TLocalizedSettingVariantsAdapter(EvacuationSettingPrefix + ".districts", districtLocalizationPrefix, locale)
        };

        for (auto&& localizedVariantsAdapter: allLocalizedVariants) {
            auto LocalizedVariants = static_cast<TFSVariants::TCompoundVariants>(localizedVariantsAdapter);
            for (auto&& variant: LocalizedVariants) {
                dynamicContext.emplace(variant.GetValue(), variant.GetText());
            }
        }
    }

    NJson::TJsonValue TEvacuationTicketIncidentContext::DoSerializeToJson() const {
        return Data;
    }

    bool TEvacuationTicketIncidentContext::DoDeserializeFromJson(const NJson::TJsonValue& data, TMessagesCollector& /* errors */) {
        Data = NJson::UnnestJson(data);  // Fix possible issues storing variable field results
        // Validate?
        return true;
    }
}
