#include "incident_ticket.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/common/scheme_adapters.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/merge.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/algorithm/type_traits.h>

namespace {
    using TContextPtr = TAtomicSharedPtr<NDrive::TIncidentTicketIncidentContext>;

    bool ApplyDefault(NDrive::IBaseDefaultSchemeElement& item, TContextPtr ctx, const TString& fieldName = {}) {
        return ctx && item.SetDefaultValueView(ctx->GetData()[Coalesce(fieldName, item.GetFieldName())]);
    }

    template <typename T, typename... TArgs>
    T& AddWithDefault(TContextPtr ctx, NDrive::TScheme& scheme, TArgs&&... args) {
        T& item = scheme.Add<T>(args...);
        ApplyDefault(item, ctx);
        return item;
    }

    template <typename... TArgs>
    TFSVariable& AddVariableWithDefault(TContextPtr ctx, NDrive::TScheme& scheme, const TString& fieldName, TArgs&&... args) {
        auto& item = scheme.Add<TFSVariable>(fieldName, args...);
        ApplyDefault(item.MutableCondition(), ctx, fieldName);
        return item;
    }

    void AddIncidentTimestamp(TContextPtr ctx, NDrive::TScheme& scheme, bool required = false) {
        auto& item = AddWithDefault<TFSNumeric>(ctx, scheme, "incident_instant", "Время инцидента");
        if (!item.HasDefault()) {
            item.SetDefault(Now().Seconds());
        }
        item.SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(required);
    }

    void AddIncidentDamageDegree(TContextPtr ctx, NDrive::TScheme& scheme, bool required = false) {
        auto& incidentDamageDegreeScheme = AddVariableWithDefault(ctx, scheme, "incident_damage_degree", "Имеются ли повреждения");
        incidentDamageDegreeScheme.MutableCondition().SetCompoundVariants(TLocalizedEnumVariantsAdapter<NDrive::EIncidentDamageDegree>::Cast()).SetRequired(required);
        {
            NDrive::TScheme someDamageDegreeScheme;
            AddWithDefault<TFSBoolean>(ctx, someDamageDegreeScheme, "incident_found_by_technicians", "Повреждения обнаружили техники");
            AddWithDefault<TFSString>(ctx, someDamageDegreeScheme, "incident_description", "Перечисли повреждения");

            incidentDamageDegreeScheme.AddVariant(TLocalizedVariantAdapter::FromEnum(NDrive::EIncidentDamageDegree::High), someDamageDegreeScheme);
            incidentDamageDegreeScheme.AddVariant(TLocalizedVariantAdapter::FromEnum(NDrive::EIncidentDamageDegree::Medium), someDamageDegreeScheme);
        }
    }

    void AddIncidentOriginator(TContextPtr ctx, NDrive::TScheme& scheme, bool required = false) {
        AddWithDefault<TFSVariants>(ctx, scheme, "incident_originator", "Виновник")
            .SetCompoundVariants(TLocalizedEnumVariantsAdapter<NDrive::EIncidentOriginator>::Cast())
            .SetRequired(required);
    }

    void AddIncidentProtocolPresence(TContextPtr ctx, NDrive::TScheme& scheme, bool required = false) {
        AddWithDefault<TFSVariants>(ctx, scheme, "incident_protocol_presence", "Есть ли справка с места происшествия")
            .SetCompoundVariants(TLocalizedEnumVariantsAdapter<NDrive::EIncidentProtocolPresence>::Cast())
            .SetRequired(required);
    }
}

namespace NDrive {
    TIncidentTicketIncidentContext::TRegistrator TIncidentTicketIncidentContext::Registrator;

    NDrive::TScheme TIncidentTicketIncidentContext::GetScheme(const IServerBase& server, const TMaybe<EIncidentType>& incidentType, TPtr existingContextPtr) {
        auto ctx = std::dynamic_pointer_cast<TIncidentTicketIncidentContext>(existingContextPtr);

        NDrive::TScheme scheme;

        auto& incidentTypeScheme = AddVariableWithDefault(ctx, scheme, "incident_type", "Тип инцидента");
        incidentTypeScheme.MutableCondition().SetRequired(true);

        TVector<EIncidentType> majorIncidentTypes = { EIncidentType::RoadAccident };
        TVector<EIncidentType> neutralIncidentTypes = { EIncidentType::AccountTransfer, EIncidentType::DriveTransfer, EIncidentType::DrunkDriver };
        TVector<EIncidentType> possibleDamageIncidentTypes = { EIncidentType::CarTheftDuringRide };
        TVector<EIncidentType> basicIncidentTypes = { EIncidentType::Incident };

        {
            if (!incidentType) {
                incidentTypeScheme.MutableCondition().SetCompoundVariants(TLocalizedEnumVariantsAdapter<EIncidentFormType>::Cast());
            } else {
                auto incidentTypeAdapter = TLocalizedVariantAdapter::FromEnum(*incidentType);
                incidentTypeScheme.MutableCondition().AddCompoundVariant(incidentTypeAdapter).SetDefault(incidentTypeAdapter.GetValue());

                const auto typeFilter = [&incidentType](const EIncidentType item){ return *incidentType != item; };
                EraseIf(majorIncidentTypes, typeFilter);
                EraseIf(neutralIncidentTypes, typeFilter);
                EraseIf(possibleDamageIncidentTypes, typeFilter);
                EraseIf(basicIncidentTypes, typeFilter);
            }
        }

        if (majorIncidentTypes) {
            NDrive::TScheme majorAccidentScheme;

            AddWithDefault<TFSString>(ctx, majorAccidentScheme, "incident_another_phone_number_source", "О ДТП сообщили с другого номера");

            auto& incidentInjuredPersonsPresenceScheme = AddVariableWithDefault(ctx, majorAccidentScheme, "incident_injured_persons_presence", "Есть ли пострадавшие");
            incidentInjuredPersonsPresenceScheme.MutableCondition().SetCompoundVariants(TLocalizedEnumVariantsAdapter<EIncidentInjuredPersonsPresence>::Cast());
            {
                NDrive::TScheme incidentInjuredPersonsScheme;
                AddWithDefault<TFSNumeric>(ctx, incidentInjuredPersonsScheme, "incident_injured_persons_count", "Количество пострадавших").SetMin(1);
                AddWithDefault<TFSString>(ctx, incidentInjuredPersonsScheme, "incident_injured_persons_state", "Состояние пострадавших");
                AddWithDefault<TFSVariants>(ctx, incidentInjuredPersonsScheme, "incident_injured_persons_location", "Кто именно пострадал").SetCompoundVariants(TLocalizedEnumVariantsAdapter<EIncidentInjuredPersonsLocation>::Cast()).SetMultiSelect(true);
                AddWithDefault<TFSString>(ctx, incidentInjuredPersonsScheme, "incident_injured_persons_hospital", "В какую больницу увезли пострадавших");

                incidentInjuredPersonsPresenceScheme.AddVariant(TLocalizedVariantAdapter::FromEnum(EIncidentInjuredPersonsPresence::Present), incidentInjuredPersonsScheme);
            }

            AddWithDefault<TFSVariants>(ctx, majorAccidentScheme, "incident_car_state", "Состояние авто").SetCompoundVariants(TLocalizedEnumVariantsAdapter<EIncidentCarState>::Cast());
            AddWithDefault<TFSVariants>(ctx, majorAccidentScheme, "incident_registration", "Оформление").SetCompoundVariants(TLocalizedEnumVariantsAdapter<EIncidentRegistration>::Cast()).SetRequired(true);
            AddWithDefault<TFSVariants>(ctx, majorAccidentScheme, "incident_police_request", "Вызов ГИБДД").SetCompoundVariants(TLocalizedEnumVariantsAdapter<EIncidentPoliceRequest>::Cast());

            AddWithDefault<TFSBoolean>(ctx, majorAccidentScheme, "incident_europrotocol_rules_familiar", "Про европротокол проинформирован");

            AddWithDefault<TFSVariants>(ctx, majorAccidentScheme, "incident_photo_sending_intent", "Фото повреждений / общего плана").SetCompoundVariants(TLocalizedEnumVariantsAdapter<EIncidentPhotoSendingIntent>::Cast());

            AddWithDefault<TFSBoolean>(ctx, majorAccidentScheme, "incident_leave_documents_as_asked", "Фото протокола, \"Объяснения\" и других документов пришлёт, оригиналы оставит в бардачке");
            AddWithDefault<TFSBoolean>(ctx, majorAccidentScheme, "incident_call_after_protocol_registration", "Перезвонит, когда всё будет оформлено");

            auto& incidentRelatedCarModelsScheme = AddWithDefault<TFSVariants>(ctx, majorAccidentScheme, "incident_related_car_models", "Марка и модель авто других участников ДТП").SetMultiSelect(true).SetEditable(true);
            {
                auto roadAccidentSpecificCarModelsJson = server.GetSettings().GetJsonValue(GenericIncidentSettingPrefix + ".road_accident_specific_car_models");
                TSet<TString> roadAccidentSpecificCarModels;
                if (NJson::TryFromJson(roadAccidentSpecificCarModelsJson, roadAccidentSpecificCarModels)) {
                    incidentRelatedCarModelsScheme.SetVariants(roadAccidentSpecificCarModels);
                }
            }

            AddWithDefault<TFSBoolean>(ctx, majorAccidentScheme, "incident_is_other_participant_carsharing", "Второй участник ДТП - каршеринг");
            AddWithDefault<TFSBoolean>(ctx, majorAccidentScheme, "incident_is_other_participant_motorcycle", "Среди участников ДТП есть мотоцикл");
            AddWithDefault<TFSBoolean>(ctx, majorAccidentScheme, "incident_are_other_participant_airbags_deployed", "У другого участника ДТП сработали подушки безопасности");
            AddWithDefault<TFSBoolean>(ctx, majorAccidentScheme, "incident_does_other_participant_get_off_the_road", "Другой участник ДТП съехал с дороги");

            AddIncidentTimestamp(ctx, majorAccidentScheme, true);
            AddIncidentDamageDegree(ctx, majorAccidentScheme, true);
            AddIncidentOriginator(ctx, majorAccidentScheme, true);
            AddIncidentProtocolPresence(ctx, majorAccidentScheme, true);

            for (auto&& majorIncidentType: majorIncidentTypes) {
                incidentTypeScheme.AddVariant(TLocalizedVariantAdapter::FromEnum(majorIncidentType), majorAccidentScheme);
            }
        }

        if (neutralIncidentTypes) {
            NDrive::TScheme neutralAccidentScheme;
            auto& incidentReporterScheme = AddVariableWithDefault(ctx, neutralAccidentScheme, "incident_reporter", "Сообщивший");
            incidentReporterScheme.MutableCondition().SetCompoundVariants(TLocalizedEnumVariantsAdapter<EIncidentReporter>::Cast());
            {
                NDrive::TScheme otherReporterScheme;
                AddWithDefault<TFSString>(ctx, otherReporterScheme, "incident_reporter_info", "Данные сообщившего");

                incidentReporterScheme.AddVariant(TLocalizedVariantAdapter::FromEnum(EIncidentReporter::Other), otherReporterScheme);
            }

            AddIncidentTimestamp(ctx, neutralAccidentScheme);
            AddIncidentDamageDegree(ctx, neutralAccidentScheme);
            AddIncidentOriginator(ctx, neutralAccidentScheme);
            AddIncidentProtocolPresence(ctx, neutralAccidentScheme);

            for (auto&& neutralIncidentType: neutralIncidentTypes) {
                incidentTypeScheme.AddVariant(TLocalizedVariantAdapter::FromEnum(neutralIncidentType), neutralAccidentScheme);
            }
        }

        if (possibleDamageIncidentTypes) {
            NDrive::TScheme possibleDamageIncidentScheme;

            AddIncidentTimestamp(ctx, possibleDamageIncidentScheme, true);
            AddIncidentDamageDegree(ctx, possibleDamageIncidentScheme, true);
            AddIncidentOriginator(ctx, possibleDamageIncidentScheme);
            AddIncidentProtocolPresence(ctx, possibleDamageIncidentScheme);

            for (auto&& possibleDamageIncidentType: possibleDamageIncidentTypes) {
                incidentTypeScheme.AddVariant(TLocalizedVariantAdapter::FromEnum(possibleDamageIncidentType), possibleDamageIncidentScheme);
            }
        }

        if (basicIncidentTypes) {
            NDrive::TScheme basicIncidentScheme;

            AddIncidentTimestamp(ctx, basicIncidentScheme, true);
            AddIncidentDamageDegree(ctx, basicIncidentScheme);
            AddIncidentOriginator(ctx, basicIncidentScheme);
            AddIncidentProtocolPresence(ctx, basicIncidentScheme);

            for (auto&& basicIncidentType: basicIncidentTypes) {
                incidentTypeScheme.AddVariant(TLocalizedVariantAdapter::FromEnum(basicIncidentType), basicIncidentScheme);
            }
        }

        {
            NDrive::TScheme defaultIncidentScheme;

            AddIncidentTimestamp(ctx, defaultIncidentScheme);
            AddIncidentDamageDegree(ctx, defaultIncidentScheme);
            AddIncidentOriginator(ctx, defaultIncidentScheme);
            AddIncidentProtocolPresence(ctx, defaultIncidentScheme);

            incidentTypeScheme.SetDefaultValue(defaultIncidentScheme);
        }

        AddWithDefault<TFSText>(ctx, scheme, "incident_timeline", "Хронология событий").SetRequired(true);

        AddWithDefault<TFSString>(ctx, scheme, "incident_address", "Адрес инцидента");

        AddWithDefault<TFSString>(ctx, scheme, "comment", "Комментарий");

        AddWithDefault<TFSArray>(ctx, scheme, "photos", "Фото повреждений или другие данные").SetElement<TFSString>().SetVisual(TFSString::EVisualType::StartrekAttachment);

        scheme.Add<TFSSeparator>("fields_to_suggest", "Автозаполняемые поля");

        AddWithDefault<TFSString>(ctx, scheme, "incident_car_address", "Адрес автомобиля");

        return scheme;
    }

    void TIncidentTicketIncidentContext::InitializeLocalizedVariants(const IServerBase& /* server */, TDynamicContext& dynamicContext, const ELocalization locale) {
        TVector<TFSVariants::TCompoundVariants> allLocalizedVariants = {
            TLocalizedEnumVariantsAdapter<EIncidentFormType>(locale),
            TLocalizedEnumVariantsAdapter<EIncidentInjuredPersonsPresence>(locale),
            TLocalizedEnumVariantsAdapter<EIncidentInjuredPersonsLocation>(locale),
            TLocalizedEnumVariantsAdapter<EIncidentCarState>(locale),
            TLocalizedEnumVariantsAdapter<EIncidentRegistration>(locale),
            TLocalizedEnumVariantsAdapter<EIncidentPoliceRequest>(locale),
            TLocalizedEnumVariantsAdapter<EIncidentPhotoSendingIntent>(locale),
            TLocalizedEnumVariantsAdapter<EIncidentReporter>(locale),
            TLocalizedEnumVariantsAdapter<EIncidentOriginator>(locale),
            TLocalizedEnumVariantsAdapter<EIncidentDamageDegree>(locale),
            TLocalizedEnumVariantsAdapter<EIncidentProtocolPresence>(locale),
        };
        for (auto&& enumLocalizedVariantsAdapter: allLocalizedVariants) {
            auto enumLocalizedVariants = static_cast<TFSVariants::TCompoundVariants>(enumLocalizedVariantsAdapter);
            for (auto&& variant: enumLocalizedVariants) {
                dynamicContext.emplace(variant.GetValue(), variant.GetText());
            }
        }
    }

    TMaybe<EIncidentType> TIncidentTicketIncidentContext::GetIncidentType() const {
        EIncidentType dataIncidentType;
        if (!NJson::ParseField(Data["incident_type"], NJson::Stringify(dataIncidentType), /* required = */ true)) {
            return {};
        }
        return dataIncidentType;
    }

    NJson::TJsonValue TIncidentTicketIncidentContext::DoSerializeToJson() const {
        return Data;
    }

    bool TIncidentTicketIncidentContext::DoDeserializeFromJson(const NJson::TJsonValue& data, TMessagesCollector& /* errors */) {
        Data = NJson::UnnestJson(data);  // Fix possible issues storing variable field results
        // Validate?
        return true;
    }
}
