#include "database.h"

namespace NDrive {
    NStorage::TTableRecord TDefaultIncidentsStorageUniquePolicy::GetUpdateCondition(const TIncidentData& entry) const {
        return NStorage::TRecordBuilder("incident_id", entry.GetIncidentId())
                                       ("incident_serial_id", entry.GetIncidentSerialId());
    }

    TIncidentsHistoryManager::TIncidentsHistoryManager(const IHistoryContext& context)
        : TBase(context, TRecord::GetHistoryTableName())
        , HistoryContext(context)
    {
    }

    TIncidentsStorage::TIncidentsStorage(const IHistoryContext& context, const THistoryConfig& hConfig)
        : TBase("drive_incidents", MakeAtomicShared<THistoryReader>(context, TRecord::GetHistoryTableName(), hConfig))
        , TDatabaseSessionConstructor(context.GetDatabase())
        , HistoryManager(context)
    {
    }

    void TIncidentsStorage::Start() {
        Y_ENSURE_BT(HistoryReader->Start());
        Y_ENSURE_BT(RebuildCache());
    }

    void TIncidentsStorage::Stop() {
        Y_ENSURE_BT(HistoryReader->Stop());
    }

    TString TIncidentsStorage::GetTableName() const {
        return TRecord::GetTableName();
    }

    TIncidentsStorage::TOptionalRecord TIncidentsStorage::Add(const TIncidentData& data, const TString& originatorId, NDrive::TEntitySession& session) const {
        NStorage::TObjectRecordsSet<TRecord> affected;
        NStorage::ITableAccessor::TPtr table = GetDatabase().GetTable(GetTableName());
        auto result = table->AddRow(data.SerializeToTableRecord(), session.GetTransaction(), "", &affected);
        if (!result || !result->IsSucceed() || affected.size() != 1) {
            session.SetErrorInfo(GetTableName(), "insert", EDriveSessionResult::TransactionProblem);
            return {};
        }
        if (!HistoryManager.AddHistory(affected.front(), originatorId, EObjectHistoryAction::Add, session)) {
            return {};
        }
        return affected.front();
    }

    TIncidentsStorage::TOptionalRecord TIncidentsStorage::Update(const TIncidentData& data, const TString& originatorId, NDrive::TEntitySession& session, const IUniquePolicy& uniquePolicy) const {
        NStorage::TObjectRecordsSet<TRecord> affected;
        NStorage::ITableAccessor::TPtr table = GetDatabase().GetTable(GetTableName());
        auto result = table->UpdateRow(uniquePolicy.GetUpdateCondition(data), data.SerializeToTableRecordForUpdate(), session.GetTransaction(), &affected);
        if (!result || !result->IsSucceed() || affected.size() != 1) {
            session.SetErrorInfo(GetTableName(), "upsert", EDriveSessionResult::TransactionProblem);
            return {};
        }
        if (!HistoryManager.AddHistory(affected.front(), originatorId, EObjectHistoryAction::UpdateData, session)) {
            return {};
        }
        return affected.front();
    }

    TIncidentsStorage::TOptionalRecord TIncidentsStorage::ApplyTransition(const TIncidentData& data, const TString& originatorId, NDrive::TEntitySession& session) const {
        return Update(data, originatorId, session, TDefaultUniquePolicy());
    }

    TIncidentsStorage::TOptionalRecordState TIncidentsStorage::GetPreviousState(const TString& incidentId, NDrive::TEntitySession& session) const {
        if (!incidentId) {
            return {};
        }

        THistoryManager::TQueryOptions options;
        options.AddGenericCondition("incident_id", incidentId);

        auto events = HistoryManager.GetEvents({}, {}, session, options);
        if (!events || events->size() < 2) {
            return {};
        }

        if (events->back().GetHistoryAction() == EObjectHistoryAction::Remove) {
            return {};  // provide state for existing entries only
        }

        return events->at(events->size() - 2);
    }

    bool TIncidentsStorage::DoRebuildCacheUnsafe() const {
        const auto& database = HistoryReader->GetDatabase();
        auto session = NDrive::TEntitySession(database.CreateTransaction(true));
        auto table = database.GetTable(TRecord::GetTableName());

        NStorage::TObjectRecordsSet<TRecord, IHistoryContext> records;
        auto reqResult = table->GetRows("", records, session.GetTransaction());
        if (!reqResult->IsSucceed()) {
            ERROR_LOG << session.GetTransaction()->GetErrors().GetStringReport() << Endl;
            return false;
        }

        for (auto&& record : records) {
            Objects.emplace(record.GetIncidentId(), record);
        }
        return true;
    }

    bool TIncidentsStorage::DoAcceptHistoryEventUnsafe(const TAtomicSharedPtr<TObjectEvent<TRecord>>& dbEvent, const bool isNewEvent) {
        if (isNewEvent) {
            if (dbEvent->GetHistoryAction() == EObjectHistoryAction::Remove) {
                Objects.erase(dbEvent->GetIncidentId());
            } else {
                Objects[dbEvent->GetIncidentId()] = *dbEvent;
            }
        }
        return true;
    }

    TStringBuf TIncidentsStorage::GetEventObjectId(const TObjectEvent<TRecord>& ev) const {
        return ev.GetIncidentId();
    }
}
