#pragma once

#include "incident.h"

#include <drive/backend/database/history/cache.h>
#include <drive/backend/database/history/manager.h>

// NB. Storage is planned to be reworked avoiding history cache usage

namespace NDrive {
    // unique policies

    class IIncidentsStorageUniquePolicy {
    public:
        virtual ~IIncidentsStorageUniquePolicy() = default;
        virtual NStorage::TTableRecord GetUpdateCondition(const TIncidentData& entry) const = 0;
    };

    class TDefaultIncidentsStorageUniquePolicy: public IIncidentsStorageUniquePolicy {
    public:
        virtual NStorage::TTableRecord GetUpdateCondition(const TIncidentData& entry) const override;
    };

    // history manager

    class TIncidentsHistoryManager: public TDatabaseHistoryManager<TIncidentData> {
        using TBase = TDatabaseHistoryManager<TIncidentData>;
        using TRecord = TIncidentData;

    public:
        explicit TIncidentsHistoryManager(const IHistoryContext& context);

        const IHistoryContext& GetContext() const {
            return HistoryContext;
        }

    private:
        const IHistoryContext& HistoryContext;
    };

    // db accessors

    class TIncidentsStorage
        : public TAutoActualizingSnapshot<TIncidentData, TString>
        , public TDatabaseSessionConstructor
    {
    private:
        using TBase = TAutoActualizingSnapshot<TIncidentData, TString>;

    public:
        using TRecord = TIncidentData;
        using TOptionalRecord = TMaybe<TRecord>;
        using TOptionalRecordState = TMaybe<TObjectEvent<TIncidentData>>;
        using THistoryManager = TIncidentsHistoryManager;
        using THistoryReader = TCallbackSequentialTableImpl<TObjectEvent<TRecord>, TString>;

        using IUniquePolicy = IIncidentsStorageUniquePolicy;
        using TDefaultUniquePolicy = TDefaultIncidentsStorageUniquePolicy;

    public:
        TIncidentsStorage(const IHistoryContext& context, const THistoryConfig& hConfig);

        void Start();
        void Stop();

        TString GetTableName() const;

        TOptionalRecord Add(const TIncidentData& data, const TString& originatorId, NDrive::TEntitySession& session) const;
        TOptionalRecord Update(const TIncidentData& data, const TString& originatorId, NDrive::TEntitySession& session, const IUniquePolicy& uniquePolicy = TDefaultUniquePolicy()) const;
        TOptionalRecord ApplyTransition(const TIncidentData& data, const TString& originatorId, NDrive::TEntitySession& session) const;

        TOptionalRecordState GetPreviousState(const TString& incidentId, NDrive::TEntitySession& session) const;

    protected:
        virtual bool DoRebuildCacheUnsafe() const override;
        virtual bool DoAcceptHistoryEventUnsafe(const TAtomicSharedPtr<TObjectEvent<TRecord>>& dbEvent, const bool isNewEvent) override;
        virtual TStringBuf GetEventObjectId(const TObjectEvent<TRecord>& ev) const override;

    private:
        THistoryManager HistoryManager;
    };
}
