#include "filters.h"

#include <util/generic/algorithm.h>

namespace NDrive {
    TIncidentFilterGroup::operator bool() const noexcept {
        return !Empty();
    }

    bool TIncidentFilterGroup::Empty() const noexcept {
        return Filters.empty();
    }

    void TIncidentFilterGroup::Append(IFilterPtr filter) {
        Filters.push_back(filter);
    }

    bool TIncidentFilterGroup::Match(const TEntry& entry) const {
        return AllOf(Filters, [&entry](IFilterPtr f) { return f->Match(entry); });
    }

    TIncidentIdFilter::TIncidentIdFilter(const TString& id)
        : Id(id)
    {
    }

    bool TIncidentIdFilter::Match(const TEntry& entry) const {
        return Id == entry.GetIncidentId();
    }

    TIncidentTypeFilter::TIncidentTypeFilter(const EIncidentType& type)
        : Types({type})
    {
    }

    TIncidentTypeFilter::TIncidentTypeFilter(const TVector<EIncidentType>& types)
        : Types(types)
    {
    }

    bool TIncidentTypeFilter::Match(const TEntry& entry) const {
        return Find(Types, entry.GetIncidentType()) != Types.end();
    }

    TIncidentStatusFilter::TIncidentStatusFilter(const EIncidentStatus& status)
        : Statuses({status})
    {
    }

    TIncidentStatusFilter::TIncidentStatusFilter(const TVector<EIncidentStatus>& statuses)
        : Statuses(statuses)
    {
    }

    bool TIncidentStatusFilter::Match(const TEntry& entry) const {
        return Find(Statuses, entry.GetStatus()) != Statuses.end();
    }

    TIncidentInitiationTimeFilter::TIncidentInitiationTimeFilter(const TMaybe<TInstant> since, const TMaybe<TInstant> until)
        : Since(since)
        , Until(until)
    {
    }

    bool TIncidentInitiationTimeFilter::Match(const TEntry& entry) const {
        const auto timestamp = entry.GetInitiatedAt();
        if (!timestamp) {
            return false;
        }
        if (Since && *Since && timestamp < *Since) {
            return false;
        }
        if (Until && *Until && timestamp >= *Until) {
            return false;
        }
        return true;
    }

    TIncidentCarFilter::TIncidentCarFilter(const TString& carId)
        : CarId(carId)
    {
    }

    bool TIncidentCarFilter::Match(const TEntry& entry) const {
        return CarId == entry.GetCarId();
    }

    TIncidentUserFilter::TIncidentUserFilter(const TString& userId)
        : UserId(userId)
    {
    }

    bool TIncidentUserFilter::Match(const TEntry& entry) const {
        return UserId == entry.GetUserId();
    }

    TIncidentSessionFilter::TIncidentSessionFilter(const TString& sessionId)
        : SessionId(sessionId)
    {
    }

    bool TIncidentSessionFilter::Match(const TEntry& entry) const {
        return SessionId == entry.GetSessionId();
    }
}
