#pragma once

#include "constants.h"
#include "incident.h"

namespace NDrive {
    class IIncidentFilter {
    public:
        using TEntry = TIncidentData;
        using TPtr = TAtomicSharedPtr<IIncidentFilter>;

        virtual ~IIncidentFilter() = default;

        virtual bool Match(const TEntry& entry) const = 0;
    };

    class TIncidentFilterGroup {
    public:
        using TEntry = IIncidentFilter::TEntry;
        using IFilterPtr = IIncidentFilter::TPtr;

        TIncidentFilterGroup() = default;
        TIncidentFilterGroup(const TIncidentFilterGroup&) = default;
        TIncidentFilterGroup(std::initializer_list<IFilterPtr> il)
            : Filters(il)
        {
        }

        template<typename TIter>
        TIncidentFilterGroup(TIter first, TIter last)
            : Filters(first, last)
        {
        }

        operator bool() const noexcept;
        bool Empty() const noexcept;

        void Append(IFilterPtr filter);
        bool Match(const TIncidentData& entry) const;

    private:
        TVector<IFilterPtr> Filters;
    };

    class TIncidentIdFilter: public IIncidentFilter {
    public:
        explicit TIncidentIdFilter(const TString& id);
        virtual bool Match(const TEntry& entry) const override;

    private:
        R_READONLY(TString, Id);
    };

    class TIncidentTypeFilter: public IIncidentFilter {
    public:
        explicit TIncidentTypeFilter(const EIncidentType& type);
        explicit TIncidentTypeFilter(const TVector<EIncidentType>& types);
        virtual bool Match(const TEntry& entry) const override;

    private:
        R_READONLY(TVector<EIncidentType>, Types);
    };

    class TIncidentStatusFilter: public IIncidentFilter {
    public:
        explicit TIncidentStatusFilter(const EIncidentStatus& status);
        explicit TIncidentStatusFilter(const TVector<EIncidentStatus>& statuses);
        virtual bool Match(const TEntry& entry) const override;

    private:
        R_READONLY(TVector<EIncidentStatus>, Statuses);
    };

    class TIncidentInitiationTimeFilter: public IIncidentFilter {
    public:
        explicit TIncidentInitiationTimeFilter(const TMaybe<TInstant> since = {}, const TMaybe<TInstant> until = {});
        virtual bool Match(const TEntry& entry) const override;

    private:
        R_OPTIONAL(TInstant, Since);
        R_OPTIONAL(TInstant, Until);
    };

    class TIncidentCarFilter: public IIncidentFilter {
    public:
        explicit TIncidentCarFilter(const TString& carId);
        virtual bool Match(const TEntry& entry) const override;

    private:
        R_READONLY(TString, CarId);
    };

    class TIncidentUserFilter: public IIncidentFilter {
    public:
        explicit TIncidentUserFilter(const TString& userId);
        virtual bool Match(const TEntry& entry) const override;

    private:
        R_READONLY(TString, UserId);
    };

    class TIncidentSessionFilter: public IIncidentFilter {
    public:
        explicit TIncidentSessionFilter(const TString& sessionId);
        virtual bool Match(const TEntry& entry) const override;

    private:
        R_READONLY(TString, SessionId);
    };
}
