#include "incident.h"

#include <drive/backend/common/scheme_adapters.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/string_utils/base64/base64.h>

#include <rtline/library/json/parse.h>
#include <rtline/util/types/messages_collector.h>
#include <rtline/util/types/uuid.h>

#include <util/generic/algorithm.h>
#include <util/generic/serialized_enum.h>

namespace NDrive {
    TIncidentData::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
        IncidentId = GetFieldDecodeIndex("incident_id", decoderBase);
        IncidentSerialId = GetFieldDecodeIndex("incident_serial_id", decoderBase);
        IncidentType = GetFieldDecodeIndex("incident_type", decoderBase);
        InitiatedAt = GetFieldDecodeIndex("initiated_at_timestamp", decoderBase);
        Status = GetFieldDecodeIndex("status", decoderBase);
        UserId = GetFieldDecodeIndex("user_id", decoderBase);
        SessionId = GetFieldDecodeIndex("session_id", decoderBase);
        CarId = GetFieldDecodeIndex("car_id", decoderBase);
        Meta = GetFieldDecodeIndex("meta", decoderBase);
    }

    TIncidentData::TIncidentData(EIncidentType type)
        : IncidentId(NUtil::CreateUUID())
        , IncidentType(type)
        , InitiatedAt(Now())
    {
    }

    TString TIncidentData::GetTableName() {
        return "drive_incidents";
    }

    TString TIncidentData::GetHistoryTableName() {
        return GetTableName() + "_history";
    }

    constexpr bool TIncidentData::IsCarIdRequired(EIncidentType incidentType) noexcept {
        return incidentType == EIncidentType::RoadAccident ||
               incidentType == EIncidentType::AccountTheft ||
               incidentType == EIncidentType::CarTheftDuringRide ||
               incidentType == EIncidentType::Evacuation;
    }

    constexpr bool TIncidentData::IsUserIdRequired(EIncidentType incidentType) noexcept {
        return incidentType == EIncidentType::RoadAccident ||
               incidentType == EIncidentType::DriveTransfer ||
               incidentType == EIncidentType::AccountTransfer ||
               incidentType == EIncidentType::DrunkDriver ||
               incidentType == EIncidentType::CarTheftDuringRide;
    }

    bool TIncidentData::HasContext(const EIncidentContextType contextType) const {
        return GetContext(contextType) != nullptr;
    }

    TIncidentData::TIncidentContextPtr TIncidentData::GetContext(const EIncidentContextType contextType) const {
        auto contextIt = FindIf(Contexts, [contextType](const auto& contextPtr){ return contextPtr->GetContextType() == contextType; });
        return (contextIt != Contexts.end()) ? (*contextIt) : nullptr;
    }

    void TIncidentData::UpsertContext(TIncidentContextPtr contextPtr) {
        if (contextPtr) {
            const EIncidentContextType contextType = contextPtr->GetContextType();
            auto contextIt = FindIf(Contexts, [contextType](const auto& existingContextPtr){ return existingContextPtr->GetContextType() == contextType; });
            if (contextIt != Contexts.end()) {
                *contextIt = contextPtr;
            } else {
                Contexts.push_back(contextPtr);
            }
        }
    }

    void TIncidentData::AddPhotoLink(TIncidentPhotoLink link) {
        PhotoLinks.push_back(std::move(link));
    }

    TMaybe<TIncidentPhotoLink> TIncidentData::GetPhotoLink(const TIncidentPhotoLink::TImageId& imageId) const {
        auto existingLinkIt = FindIf(PhotoLinks, [&imageId](const auto& photoLink){ return imageId == photoLink.GetImageId(); });
        if (existingLinkIt != PhotoLinks.end()) {
            return *existingLinkIt;
        }
        return {};
    }

    void TIncidentData::UpsertPhotoLink(TIncidentPhotoLink link, const TIncidentPhotoLink::TImageId& imageId) {
        auto existingLinkIt = FindIf(PhotoLinks, [&imageId](const auto& photoLink){ return imageId == photoLink.GetImageId(); });
        if (existingLinkIt != PhotoLinks.end()) {
            *existingLinkIt = std::move(link);
        } else {
            PhotoLinks.push_back(std::move(link));
        }
    }

    void TIncidentData::RemovePhotoLink(const TIncidentPhotoLink::TImageId& imageId) {
        RemovePhotoLinks({imageId});
    }

    void TIncidentData::RemovePhotoLinks(const TSet<TIncidentPhotoLink::TImageId>& imageIds) {
        EraseIf(PhotoLinks, [&imageIds](const auto& photoLink){ return imageIds.contains(photoLink.GetImageId()); });
    }

    void TIncidentData::AddTagLink(TIncidentTagLink link) {
        TagLinks.push_back(std::move(link));
    }

    TMaybe<TIncidentTagLink> TIncidentData::GetTagLink(const TString& tagId) const {
        auto existingLinkIt = FindIf(TagLinks, [&tagId](const auto& tagLink){ return tagId == tagLink.GetTagId(); });
        if (existingLinkIt != TagLinks.end()) {
            return *existingLinkIt;
        }
        return {};
    }

    TVector<TIncidentTagLink> TIncidentData::GetTagLinks(const EIncidentTransition transitionType) const {
        TVector<TIncidentTagLink> links;
        for (const auto& tagLink: TagLinks) {
            if (transitionType == tagLink.OptionalSourceTransitionId()) {
                links.push_back(tagLink);
            }
        }
        return links;
    }

    void TIncidentData::UpsertTagLink(TIncidentTagLink link, const TString& tagId) {
        auto existingLinkIt = FindIf(TagLinks, [&tagId](const auto& tagLink){ return tagId == tagLink.GetTagId(); });
        if (existingLinkIt != TagLinks.end()) {
            *existingLinkIt = std::move(link);
        } else {
            TagLinks.push_back(std::move(link));
        }
    }

    void TIncidentData::RemoveTagLink(const TString& tagId) {
        RemoveTagLinks({tagId});
    }

    void TIncidentData::RemoveTagLinks(const TSet<TString>& tagIds) {
        EraseIf(TagLinks, [&tagIds](const auto& tagLink){ return tagIds.contains(tagLink.GetTagId()); });
    }

    void TIncidentData::AddStartrekTicketLink(TIncidentStartrekTicketLink link) {
        StartrekTicketLinks.push_back(std::move(link));
    }

    TMaybe<TIncidentStartrekTicketLink> TIncidentData::GetStartrekTicketLink(const EIncidentTransition transitionType) const {
        auto existingLinkIt = FindIf(StartrekTicketLinks, [transitionType](const auto& link){ return transitionType == link.OptionalSourceTransitionId(); });
        if (existingLinkIt != StartrekTicketLinks.end()) {
            return *existingLinkIt;
        }
        return {};
    }

    void TIncidentData::UpsertStartrekTicketLink(TIncidentStartrekTicketLink ticketLink, const EIncidentTransition transitionType) {
        auto existingLinkIt = FindIf(StartrekTicketLinks, [transitionType](const auto& link){ return transitionType == link.OptionalSourceTransitionId(); });
        if (existingLinkIt != StartrekTicketLinks.end()) {
            *existingLinkIt = std::move(ticketLink);
        } else {
            StartrekTicketLinks.push_back(std::move(ticketLink));
        }
    }

    NDrive::TScheme TIncidentData::GetInitialScheme(const TMaybe<TIncidentData>& incident) {
        NDrive::TScheme scheme;

        auto& incidentTypeScheme = scheme.Add<TFSVariable>("incident_type");
        incidentTypeScheme.MutableCondition().SetDescription("Тип инцидента").SetRequired(true);

        if (incident) {
            auto variantAdapter = TLocalizedVariantAdapter::FromEnum(incident->GetIncidentType());
            incidentTypeScheme.MutableCondition().AddCompoundVariant(variantAdapter).SetDefault(variantAdapter.GetValue());

            NDrive::TScheme innerScheme;

            innerScheme.Add<TFSString>("car_id", "Id автомобиля")
                       .SetVisual(TFSString::EVisualType::ObjectId)
                       .SetDefault(incident->GetCarId())
                       .SetRequired(IsCarIdRequired(incident->GetIncidentType()));
            innerScheme.Add<TFSString>("user_id", "Id пользователя")
                       .SetVisual(TFSString::EVisualType::ObjectId)
                       .SetDefault(incident->GetUserId())
                       .SetRequired(IsUserIdRequired(incident->GetIncidentType()));
            innerScheme.Add<TFSString>("session_id", "Id сессии")
                       .SetVisual(TFSString::EVisualType::GUID)
                       .SetDefault(incident->GetSessionId());

            incidentTypeScheme.AddVariant(variantAdapter, innerScheme);
        } else {
            for (auto&& incidentType: GetEnumAllValues<EIncidentType>()) {
                auto variantAdapter = TLocalizedVariantAdapter::FromEnum(incidentType);

                NDrive::TScheme innerScheme;

                innerScheme.Add<TFSString>("car_id", "Id автомобиля")
                           .SetVisual(TFSString::EVisualType::ObjectId)
                           .SetRequired(IsCarIdRequired(incidentType));
                innerScheme.Add<TFSString>("user_id", "Id пользователя")
                           .SetVisual(TFSString::EVisualType::ObjectId)
                           .SetRequired(IsUserIdRequired(incidentType));
                innerScheme.Add<TFSString>("session_id", "Id сессии")
                           .SetVisual(TFSString::EVisualType::GUID);

                incidentTypeScheme.AddVariant(variantAdapter, innerScheme);
            }
        }

        return scheme;
    }

    TMaybe<TIncidentData> TIncidentData::ConstructInitial(const NJson::TJsonValue& data, TMessagesCollector& errors) {
        EIncidentType incidentType;
        if (!NJson::ParseField(data["incident_type"], NJson::Stringify(incidentType), /* required = */ true, errors)) {
            return {};
        }

        TIncidentData instance(incidentType);
        if (!instance.DeserializeDataFromJson(data, errors)) {
            return {};
        }
        return instance;
    }

    bool TIncidentData::DeserializeDataFromJson(const NJson::TJsonValue& data, TMessagesCollector& errors) {
        return NJson::ParseField(data["car_id"], CarId, /* required = */ IsCarIdRequired(GetIncidentType()), errors) &&
               NJson::ParseField(data["user_id"], UserId, /* required = */ IsUserIdRequired(GetIncidentType()), errors) &&
               NJson::ParseField(data["session_id"], SessionId, /* required = */ false, errors);
    }

    NJson::TJsonValue TIncidentData::BuildReport(const ELocalization locale) const {
        NJson::TJsonValue result;

        NJson::InsertField(result, "incident_id", IncidentId);
        NJson::InsertField(result, "incident_type", TLocalizedVariantAdapter::FromEnum(IncidentType, {}, locale).GetLocalizedValue());
        NJson::InsertField(result, "initiated_at", NJson::Seconds(InitiatedAt));

        NJson::InsertField(result, "incident_status", TLocalizedVariantAdapter::FromEnum(Status, {}, locale).GetLocalizedValue());

        NJson::InsertField(result, "session_id", SessionId);
        NJson::InsertField(result, "user_id", UserId);
        NJson::InsertField(result, "car_id", CarId);

        auto& contexts = result.InsertValue("contexts", NJson::JSON_ARRAY);
        for (const auto& contextPtr: Contexts) {
            contexts.AppendValue(contextPtr->SerializeToJson());
        }

        auto& links = result.InsertValue("links", NJson::JSON_ARRAY);
        {
            for (const auto& link: DocumentLinks) {
                links.AppendValue(link.BuildReport());
            }
            for (const auto& link: PhotoLinks) {
                links.AppendValue(link.BuildReport());
            }
            for (const auto& link: TagLinks) {
                links.AppendValue(link.BuildReport());
            }
            for (const auto& link: StartrekTicketLinks) {
                links.AppendValue(link.BuildReport());
            }
            for (const auto& link: UserDocumentPhotoLinks) {
                links.AppendValue(link.BuildReport());
            }
        }

        return result;
    }

    bool TIncidentData::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /* hContext */) {
        READ_DECODER_VALUE(decoder, values, IncidentId);
        READ_DECODER_VALUE(decoder, values, IncidentSerialId);
        READ_DECODER_VALUE(decoder, values, IncidentType);
        READ_DECODER_VALUE_INSTANT(decoder, values, InitiatedAt);

        READ_DECODER_VALUE(decoder, values, Status);

        READ_DECODER_VALUE_DEF(decoder, values, SessionId, Default<TString>());
        READ_DECODER_VALUE_DEF(decoder, values, UserId, Default<TString>());
        READ_DECODER_VALUE_DEF(decoder, values, CarId, Default<TString>());

        {
            NDrive::NProto::TIncidentMeta metaProto;
            {
                TString meta;
                READ_DECODER_VALUE_TEMP(decoder, values, meta, Meta);
                TString decoded = Base64Decode(meta);
                if (!metaProto.ParseFromArray(decoded.Data(), decoded.Size())) {
                    return false;
                }
            }

            if (metaProto.HasContext()) {
                NJson::TJsonValue contextJson;
                if (!NJson::ReadJsonTree(metaProto.GetContext(), &contextJson) || !contextJson.IsArray()) {
                    return false;
                }

                TMessagesCollector errors;
                for (const auto& contextItemJson: contextJson.GetArray()) {
                    TIncidentContextPtr contextPtr = IIncidentContext::Construct(contextItemJson, errors);
                    if (contextPtr) {
                        Contexts.push_back(contextPtr);
                    } else {
                        return false;
                    }
                }
            }

            for (auto&& linkProto: metaProto.GetDocumentLinks()) {
                TIncidentDocumentLink documentLink;
                if (!documentLink.ParseFromProto(linkProto)) {
                    return false;
                }
                DocumentLinks.push_back(std::move(documentLink));
            }

            for (auto&& linkProto: metaProto.GetPhotoLinks()) {
                TIncidentPhotoLink photoLink;
                if (!photoLink.ParseFromProto(linkProto)) {
                    return false;
                }
                PhotoLinks.push_back(std::move(photoLink));
            }

            for (auto&& linkProto: metaProto.GetTagLinks()) {
                TIncidentTagLink tagLink;
                if (!tagLink.ParseFromProto(linkProto)) {
                    return false;
                }
                TagLinks.push_back(std::move(tagLink));
            }

            for (auto&& linkProto: metaProto.GetStartrekTicketLinks()) {
                TIncidentStartrekTicketLink startrekTicketLink;
                if (!startrekTicketLink.ParseFromProto(linkProto)) {
                    return false;
                }
                StartrekTicketLinks.push_back(std::move(startrekTicketLink));
            }

            for (auto&& linkProto: metaProto.GetUserDocumentPhotoLinks()) {
                TUserDocumentPhotoLink link;
                if (!link.ParseFromProto(linkProto)) {
                    return false;
                }
                UserDocumentPhotoLinks.push_back(std::move(link));
            }
        }

        return true;
    }

    NStorage::TTableRecord TIncidentData::SerializeToTableRecord() const {
        NStorage::TTableRecord record;

        record.Set("incident_id", IncidentId);
        record.Set("incident_serial_id", (IncidentSerialId) ? ::ToString(IncidentSerialId) : "nextval('drive_incidents_incident_serial_id_seq')");
        record.Set("incident_type", IncidentType);
        record.Set("initiated_at_timestamp", InitiatedAt.Seconds());

        record.Set("status", Status);

        record.Set("session_id", (SessionId) ? SessionId : "get_null()");
        record.Set("user_id", (UserId) ? UserId : "get_null()");
        record.Set("car_id", (CarId) ? CarId : "get_null()");

        // NB. Meta proto has to be base64-encoded as
        //   sql&postres wrappers bundle does not implement binary string escaping, quotation
        //   and passing into corresponding libpqxx methods as well as table column has text type instead of bytea
        TString metaProtoStr;
        {
            NDrive::NProto::TIncidentMeta metaProto;

            if (Contexts) {
                NJson::TJsonValue contextJson(NJson::JSON_ARRAY);
                for (const auto& contextPtr: Contexts) {
                    contextJson.AppendValue(contextPtr->SerializeToJson());
                }
                metaProto.SetContext(contextJson.GetStringRobust());
            }

            for (auto&& documentLink: DocumentLinks) {
                auto documentLinkProtoPtr = metaProto.AddDocumentLinks();
                documentLink.SerializeToProto(*documentLinkProtoPtr);
            }

            for (auto&& photoLink: PhotoLinks) {
                auto photoLinkProtoPtr = metaProto.AddPhotoLinks();
                photoLink.SerializeToProto(*photoLinkProtoPtr);
            }

            for (auto&& tagLink: TagLinks) {
                auto tagLinkProtoPtr = metaProto.AddTagLinks();
                tagLink.SerializeToProto(*tagLinkProtoPtr);
            }

            for (auto&& startrekTicketLink: StartrekTicketLinks) {
                auto startrekTicketLinkProtoPtr = metaProto.AddStartrekTicketLinks();
                startrekTicketLink.SerializeToProto(*startrekTicketLinkProtoPtr);
            }

            for (auto&& link: UserDocumentPhotoLinks) {
                auto linkProtoPtr = metaProto.AddUserDocumentPhotoLinks();
                link.SerializeToProto(*linkProtoPtr);
            }

            metaProtoStr = metaProto.SerializeAsString();
        }
        record.Set("meta", (metaProtoStr) ? Base64Encode(metaProtoStr) : "get_null()");

        return record;
    }

    NStorage::TTableRecord TIncidentData::SerializeToTableRecordForUpdate() const {
        auto record = SerializeToTableRecord();
        UpdateRecordRevisionInfo(record);
        return record;
    }

    void TIncidentData::UpdateRecordRevisionInfo(NStorage::TTableRecord& record) const {
        record.ForceSet("incident_serial_id", "nextval('drive_incidents_incident_serial_id_seq')");
    }
}
