#pragma once

#include "constants.h"
#include "incident_context.h"
#include "incident_link.h"

#include <drive/backend/incident/proto/incident.pb.h>

#include <drive/backend/abstract/localization.h>
#include <drive/backend/database/history/manager.h>

#include <library/cpp/json/json_value.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/types/accessor.h>

class TMessagesCollector;

namespace NDrive {
    class TScheme;
}

namespace NDrive {
    class TIncidentData {
    public:
        class TDecoder: public TBaseDecoder {
        public:
            TDecoder() = default;
            TDecoder(const TMap<TString, ui32>& decoderBase);

        private:
            R_FIELD(i32, IncidentId, -1);
            R_FIELD(i32, IncidentSerialId, -1);
            R_FIELD(i32, IncidentType, -1);
            R_FIELD(i32, InitiatedAt, -1);
            R_FIELD(i32, Status, -1);
            R_FIELD(i32, SessionId, -1);
            R_FIELD(i32, UserId, -1);
            R_FIELD(i32, CarId, -1);
            R_FIELD(i32, Meta, -1);
        };

    public:
        using TIncidentContextPtr = IIncidentContext::TPtr;
        using TIncidentContextPtrs = TVector<TIncidentContextPtr>;

        using TIncidentLinkPtr = IIncidentLink::TPtr;
        using TIncidentLinkPtrs = TVector<TIncidentLinkPtr>;

        explicit TIncidentData(EIncidentType type = EIncidentType::RoadAccident);

        static TString GetTableName();
        static TString GetHistoryTableName();

        static constexpr bool IsCarIdRequired(EIncidentType incidentType) noexcept;
        static constexpr bool IsUserIdRequired(EIncidentType incidentType) noexcept;

        bool HasContext(const EIncidentContextType contextType) const;

        template <typename TContext>
        bool HasContext() const {
            return HasContext(TContext::GetTypeName());
        }

        TIncidentContextPtr GetContext(const EIncidentContextType contextType) const;

        template <typename TContext>
        TAtomicSharedPtr<TContext> GetContext() const {
            return std::dynamic_pointer_cast<TContext>(GetContext(TContext::GetTypeName()));
        }

        void UpsertContext(TIncidentContextPtr contextPtr);

        void AddPhotoLink(TIncidentPhotoLink link);
        TMaybe<TIncidentPhotoLink> GetPhotoLink(const TIncidentPhotoLink::TImageId& imageId) const;
        void UpsertPhotoLink(TIncidentPhotoLink link, const TIncidentPhotoLink::TImageId& imageId);
        void RemovePhotoLink(const TIncidentPhotoLink::TImageId& imageId);
        void RemovePhotoLinks(const TSet<TIncidentPhotoLink::TImageId>& imageIds);

        void AddTagLink(TIncidentTagLink link);
        TMaybe<TIncidentTagLink> GetTagLink(const TString& tagId) const;
        TVector<TIncidentTagLink> GetTagLinks(const EIncidentTransition transitionType) const;
        void UpsertTagLink(TIncidentTagLink link, const TString& tagId);
        void RemoveTagLink(const TString& tagId);
        void RemoveTagLinks(const TSet<TString>& tagIds);

        void AddStartrekTicketLink(TIncidentStartrekTicketLink link);
        TMaybe<TIncidentStartrekTicketLink> GetStartrekTicketLink(const EIncidentTransition transitionType) const;
        void UpsertStartrekTicketLink(TIncidentStartrekTicketLink link, const EIncidentTransition transitionType);

        static NDrive::TScheme GetInitialScheme(const TMaybe<TIncidentData>& incident = {});
        static TMaybe<TIncidentData> ConstructInitial(const NJson::TJsonValue& data, TMessagesCollector& errors);

        bool DeserializeDataFromJson(const NJson::TJsonValue& data, TMessagesCollector& errors);
        NJson::TJsonValue BuildReport(const ELocalization locale = ELocalization::Rus) const;

        bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext);
        NStorage::TTableRecord SerializeToTableRecord() const;
        NStorage::TTableRecord SerializeToTableRecordForUpdate() const;

    private:
        void UpdateRecordRevisionInfo(NStorage::TTableRecord& record) const;

    private:
        R_READONLY(TString, IncidentId);
        R_READONLY(ui64, IncidentSerialId, 0);
        R_READONLY(EIncidentType, IncidentType, EIncidentType::RoadAccident);
        R_READONLY(TInstant, InitiatedAt);

        R_FIELD(EIncidentStatus, Status, EIncidentStatus::New);

        R_FIELD(TString, SessionId);
        R_FIELD(TString, UserId);
        R_FIELD(TString, CarId);

        R_READONLY(TIncidentContextPtrs, Contexts);

        R_FIELD(TVector<TIncidentDocumentLink>, DocumentLinks);
        R_FIELD(TVector<TIncidentPhotoLink>, PhotoLinks);
        R_FIELD(TVector<TIncidentTagLink>, TagLinks);
        R_FIELD(TVector<TIncidentStartrekTicketLink>, StartrekTicketLinks);
        R_FIELD(TVector<TUserDocumentPhotoLink>, UserDocumentPhotoLinks);
    };
}
