#include "incident_link.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/abstract/frontend.h>

#include <drive/library/cpp/startrek/client.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>

#include <util/string/cast.h>

namespace NDrive {
    NJson::TJsonValue IIncidentLink::BuildReport() const {
        NJson::TJsonValue result = DoBuildReport();
        NJson::InsertField(result, "type", ::ToString(GetType()));
        NJson::InsertField(result, "last_modified", NJson::Seconds(LastModifiedAt));
        NJson::InsertField(result, "originator_id", OriginatorId);
        NJson::InsertField(result, "comment", Comment);
        NJson::InsertField(result, "source_transition_id", (SourceTransitionId) ? ::ToString(*SourceTransitionId) : "");
        return result;
    }

    TIncidentDocumentLink::ELinkType TIncidentDocumentLink::GetType() const {
        return ELinkType::Document;
    }

    bool TIncidentDocumentLink::DoParseFromProto(const TProto& proto) {
        DocumentId = proto.GetDocumentId();
        if (proto.HasDocumentType()) {
            DocumentType = proto.GetDocumentType();
        }
        if (proto.HasDocumentUrl()) {
            DocumentUrl = proto.GetDocumentUrl();
        }
        if (proto.HasAttachmentCode()) {
            AttachmentCode = proto.GetAttachmentCode();
        }
        if (proto.HasLastSentTimestamp()) {
            LastSentAt = TInstant::Seconds(proto.GetLastSentTimestamp());
        }
        return true;
    }

    void TIncidentDocumentLink::DoSerializeToProto(TProto& proto) const {
        proto.SetDocumentId(DocumentId);
        if (DocumentType) {
            proto.SetDocumentType(DocumentType);
        }
        if (DocumentUrl) {
            proto.SetDocumentUrl(DocumentUrl);
        }
        if (AttachmentCode) {
            proto.SetAttachmentCode(AttachmentCode);
        }
        if (LastSentAt) {
            proto.SetLastSentTimestamp(LastSentAt->Seconds());
        }
    }

    NJson::TJsonValue TIncidentDocumentLink::DoBuildReport() const {
        return NJson::TMapBuilder("attachment_code", AttachmentCode)
                                 ("document_id", DocumentId)
                                 ("document_type", DocumentType)
                                 ("document_url", DocumentUrl)
                                 ("last_sent", (LastSentAt) ? NJson::TJsonValue(LastSentAt->Seconds()) : NJson::TJsonValue(NJson::JSON_NULL));
    }

    TIncidentPhotoLink::TIncidentPhotoLink(const TImageId& imageId, const TString& imageUrl, const TString& attachmentCode, const EIncidentTransition transitionId, const TString& performerId, const TInstant lastModifiedAt)
        : ImageId(imageId)
        , ImageUrl(imageUrl)
        , AttachmentCode(attachmentCode)
    {
        SetOriginatorId(performerId);
        SetLastModifiedAt(lastModifiedAt ? lastModifiedAt : Now());
        SetSourceTransitionId(transitionId);
    }

    TIncidentPhotoLink::ELinkType TIncidentPhotoLink::GetType() const {
        return ELinkType::Photo;
    }

    bool TIncidentPhotoLink::DoParseFromProto(const TProto& proto) {
        ImageId = proto.GetImageId();
        if (proto.HasImageUrl()) {
            ImageUrl = proto.GetImageUrl();
        }
        if (proto.HasAttachmentCode()) {
            AttachmentCode = proto.GetAttachmentCode();
        }
        if (proto.HasLastSentTimestamp()) {
            LastSentAt = TInstant::Seconds(proto.GetLastSentTimestamp());
        }
        return true;
    }

    void TIncidentPhotoLink::DoSerializeToProto(TProto& proto) const {
        proto.SetImageId(ImageId);
        if (ImageUrl) {
            proto.SetImageUrl(ImageUrl);
        }
        if (AttachmentCode) {
            proto.SetAttachmentCode(AttachmentCode);
        }
        if (LastSentAt) {
            proto.SetLastSentTimestamp(LastSentAt->Seconds());
        }
    }

    NJson::TJsonValue TIncidentPhotoLink::DoBuildReport() const {
        return NJson::TMapBuilder
            ("image_id", ImageId)
            ("image_url", ImageUrl)
            ("attachment_code", AttachmentCode)
            ("last_sent", (LastSentAt) ? NJson::TJsonValue(LastSentAt->Seconds()) : NJson::TJsonValue(NJson::JSON_NULL));
    }

    TIncidentTagLink::TIncidentTagLink(const TString& tagId, const NEntityTagsManager::EEntityType tagType, const EIncidentTransition transitionId, const TString& performerId, const TInstant lastModifiedAt)
        : TagId(tagId)
        , TagEntityType(tagType)
    {
        SetOriginatorId(performerId);
        SetLastModifiedAt(lastModifiedAt ? lastModifiedAt : Now());
        SetSourceTransitionId(transitionId);
    }

    TIncidentTagLink::ELinkType TIncidentTagLink::GetType() const {
        return ELinkType::Tag;
    }

    bool TIncidentTagLink::DoParseFromProto(const TProto& proto) {
        TagId = proto.GetTagId();

        TString tagEntityTypeStr = proto.GetTagEntityType();
        if (!TryFromString(tagEntityTypeStr, TagEntityType)) {
            return false;
        }

        return true;
    }

    void TIncidentTagLink::DoSerializeToProto(TProto& proto) const {
        proto.SetTagId(TagId);
        proto.SetTagEntityType(::ToString(TagEntityType));
    }

    NJson::TJsonValue TIncidentTagLink::DoBuildReport() const {
        return NJson::TMapBuilder("tag_id", TagId)("tag_entity_type", ::ToString(TagEntityType));
    }

    TIncidentStartrekTicketLink::TIncidentStartrekTicketLink(const TString& ticketKey, const EIncidentTransition transitionId, const TString& performerId, const TInstant lastModifiedAt)
        : TicketKey(ticketKey)
    {
        SetOriginatorId(performerId);
        SetLastModifiedAt(lastModifiedAt ? lastModifiedAt : Now());
        SetSourceTransitionId(transitionId);
    }

    TIncidentStartrekTicketLink::ELinkType TIncidentStartrekTicketLink::GetType() const {
        return ELinkType::StartrekTicket;
    }

    TString TIncidentStartrekTicketLink::GetTicketUrl(const NDrive::IServer& server) const {
        return server.GetStartrekClient() ? server.GetStartrekClient()->GetTicketUri(TicketKey) : "";
    }

    bool TIncidentStartrekTicketLink::DoParseFromProto(const TProto& proto) {
        TicketKey = proto.GetTicketKey();
        return true;
    }

    void TIncidentStartrekTicketLink::DoSerializeToProto(TProto& proto) const {
        proto.SetTicketKey(TicketKey);
    }

    NJson::TJsonValue TIncidentStartrekTicketLink::DoBuildReport() const {
        TString ticketUrl = NDrive::HasServer() ? GetTicketUrl(NDrive::GetServer().GetAsSafe<NDrive::IServer>()): "";
        return NJson::TMapBuilder("ticket_key", TicketKey)("ticket_url", ticketUrl);
    }

    TUserDocumentPhotoLink::ELinkType TUserDocumentPhotoLink::GetType() const {
        return ELinkType::UserDocumentPhoto;
    }

    bool TUserDocumentPhotoLink::DoParseFromProto(const TProto& proto) {
        DocumentId = proto.GetDocumentId();
        if (proto.HasDocumentType()) {
            if (!TryFromString(proto.GetDocumentType(), DocumentType)) {
                return false;
            }
        }
        if (proto.HasAttachmentCode()) {
            AttachmentCode = proto.GetAttachmentCode();
        }
        return true;
    }

    void TUserDocumentPhotoLink::DoSerializeToProto(TProto& proto) const {
        proto.SetDocumentId(DocumentId);
        if (DocumentType) {
            proto.SetDocumentType(ToString(DocumentType));
        }
        if (AttachmentCode) {
            proto.SetAttachmentCode(AttachmentCode);
        }
    }

    NJson::TJsonValue TUserDocumentPhotoLink::DoBuildReport() const {
        return NJson::TMapBuilder
            ("attachment_code", AttachmentCode)
            ("document_id", DocumentId)
            ("document_type", ToString(DocumentType))
        ;
    }
}
