#pragma once

#include "constants.h"

#include <drive/backend/incident/proto/incident.pb.h>

#include <drive/backend/database/transaction/tx.h>
#include <drive/backend/tags/abstract.h>
#include <drive/backend/users/user_documents.h>

#include <library/cpp/json/json_value.h>

#include <rtline/util/types/accessor.h>

#include <util/string/cast.h>

namespace NDrive {
    class IServer;
}

namespace NDrive {
    class IIncidentLink {
    public:
        using TPtr = TAtomicSharedPtr<IIncidentLink>;
        using ELinkType = EIncidentLinkType;

        virtual ~IIncidentLink() = default;

        virtual ELinkType GetType() const = 0;

        NJson::TJsonValue BuildReport() const;

    private:
        virtual NJson::TJsonValue DoBuildReport() const = 0;

    private:
        R_FIELD(TInstant, LastModifiedAt);
        R_FIELD(TString, OriginatorId);
        R_FIELD(TString, Comment);

        R_OPTIONAL(EIncidentTransition, SourceTransitionId);
    };

    template <typename TProto>
    class IProtoSerializableIncidentLink: public IIncidentLink {
    public:
        virtual ~IProtoSerializableIncidentLink() = default;

        bool ParseFromProto(const TProto& proto) {
            if (proto.HasLastModifiedTimestamp()) {
                SetLastModifiedAt(TInstant::Seconds(proto.GetLastModifiedTimestamp()));
            }
            if (proto.HasOriginatorId()) {
                SetOriginatorId(proto.GetOriginatorId());
            }
            if (proto.HasComment()) {
                SetComment(proto.GetComment());
            }
            if (proto.HasSourceTransitionId()) {
                EIncidentTransition sourceTransitionId;
                TString sourceTransitionIdStr = proto.GetSourceTransitionId();
                if (TryFromString(sourceTransitionIdStr, sourceTransitionId)) {
                    SetSourceTransitionId(sourceTransitionId);
                }
            }
            return DoParseFromProto(proto);
        }

        void SerializeToProto(TProto& proto) const {
            DoSerializeToProto(proto);
            if (GetLastModifiedAt()) {
                proto.SetLastModifiedTimestamp(GetLastModifiedAt().Seconds());
            }
            if (GetOriginatorId()) {
                proto.SetOriginatorId(GetOriginatorId());
            }
            if (GetComment()) {
                proto.SetComment(GetComment());
            }
            if (HasSourceTransitionId()) {
                proto.SetSourceTransitionId(::ToString(GetSourceTransitionIdRef()));
            }
        }

    private:
        virtual bool DoParseFromProto(const TProto& proto) = 0;
        virtual void DoSerializeToProto(TProto& proto) const = 0;
    };

    class TIncidentDocumentLink: public IProtoSerializableIncidentLink<NDrive::NProto::TIncidentMeta::TDocumentLink> {
    public:
        using TProto = NDrive::NProto::TIncidentMeta::TDocumentLink;

        virtual ELinkType GetType() const override;

    private:
        virtual bool DoParseFromProto(const TProto& proto) override;
        virtual void DoSerializeToProto(TProto& proto) const override;

        virtual NJson::TJsonValue DoBuildReport() const override;

    private:
        R_FIELD(TString, AttachmentCode);  // Code used to identify Renins document types

        R_FIELD(TString, DocumentId);
        R_FIELD(TString, DocumentType);  // TString currenlty as usage cases are not determined
        R_FIELD(TString, DocumentUrl);

        R_OPTIONAL(TInstant, LastSentAt);
    };

    class TIncidentPhotoLink: public IProtoSerializableIncidentLink<NDrive::NProto::TIncidentMeta::TPhotoLink> {
    public:
        using TImageId = ui64;
        using TProto = NDrive::NProto::TIncidentMeta::TPhotoLink;

        TIncidentPhotoLink() = default;
        TIncidentPhotoLink(const TImageId& imageId, const TString& imageUrl, const TString& attachmentCode, const EIncidentTransition transitionId, const TString& performerId = {}, const TInstant lastModifiedAt = {});

        virtual ELinkType GetType() const override;

    private:
        virtual bool DoParseFromProto(const TProto& proto) override;
        virtual void DoSerializeToProto(TProto& proto) const override;

        virtual NJson::TJsonValue DoBuildReport() const override;

    private:
        R_FIELD(TImageId, ImageId, Default<TImageId>());
        R_FIELD(TString, ImageUrl);
        R_FIELD(TString, AttachmentCode);  // Code used to identify Renins document types

        R_OPTIONAL(TInstant, LastSentAt);
    };

    class TIncidentTagLink: public IProtoSerializableIncidentLink<NDrive::NProto::TIncidentMeta::TTagLink> {
    public:
        using TProto = NDrive::NProto::TIncidentMeta::TTagLink;

        TIncidentTagLink() = default;
        TIncidentTagLink(const TString& tagId, const NEntityTagsManager::EEntityType tagType, const EIncidentTransition transitionId, const TString& performerId = {}, const TInstant lastModifiedAt = {});

        virtual ELinkType GetType() const override;

    private:
        virtual bool DoParseFromProto(const TProto& proto) override;
        virtual void DoSerializeToProto(TProto& proto) const override;

        virtual NJson::TJsonValue DoBuildReport() const override;

    private:
        R_FIELD(TString, TagId);
        R_FIELD(NEntityTagsManager::EEntityType, TagEntityType);
    };

    class TIncidentStartrekTicketLink: public IProtoSerializableIncidentLink<NDrive::NProto::TIncidentMeta::TStartrekTicketLink> {
    public:
        using TProto = NDrive::NProto::TIncidentMeta::TStartrekTicketLink;

        TIncidentStartrekTicketLink() = default;
        TIncidentStartrekTicketLink(const TString& ticketKey, const EIncidentTransition transitionId, const TString& performerId = {}, const TInstant lastModifiedAt = {});

        virtual ELinkType GetType() const override;

        TString GetTicketUrl(const NDrive::IServer& server) const;

    private:
        virtual bool DoParseFromProto(const TProto& proto) override;
        virtual void DoSerializeToProto(TProto& proto) const override;

        virtual NJson::TJsonValue DoBuildReport() const override;

    private:
        R_FIELD(TString, TicketKey);
    };

    class TUserDocumentPhotoLink: public IProtoSerializableIncidentLink<NDrive::NProto::TIncidentMeta::TUserDocumentPhotoLink> {
    public:
        using TProto = NDrive::NProto::TIncidentMeta::TUserDocumentPhotoLink;

        virtual ELinkType GetType() const override;

    private:
        virtual bool DoParseFromProto(const TProto& proto) override;
        virtual void DoSerializeToProto(TProto& proto) const override;

        virtual NJson::TJsonValue DoBuildReport() const override;

    private:
        R_FIELD(TString, AttachmentCode);  // Code used to identify Renins document types

        R_FIELD(TString, DocumentId); // Also a PhotoId
        R_FIELD(NUserDocument::EType, DocumentType, NUserDocument::EType::Unknown);
    };
}
