#include "manager.h"

#include <util/generic/algorithm.h>
#include <util/string/cast.h>

namespace NDrive {
    TIncidentsManager::TIncidentsManager(const TIncidentsManagerConfig& config, const IHistoryContext& context, const NDrive::IServer& server)
        : TDatabaseSessionConstructor(context.GetDatabase())
        , Config(config)
        , Server(server)
        , IncidentsStorage(context, config.GetHistoryConfig())
    {
        Start();
    }

    TIncidentsManager::~TIncidentsManager() {
        Stop();
    }

    void TIncidentsManager::Start() {
        IncidentsStorage.Start();
    }

    void TIncidentsManager::Stop() {
        IncidentsStorage.Stop();
    }

    TIncidentsManager::TOptionalIncident TIncidentsManager::CreateIncident(const TIncidentData& data, const TString& orginatorId, NDrive::TEntitySession& session) const {
        return IncidentsStorage.Add(data, orginatorId, session);
    }

    TIncidentsManager::TOptionalIncident TIncidentsManager::UpdateIncident(const TIncidentData& data, const TString& orginatorId, NDrive::TEntitySession& session) const {
        return IncidentsStorage.Update(data, orginatorId, session);
    }

    TIncidentsManager::TTransitionPtr TIncidentsManager::ConstructTransition(const EIncidentTransition transitionId, const NJson::TJsonValue& transitionData, const TIncidentStateContext& context, TMessagesCollector& errors) const {
        return StateManager.ConstructTransition(::ToString(transitionId), transitionData, context, errors);
    }

    bool TIncidentsManager::CheckTransitionAllowed(const TString& incidentId, const TString& transitionId, NDrive::TEntitySession& session) const {
        TTransitions transitions;
        if (!GetAvailableTransitions(incidentId, {transitionId}, transitions, session) || transitions.size() != 1) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "invalid transition", EDriveSessionResult::IncorrectRequest);
            return false;
        }
        return true;
    }

    bool TIncidentsManager::GetAvailableTransitions(const TString& incidentId, TTransitions& transitions, NDrive::TEntitySession& session) const {
        auto incidentInstance = GetIncident(incidentId, session);
        if (!incidentInstance) {
            return false;
        }
        auto result = GetAvailableTransitions(*incidentInstance, {}, session);
        if (!result) {
            return false;
        }
        transitions = std::move(*result);
        return true;
    }

    bool TIncidentsManager::GetAvailableTransitions(const TString& incidentId, const TSet<TString>& transitionIds, TTransitions& transitions, NDrive::TEntitySession& session) const {
        auto incidentInstance = GetIncident(incidentId, session);
        if (!incidentInstance) {
            return false;
        }
        auto result = GetAvailableTransitions(*incidentInstance, transitionIds, session);
        if (!result) {
            return false;
        }
        transitions = std::move(*result);
        return true;
    }

    TIncidentsManager::TOptionalTransitions TIncidentsManager::GetAvailableTransitions(const TIncidentData& incidentInstance, NDrive::TEntitySession& session) const {
        return GetAvailableTransitions(incidentInstance, {}, session);
    }

    TIncidentsManager::TOptionalTransitions TIncidentsManager::GetAvailableTransitions(const TIncidentData& incidentInstance, const TSet<TString>& transitionIds, NDrive::TEntitySession& session) const {
        TIncidentStateContext context(&Server, incidentInstance, {});
        return StateManager.GetAvailableTransitions(context, session, transitionIds);
    }

    TIncidentsManager::TOptionalIncident TIncidentsManager::PerformTransition(const TString& incidentId, const EIncidentTransition transitionId, const NJson::TJsonValue& transitionData, const TString& originatorId, NDrive::TEntitySession& session) const {
        auto incidentInstance = GetIncident(incidentId, session);
        return PerformTransition(incidentInstance, transitionId, transitionData, originatorId, session);
    }

    TIncidentsManager::TOptionalIncident TIncidentsManager::PerformTransition(const TOptionalIncident& incidentInstance, const EIncidentTransition transitionId, const NJson::TJsonValue& transitionData, const TString& originatorId, NDrive::TEntitySession& session, const bool quiet /* = false */) const {
        if (!incidentInstance) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "empry incident data", EDriveSessionResult::InternalError);
            return {};
        }

        TIncidentStateContext context(&Server, incidentInstance, originatorId);
        TMessagesCollector errors;

        auto transitionPtr = ConstructTransition(transitionId, transitionData, context, errors);
        if (!transitionPtr) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "error initializing transition context: " + errors.GetStringReport(), EDriveSessionResult::InternalError);
            return {};
        }

        if (auto isApplicable = transitionPtr->IsApplicable(context, session)) {
            if (!*isApplicable) {
                if (quiet) {
                    return incidentInstance;
                }
                session.SetErrorInfo(TIncidentData::GetTableName(), "transition is not allowed", EDriveSessionResult::IncorrectRequest);
                return {};
            }
        } else {
            return {};
        }

        if (!transitionPtr->Perform(context, session)) {
            return {};
        }

        return IncidentsStorage.ApplyTransition(context.GetInstanceRef(), originatorId, session);
    }

    TIncidentsManager::TOptionalIncident TIncidentsManager::GetIncident(const TString& incidentId, NDrive::TEntitySession& session) const {
        TOptionalIncident instance = IncidentsStorage.GetObject(incidentId, Now());  // request immediate update; incident storage related methods to be reworked
        if (!instance) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "error fetching incident", EDriveSessionResult::IncorrectRequest);
        }
        return instance;
    }

    TIncidentsManager::TOptionalIncidents TIncidentsManager::GetIncidents(const TIncidentFilterGroup& filters, NDrive::TEntitySession& session) const {
        TVector<TIncidentData> instances;
        const auto actor = [&filters, &instances](const TString& /* objectId */, const TIncidentData& entry) {
            if (filters.Match(entry)) {
                instances.push_back(entry);
            }
        };
        if (!IncidentsStorage.ForObjectsMap(actor, Now())) {  // request immediate update; incident storage related methods to be reworked
            session.SetErrorInfo(TIncidentData::GetTableName(), "error fetching incidents", EDriveSessionResult::IncorrectRequest);
            return {};
        }
        return instances;
    }

    TIncidentsManager::TOptionalIncidentState TIncidentsManager::GetPreviousIncidentState(const TString& incidentId, NDrive::TEntitySession& session) const {
        return IncidentsStorage.GetPreviousState(incidentId, session);
    }
}
