#pragma once

#include "config.h"
#include "constants.h"
#include "database.h"
#include "filters.h"
#include "states.h"

#include <drive/backend/database/transaction/tx.h>

namespace NDrive {
    class TIncidentsManager: public TDatabaseSessionConstructor {
    public:
        using TOptionalIncident = TMaybe<TIncidentData>;
        using TOptionalIncidents = TMaybe<TVector<TIncidentData>>;
        using TOptionalIncidentState = TIncidentsStorage::TOptionalRecordState;

        using TTransitionPtr = TIncidentStateManager::TTransitionPtr;
        using TTransitions = TIncidentStateManager::TTransitions;
        using TOptionalTransitions = TMaybe<TTransitions>;

        TIncidentsManager(const TIncidentsManagerConfig& config, const IHistoryContext& context, const NDrive::IServer& server);
        ~TIncidentsManager();

        void Start();
        void Stop();

        TOptionalIncident CreateIncident(const TIncidentData& data, const TString& orginatorId, NDrive::TEntitySession& session) const;
        TOptionalIncident UpdateIncident(const TIncidentData& data, const TString& orginatorId, NDrive::TEntitySession& session) const;

        TTransitionPtr ConstructTransition(const EIncidentTransition transitionId, const NJson::TJsonValue& transitionData, const TIncidentStateContext& context, TMessagesCollector& errors) const;

        bool CheckTransitionAllowed(const TString& incidentId, const TString& transitionId, NDrive::TEntitySession& session) const;

        bool GetAvailableTransitions(const TString& incidentId, TTransitions& transitions, NDrive::TEntitySession& session) const;
        bool GetAvailableTransitions(const TString& incidentId, const TSet<TString>& transitionIds, TTransitions& transitions, NDrive::TEntitySession& session) const;

        TOptionalTransitions GetAvailableTransitions(const TIncidentData& incidentInstance, NDrive::TEntitySession& session) const;
        TOptionalTransitions GetAvailableTransitions(const TIncidentData& incidentInstance, const TSet<TString>& transitionIds, NDrive::TEntitySession& session) const;

        TOptionalIncident PerformTransition(const TString& incidentId, const EIncidentTransition transitionId, const NJson::TJsonValue& transitionData, const TString& originatorId, NDrive::TEntitySession& session) const;
        TOptionalIncident PerformTransition(const TOptionalIncident& incidentInstance, const EIncidentTransition transitionId, const NJson::TJsonValue& transitionData, const TString& originatorId, NDrive::TEntitySession& session, const bool quiet = false) const;

        TOptionalIncident GetIncident(const TString& incidentId, NDrive::TEntitySession& session) const;
        TOptionalIncidents GetIncidents(const TIncidentFilterGroup& filters, NDrive::TEntitySession& session) const;

        TOptionalIncidentState GetPreviousIncidentState(const TString& incidentId, NDrive::TEntitySession& session) const;

    private:
        TIncidentsManagerConfig Config;

        const NDrive::IServer& Server;

        TIncidentsStorage IncidentsStorage;
        TIncidentStateManager StateManager;
    };
}
