#include "catalogue_entity_base.h"

#include <drive/backend/abstract/base.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/json/field.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/ptr.h>

DECLARE_FIELDS_JSON_SERIALIZER(NDrive::NRenins::ICatalogueEntry::TRawEntry);

namespace NDrive::NRenins {
    const ISettings* ICatalogueEntry::GetSettings(const IServerBase* server) const {
        if (!server || !server->HasSettings()) {
            return nullptr;
        }
        return &(server->GetSettings());
    }

    bool ICatalogueEntry::StoreSettingValue(const IServerBase* server, const TString& userId, bool checkExistingEquals) const {
        TString key = GetSettingKey();
        TString update = SerializeToJson().GetStringRobust();
        if (checkExistingEquals) {
            TMaybe<TString> existing = Yensured(GetSettings(server))->GetValue<TString>(key);
            if (existing.GetOrElse({}) == update) {
                return true;
            }
        }
        return Yensured(GetSettings(server))->SetValue(key, update, userId);
    }

    bool ICatalogueEntry::LoadSettingValue(const IServerBase* server) {
        auto rawValue = Yensured(GetSettings(server))->GetJsonValue(GetSettingKey());
        return DeserializeFromJson(rawValue);
    }

    bool ICatalogueEntry::IsRequestable() const {
        return true;
    }

    bool ICatalogueEntry::Validate(TMessagesCollector& /* errors */) const {
        return true;
    }

    NJson::TJsonValue ICatalogueEntry::MakeRequestData(const TReninsClaimClientConfig& config) const {
        return (
            NJson::TMapBuilder
            ("Login", config.GetLogin())
            ("Password", config.GetPassword())
            ("ClaimType", ::ToString(GetClaimType()))
            ("CatalogCode", ::ToString(GetType()))
        );
    }

    bool ICatalogueEntry::ParseResponseRawEntries(const NJson::TJsonValue& data, TRawEntries& rawEntries) const {
        auto catalogues = data["Payload"]["Catalog"];
        if (!catalogues.IsArray() || catalogues.GetArray().size() != 1) {
            return false;
        }
        auto entries = catalogues.GetArray().front()["CatalogEntry"];
        if (!entries.IsArray()) {
            return false;
        }
        return NJson::TryFromJson(entries, rawEntries);
    }

    bool ISeriesCatalogueEntry::ParseResponse(const NJson::TJsonValue& data) {
        Values.clear();

        TRawEntries rawEntries;
        if (!ParseResponseRawEntries(data, rawEntries)) {
            return false;
        }

        for (auto&& rawEntry: rawEntries) {
            if ((!!rawEntry.Value ^ !!rawEntry.Description) && (!rawEntry.Low)) {
                Values.emplace((rawEntry.Value) ? rawEntry.Value : rawEntry.Description);
            } else {
                return false;
            }
        }
        return true;
    }

    NJson::TJsonValue ISeriesCatalogueEntry::SerializeToJson() const {
        return NJson::ToJson(Values);
    }

    bool ISeriesCatalogueEntry::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::TryFromJson(data, Values);
    }

    ISeriesCatalogueEntry::operator TFSVariants::TCompoundVariants() const {
        TFSVariants::TCompoundVariants variants;
        for (auto&& value: Values) {
            variants.emplace(TFSVariants::TCompoundVariant(value));
        }
        return variants;
    }

    bool IMappedCatalogueEntry::ParseResponse(const NJson::TJsonValue& data) {
        Values.clear();

        TRawEntries rawEntries;
        if (!ParseResponseRawEntries(data, rawEntries)) {
            return false;
        }

        for (auto&& rawEntry: rawEntries) {
            if ((!!rawEntry.Value && !!rawEntry.Description) && (!rawEntry.Low)) {
                Values.emplace(rawEntry.Value, rawEntry.Description);
            } else {
                return false;
            }
        }
        return true;
    }

    NJson::TJsonValue IMappedCatalogueEntry::SerializeToJson() const {
        return NJson::ToJson(NJson::Dictionary(Values));
    }

    bool IMappedCatalogueEntry::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::TryFromJson(data, NJson::Dictionary(Values));
    }

    IMappedCatalogueEntry::operator TFSVariants::TCompoundVariants() const {
        TFSVariants::TCompoundVariants variants;
        for (auto&& [name, text]: Values) {
            variants.emplace(TFSVariants::TCompoundVariant(name, text));
        }
        return variants;
    }
}
