#pragma once

#include "config.h"
#include "constants.h"
#include "entity_base.h"

#include <drive/backend/abstract/settings.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/json/json_value.h>
#include <library/cpp/object_factory/object_factory.h>

#include <rtline/util/types/accessor.h>
#include <rtline/util/types/field.h>

#include <util/generic/map.h>
#include <util/generic/set.h>

class IServerBase;

namespace NDrive::NRenins {
    #define FACTORY_RENINS_CATALOGUE_ENTRY(ClassName, EnumValue)             \
    public:                                                                  \
        using TRegistrator = TFactory::TRegistrator<ClassName>;              \
        static auto GetTypeName() {                                          \
            return EnumValue;                                                \
        }                                                                    \
        virtual TString GetType() const override {                           \
            return ::ToString(EnumValue);                                    \
        }                                                                    \
    private:                                                                 \
        static TRegistrator Registrator;

    class ICatalogueEntry: public IRequestEntry {
    public:
        struct TRawEntry {
            TString SubType;
            TString Value;
            TString Comments;
            TString Low;
            TString Description;
            TString Name;

            DECLARE_FIELDS(
                Field(SubType, "SubType"),
                Field(Value, "Value"),
                Field(Comments, "Comments"),
                Field(Low, "Low"),
                Field(Description, "Description"),
                Field(Name, "Name")
            );
        };

        using TRawEntries = TVector<TRawEntry>;

    public:
        virtual TString GetClaimType() const = 0;
        virtual TString GetType() const = 0;

        virtual TString GetSettingKey() const = 0;

        bool StoreSettingValue(const IServerBase* server, const TString& userId, bool checkExistingEquals = true) const;
        bool LoadSettingValue(const IServerBase* server);

        virtual bool IsRequestable() const;
        virtual bool Validate(TMessagesCollector& errors) const override;

        virtual NJson::TJsonValue MakeRequestData(const TReninsClaimClientConfig& config) const override;

        virtual operator TFSVariants::TCompoundVariants() const = 0;

    protected:
        const ISettings* GetSettings(const IServerBase* server) const;

        bool ParseResponseRawEntries(const NJson::TJsonValue& data, TRawEntries& rawEntries) const;
    };

    class ISeriesCatalogueEntry: virtual public ICatalogueEntry {
    public:
        virtual bool ParseResponse(const NJson::TJsonValue& data) override;

        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& data) override;

        virtual operator TFSVariants::TCompoundVariants() const override;

    private:
        R_READONLY(TSet<TString>, Values);
    };

    class IMappedCatalogueEntry: virtual public ICatalogueEntry {
    public:
        using TMapping = TMap<TString, TString>;

        virtual bool ParseResponse(const NJson::TJsonValue& data) override;

        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& data) override;

        virtual operator TFSVariants::TCompoundVariants() const override;

    private:
        R_READONLY(TMapping, Values);
    };
}
