#include "catalogue_entity_kasko.h"

#include "json_adapters.h"

#include <drive/backend/abstract/base.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/cast.h>
#include <rtline/util/algorithm/type_traits.h>

#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/string/join.h>

#include <tuple>

namespace NJson {
    template <>
    TJsonValue ToJson(const NDrive::NRenins::TKaskoLossCodeInfo& object) {
        return object.SerializeToJson();
    }

    template <>
    bool TryFromJson(const NJson::TJsonValue& value, NDrive::NRenins::TKaskoLossCodeInfo& result) {
        return result.DeserializeFromJson(value);
    }

    template <>
    TJsonValue ToJson(const NDrive::NRenins::TKaskoSTOAAddressInfo& object) {
        return object.SerializeToJson();
    }

    template <>
    bool TryFromJson(const NJson::TJsonValue& value, NDrive::NRenins::TKaskoSTOAAddressInfo& result) {
        return result.DeserializeFromJson(value);
    }

    template <>
    TJsonValue ToJson(const NDrive::NRenins::TKaskoTerritoryInfo& object) {
        return object.SerializeToJson();
    }

    template <>
    bool TryFromJson(const NJson::TJsonValue& value, NDrive::NRenins::TKaskoTerritoryInfo& result) {
        return result.DeserializeFromJson(value);
    }
}

namespace NDrive::NRenins {
    TKaskoCompensationTypeEntry::TRegistrator TKaskoCompensationTypeEntry::Registrator;
    TKaskoDamageListEntry::TRegistrator TKaskoDamageListEntry::Registrator;
    TKaskoDocumentTypeEntry::TRegistrator TKaskoDocumentTypeEntry::Registrator;
    TKaskoGuiltyEntry::TRegistrator TKaskoGuiltyEntry::Registrator;
    TKaskoLocationTypeEntry::TRegistrator TKaskoLocationTypeEntry::Registrator;
    TKaskoLossCodeEntry::TRegistrator TKaskoLossCodeEntry::Registrator;
    TKaskoPartnerRoleEntry::TRegistrator TKaskoPartnerRoleEntry::Registrator;
    TKaskoRegistrationTypeEntry::TRegistrator TKaskoRegistrationTypeEntry::Registrator;
    TKaskoRoleEntry::TRegistrator TKaskoRoleEntry::Registrator;
    TKaskoSiteEntry::TRegistrator TKaskoSiteEntry::Registrator;
    TKaskoSTOAAddressEntry::TRegistrator TKaskoSTOAAddressEntry::Registrator;
    TKaskoTerritoryEntry::TRegistrator TKaskoTerritoryEntry::Registrator;
    TKaskoVehicleColorEntry::TRegistrator TKaskoVehicleColorEntry::Registrator;

    TString TKaskoLossCodeInfo::GetDescription() const {
        return TStringBuilder() << Category << ": " << Subcategory;
    }

    bool TKaskoLossCodeInfo::operator < (const TKaskoLossCodeInfo& other) const {
        return Subcategory < other.Subcategory;
    }

    NDrive::TScheme TKaskoLossCodeInfo::GetScheme(const IServerBase& /* server */) {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("category", "Category");
        scheme.Add<TFSString>("subcategory", "Subcategory").SetRequired(true);
        return scheme;
    }

    NJson::TJsonValue TKaskoLossCodeInfo::MakeRequestData(const TReninsClaimClientConfig& /* config */) const {
        return NJson::TMapBuilder
            ("LossCode1", Category)
            ("LossCode2", Subcategory);
    }

    bool TKaskoLossCodeInfo::ParseResponse(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["LossCode1"], Category) &&
               NJson::ParseField(data["LossCode2"], Subcategory);
    }

    NJson::TJsonValue TKaskoLossCodeInfo::SerializeToJson() const {
        return NJson::TMapBuilder("category", Category)("subcategory", Subcategory);
    }

    bool TKaskoLossCodeInfo::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["category"], Category) &&
               NJson::ParseField(data["subcategory"], Subcategory, /* required = */ true);
    }

    TKaskoLossCodeInfo::operator TFSVariants::TCompoundVariant() const {
        return TFSVariants::TCompoundVariant(GetSubcategory(), GetDescription());
    }

    NDrive::TScheme TKaskoSTOAAddressInfo::GetScheme(const IServerBase& /* server */) {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("id", "Id").SetRequired(true);
        scheme.Add<TFSString>("address", "Address");
        scheme.Add<TFSString>("kladr", "KLADR");
        scheme.Add<TFSString>("okato_code", "Okato code");
        return scheme;
    }

    NJson::TJsonValue TKaskoSTOAAddressInfo::MakeRequestData(const TReninsClaimClientConfig& /* config */) const {
        return NJson::TMapBuilder("AddressId", Id);
    }

    bool TKaskoSTOAAddressInfo::ParseResponse(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["AddressId"], Id) &&
               NJson::ParseField(data["Address"], Address) &&
               NJson::ParseField(data["Id"], Id) &&
               NJson::ParseField(data["KLADR"], KLADR) &&
               NJson::ParseField(data["OKATO"], OkatoCode) &&
               NJson::ParseField(data["X_CLAIM_ADDRESS"], Address) &&
               NJson::ParseField(data["X_KLADR"], KLADR) &&
               NJson::ParseField(data["x_ren_okato_code"], OkatoCode);
    }

    NJson::TJsonValue TKaskoSTOAAddressInfo::SerializeToJson() const {
        return NJson::TMapBuilder("id", Id)("address", Address)("kladr", KLADR)("okato_code", OkatoCode);
    }

    bool TKaskoSTOAAddressInfo::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["id"], Id, /* required = */ true) &&
               NJson::ParseField(data["address"], Address) &&
               NJson::ParseField(data["kladr"], KLADR) &&
               NJson::ParseField(data["okato_code"], OkatoCode);
    }

    TKaskoSTOAAddressInfo::operator TFSVariants::TCompoundVariant() const {
        return TFSVariants::TCompoundVariant(Id, Address);
    }

    TString TKaskoTerritoryInfo::GetDescription() const {
        return TStringBuilder() << City << " (" << Area << ")";
    }

    NDrive::TScheme TKaskoTerritoryInfo::GetScheme(const IServerBase& /* server */) {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("city", "City").SetRequired(true);
        scheme.Add<TFSString>("area", "Area");
        return scheme;
    }

    NJson::TJsonValue TKaskoTerritoryInfo::MakeRequestData(const TReninsClaimClientConfig& /* config */) const {
        return NJson::TMapBuilder
            ("City", City);
    }

    bool TKaskoTerritoryInfo::ParseResponse(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["City"], City);
    }

    NJson::TJsonValue TKaskoTerritoryInfo::SerializeToJson() const {
        return NJson::TMapBuilder("city", City)("area", Area);
    }

    bool TKaskoTerritoryInfo::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["city"], City, /* required = */ true) &&
               NJson::ParseField(data["area"], Area);
    }

    TKaskoTerritoryInfo::operator TFSVariants::TCompoundVariant() const {
        return TFSVariants::TCompoundVariant(GetCity(), GetDescription());
    }

    EClaimType IKaskoCatalogueEntry::GetClaimTypeName() {
        return EClaimType::Kasko;
    }

    TString IKaskoCatalogueEntry::GetSettingPrefix() {
        return JoinSeq(".", {KaskoSettingPrefix, TStringBuf("catalogue_values")});
    }

    TString IKaskoCatalogueEntry::GetClaimType() const {
        return ::ToString(GetClaimTypeName());
    }

    TString IKaskoCatalogueEntry::GetSettingKey() const {
        return JoinSeq(".", {GetSettingPrefix(), GetType()});
    }

    TMaybe<TKaskoLossCodeInfo> TKaskoLossCodeEntry::Restore(const IServerBase* server, const NJson::TJsonValue& data) {
        TKaskoLossCodeInfo value;
        if (value.DeserializeFromJson(data)) {
            return value;
        }

        TString subcategory;
        TKaskoLossCodeEntry entry;
        if (server && NJson::TryFromJson(data, subcategory) && entry.LoadSettingValue(server)) {
            auto it = FindIf(entry.GetValues(), [&subcategory](const auto& lossCodeInfo){ return lossCodeInfo.GetSubcategory() == subcategory; });
            if (it != entry.GetValues().end()) {
                return *it;
            }
        }

        return {};
    }

    NJson::TJsonValue TKaskoLossCodeEntry::MakeRequestData(const TReninsClaimClientConfig& /* config */) const {
        return {};
    }

    bool TKaskoLossCodeEntry::ParseResponse(const NJson::TJsonValue& /* data */) {
        return false;
    }

    NJson::TJsonValue TKaskoLossCodeEntry::SerializeToJson() const {
        return NJson::ToJson(Values);
    }

    bool TKaskoLossCodeEntry::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::TryFromJson(data, Values);
    }

    TKaskoLossCodeEntry::operator TFSVariants::TCompoundVariants() const {
        TFSVariants::TCompoundVariants variants;
        for (auto&& value: Values) {
            variants.emplace(static_cast<TFSVariants::TCompoundVariant>(value));
        }
        return variants;
    }

    TMaybe<TKaskoSTOAAddressInfo> TKaskoSTOAAddressEntry::Restore(const IServerBase* server, const NJson::TJsonValue& data) {
        TKaskoSTOAAddressInfo value;
        if (value.DeserializeFromJson(data)) {
            return value;
        }

        TString addressId;
        TKaskoSTOAAddressEntry entry;
        if (server && NJson::TryFromJson(data, addressId) && entry.LoadSettingValue(server)) {
            auto valuePtr = entry.GetAddresses().FindPtr(addressId);
            if (valuePtr) {
                return *valuePtr;
            }
        }

        return {};
    }

    bool TKaskoSTOAAddressEntry::ParseResponse(const NJson::TJsonValue& data) {
        const auto& records = data["RENGetStoaList"]["PersonalAccount"]["ListOfSTOA"]["STOA"];
        if (records.IsDefined()) {
            if (!records.IsArray()) {
                return false;
            }
            for (auto&& record : records.GetArray()) {
                TKaskoSTOAAddressInfo info;
                if (!info.ParseResponse(record)) {
                    return false;
                }
                auto id = info.GetId();
                Addresses.emplace(std::move(id), std::move(info));
            }
        }
        return true;
    }

    NJson::TJsonValue TKaskoSTOAAddressEntry::SerializeToJson() const {
        return NJson::ToJson(NJson::InnerKeyValue(Addresses, "id"));
    }

    bool TKaskoSTOAAddressEntry::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::TryFromJson(data, NJson::InnerKeyValue(Addresses, "id"));
    }

    TKaskoSTOAAddressEntry::operator TFSVariants::TCompoundVariants() const {
        TFSVariants::TCompoundVariants variants;
        for (auto&& [addressId, value]: Addresses) {
            variants.emplace(static_cast<TFSVariants::TCompoundVariant>(value));
        }
        return variants;
    }

    TMaybe<TKaskoTerritoryInfo> TKaskoTerritoryEntry::Restore(const IServerBase* server, const NJson::TJsonValue& data) {
        TKaskoTerritoryInfo value;
        if (value.DeserializeFromJson(data)) {
            return value;
        }

        TString city;
        TKaskoTerritoryEntry entry;
        if (server && NJson::TryFromJson(data, city) && entry.LoadSettingValue(server)) {
            // NB. Find first value - maybe multiple cities with different areas exist
            auto it = FindIf(entry.GetTerritories(), [&city](const auto& territory){ return territory.GetCity() == city; });
            if (it != entry.GetTerritories().end()) {
                return *it;
            }
        }

        return {};
    }

    bool TKaskoTerritoryEntry::ParseResponse(const NJson::TJsonValue& data) {
        Territories.clear();

        TRawEntries rawEntries;
        if (!ParseResponseRawEntries(data, rawEntries)) {
            return false;
        }

        for (auto&& rawEntry: rawEntries) {
            TKaskoTerritoryInfo territory;
            if (rawEntry.Name) {
                territory.SetCity(rawEntry.Name).SetArea(Coalesce(rawEntry.Description, rawEntry.Name));
                Territories.push_back(std::move(territory));
            } else {
                return false;
            }
        }
        return true;
    }

    NJson::TJsonValue TKaskoTerritoryEntry::SerializeToJson() const {
        return NJson::ToJson(Territories);
    }

    bool TKaskoTerritoryEntry::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::TryFromJson(data, Territories);
    }

    TKaskoTerritoryEntry::operator TFSVariants::TCompoundVariants() const {
        TFSVariants::TCompoundVariants variants;
        for (const auto& value: Territories) {
            variants.emplace(static_cast<TFSVariants::TCompoundVariant>(value));
        }
        return variants;
    }
}
