#pragma once

#include "catalogue_entity_base.h"
#include "entity_base.h"

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/util/types/accessor.h>

class IServerBase;

namespace NDrive::NRenins {
    class TKaskoLossCodeInfo: public IRequestInfoEntry {
    public:
        TString GetDescription() const;

        bool operator < (const TKaskoLossCodeInfo& other) const;

        static NDrive::TScheme GetScheme(const IServerBase& server);

        virtual NJson::TJsonValue MakeRequestData(const TReninsClaimClientConfig& config) const override;
        virtual bool ParseResponse(const NJson::TJsonValue& data) override;

        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& data) override;

        operator TFSVariants::TCompoundVariant() const;

    private:
        R_FIELD(TString, Category);
        R_FIELD(TString, Subcategory);
    };

    class TKaskoSTOAAddressInfo: public IRequestInfoEntry {
    public:
        static NDrive::TScheme GetScheme(const IServerBase& server);

        virtual NJson::TJsonValue MakeRequestData(const TReninsClaimClientConfig& config) const override;
        virtual bool ParseResponse(const NJson::TJsonValue& data) override;

        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& data) override;

        operator TFSVariants::TCompoundVariant() const;

    private:
        R_FIELD(TString, Id);
        R_FIELD(TString, Address);
        R_FIELD(TString, KLADR);
        R_FIELD(TString, OkatoCode);
    };

    class TKaskoTerritoryInfo: public IRequestInfoEntry {
    public:
        TString GetDescription() const;

        static NDrive::TScheme GetScheme(const IServerBase& server);

        virtual NJson::TJsonValue MakeRequestData(const TReninsClaimClientConfig& config) const override;
        virtual bool ParseResponse(const NJson::TJsonValue& data) override;

        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& data) override;

        operator TFSVariants::TCompoundVariant() const;

    private:
        R_FIELD(TString, City);
        R_FIELD(TString, Area);
    };

    // catalogue entries

    class IKaskoCatalogueEntry: virtual public ICatalogueEntry {
    public:
        using TFactory = NObjectFactory::TParametrizedObjectFactory<IKaskoCatalogueEntry, EKaskoCatalogueType>;

        static EClaimType GetClaimTypeName();
        static TString GetSettingPrefix();

        virtual TString GetClaimType() const override;
        virtual TString GetSettingKey() const override;
    };

    class TKaskoCompensationTypeEntry: public IKaskoCatalogueEntry, public ISeriesCatalogueEntry {
        FACTORY_RENINS_CATALOGUE_ENTRY(TKaskoCompensationTypeEntry, EKaskoCatalogueType::CompensationType);
    };

    class TKaskoDamageListEntry: public IKaskoCatalogueEntry, public ISeriesCatalogueEntry {
        FACTORY_RENINS_CATALOGUE_ENTRY(TKaskoDamageListEntry, EKaskoCatalogueType::DamageList);
    };

    class TKaskoDocumentTypeEntry: public IKaskoCatalogueEntry, public IMappedCatalogueEntry {
        FACTORY_RENINS_CATALOGUE_ENTRY(TKaskoDocumentTypeEntry, EKaskoCatalogueType::DocumentType);
    };

    class TKaskoGuiltyEntry: public IKaskoCatalogueEntry, public ISeriesCatalogueEntry {
        FACTORY_RENINS_CATALOGUE_ENTRY(TKaskoGuiltyEntry, EKaskoCatalogueType::Guilty);
    };

    class TKaskoLocationTypeEntry: public IKaskoCatalogueEntry, public ISeriesCatalogueEntry {
        FACTORY_RENINS_CATALOGUE_ENTRY(TKaskoLocationTypeEntry, EKaskoCatalogueType::LocationType);
    };

    class TKaskoLossCodeEntry: public IKaskoCatalogueEntry {
        FACTORY_RENINS_CATALOGUE_ENTRY(TKaskoLossCodeEntry, EKaskoCatalogueType::LossCode);

    public:
        static TMaybe<TKaskoLossCodeInfo> Restore(const IServerBase* server, const NJson::TJsonValue& data);

        bool IsRequestable() const override {
            return false;  // contains constant values, cannot be updated
        }

        virtual NJson::TJsonValue MakeRequestData(const TReninsClaimClientConfig& config) const override;
        virtual bool ParseResponse(const NJson::TJsonValue& data) override;

        NJson::TJsonValue SerializeToJson() const override;
        bool DeserializeFromJson(const NJson::TJsonValue& data) override;

        virtual operator TFSVariants::TCompoundVariants() const override;

    private:
        R_FIELD(TSet<TKaskoLossCodeInfo>, Values);
    };

    class TKaskoPartnerRoleEntry: public IKaskoCatalogueEntry, public ISeriesCatalogueEntry {
        FACTORY_RENINS_CATALOGUE_ENTRY(TKaskoPartnerRoleEntry, EKaskoCatalogueType::PartnerRole);
    };

    class TKaskoRegistrationTypeEntry: public IKaskoCatalogueEntry, public ISeriesCatalogueEntry {
        FACTORY_RENINS_CATALOGUE_ENTRY(TKaskoRegistrationTypeEntry, EKaskoCatalogueType::RegistrationType);
    };

    class TKaskoRoleEntry: public IKaskoCatalogueEntry, public ISeriesCatalogueEntry {
        FACTORY_RENINS_CATALOGUE_ENTRY(TKaskoRoleEntry, EKaskoCatalogueType::Role);
    };

    class TKaskoSiteEntry: public IKaskoCatalogueEntry, public ISeriesCatalogueEntry {
        FACTORY_RENINS_CATALOGUE_ENTRY(TKaskoSiteEntry, EKaskoCatalogueType::Site);
    };

    class TKaskoSTOAAddressEntry: public IKaskoCatalogueEntry {
        using TAddressMapping = TMap<TString, TKaskoSTOAAddressInfo>;

        FACTORY_RENINS_CATALOGUE_ENTRY(TKaskoSTOAAddressEntry, EKaskoCatalogueType::STOAAddress);

    public:
        static TMaybe<TKaskoSTOAAddressInfo> Restore(const IServerBase* server, const NJson::TJsonValue& data);

        bool IsRequestable() const override {
            return false;  // currently cannot be updated or can be updated via another method
        }

        virtual bool ParseResponse(const NJson::TJsonValue& data) override;

        NJson::TJsonValue SerializeToJson() const override;
        bool DeserializeFromJson(const NJson::TJsonValue& data) override;

        virtual operator TFSVariants::TCompoundVariants() const override;

    private:
        R_FIELD(TAddressMapping, Addresses);
    };

    class TKaskoTerritoryEntry: public IKaskoCatalogueEntry {
        using TKaskoTerritories = TVector<TKaskoTerritoryInfo>;

        FACTORY_RENINS_CATALOGUE_ENTRY(TKaskoTerritoryEntry, EKaskoCatalogueType::Territory);

    public:
        static TMaybe<TKaskoTerritoryInfo> Restore(const IServerBase* server, const NJson::TJsonValue& data);

        virtual bool ParseResponse(const NJson::TJsonValue& data) override;

        NJson::TJsonValue SerializeToJson() const override;
        bool DeserializeFromJson(const NJson::TJsonValue& data) override;

        virtual operator TFSVariants::TCompoundVariants() const override;

    private:
        R_FIELD(TKaskoTerritories, Territories);
    };

    class TKaskoVehicleColorEntry: public IKaskoCatalogueEntry, public ISeriesCatalogueEntry {
        FACTORY_RENINS_CATALOGUE_ENTRY(TKaskoVehicleColorEntry, EKaskoCatalogueType::VehicleColor);
    };
}
