#include "claim_entity_kasko.h"

#include "json_adapters.h"
#include "scheme_adapters.h"

#include <drive/backend/abstract/base.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/merge.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/string/builder.h>

namespace {
    bool ParseOptionalReninsBoolean(const NJson::TJsonValue& data, TStringBuf fieldName, TMaybe<bool>& target) {
        if (data.Has(fieldName)) {
            bool value;
            if (!NJson::ParseField(data[fieldName], NJson::ReninsBoolean(value))) {
                return false;
            }
            target = value;
        }
        return true;
    }

    void InsertOptionalReninsBoolean(NJson::TJsonValue& result, TStringBuf fieldName, const TMaybe<bool>& value) {
        if (value) {
            NJson::InsertField(result, fieldName, NJson::ToJson(NJson::ReninsBoolean(*value)));
        }
    }

    template <typename T>
    bool SetEntrySchemeDefaults(NDrive::TScheme& scheme, const TMaybe<T>& entry, const TSet<TString>& excludedFields = {}) {
        if (!entry) {
            return true;
        }

        auto entryData = entry->SerializeToJson();
        for (auto&& [name, value]: entryData.GetMap()) {
            if (excludedFields.contains(name)) {
                continue;
            }

            auto defaultValueSchemePtr = std::dynamic_pointer_cast<NDrive::IBaseDefaultSchemeElement>(scheme.Get(name));
            if (defaultValueSchemePtr && !defaultValueSchemePtr->SetDefaultValueView(value)) {
                ERROR_LOG << "Error setting up a default value for " << name << " field" << Endl;
            }
        }

        return true;
    }
}

namespace NJson {
    template <>
    TJsonValue ToJson(const NDrive::NRenins::TKaskoParticipant& object) {
        return object.SerializeToJson();
    }

    template <>
    bool TryFromJson(const NJson::TJsonValue& value, NDrive::NRenins::TKaskoParticipant& result) {
        return result.DeserializeFromJson(value);
    }
}

namespace NDrive::NRenins {
    NDrive::TScheme TKaskoGeneralClaimInfo::GetScheme(const IServerBase& /* server */, const TMaybe<TKaskoGeneralClaimInfo>& entry) {
        NDrive::TScheme scheme;

        scheme.Add<TFSVariants>("partner_role", "Роль партнера").SetCompoundVariants(TReninsCatalogueEntryAdapter<TKaskoPartnerRoleEntry>::Cast()).SetRequired(true);
        scheme.Add<TFSBoolean>("do_register_claim", "Полноценный убыток").SetRequired(true);

        SetEntrySchemeDefaults(scheme, entry);

        return scheme;
    }

    NJson::TJsonValue TKaskoGeneralClaimInfo::MakeRequestData(const TReninsClaimClientConfig& config) const {
        return NJson::TMapBuilder
            ("Login", config.GetLogin())
            ("Password", config.GetPassword())
            ("PartnerRole", PartnerRole)
            ("RegClaim", NJson::ToJson(NJson::ReninsBoolean(DoRegisterClaim)));
    }

    bool TKaskoGeneralClaimInfo::ParseResponse(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["PartnerRole"], PartnerRole) &&
               NJson::ParseField(data["RegClaim"], NJson::ReninsBoolean(DoRegisterClaim));
    }

    NJson::TJsonValue TKaskoGeneralClaimInfo::SerializeToJson() const {
        return NJson::TMapBuilder
            ("partner_role", PartnerRole)
            ("do_register_claim", DoRegisterClaim);
    }

    bool TKaskoGeneralClaimInfo::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["partner_role"], PartnerRole) &&
               NJson::ParseField(data["do_register_claim"], DoRegisterClaim);
    }

    NDrive::TScheme TKaskoClaim::GetScheme(const IServerBase& /* server */, const TMaybe<TKaskoClaim>& entry) {
        NDrive::TScheme scheme;

        scheme.Add<TFSString>("policy_number", "Номер полиса");  // required depends
        scheme.Add<TFSString>("claim_number", "Номер убытка");
        scheme.Add<TFSNumeric>("incident_timestamp", "Дата и время происшествия").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(true);

        auto& lossCodeScheme = scheme.Add<TFSVariants>("loss_code_info", "Что случилось");
        lossCodeScheme.SetCompoundVariants(TReninsCatalogueEntryAdapter<TKaskoLossCodeEntry>::Cast()).SetRequired(true);
        if (entry) {
            lossCodeScheme.SetDefault(entry->GetLossCodeInfo().GetSubcategory());
        }

        scheme.Add<TFSVariants>("location", "Город заявления").SetCompoundVariants(TReninsCatalogueEntryAdapter<TKaskoSiteEntry>::Cast()).SetRequired(true);
        scheme.Add<TFSVariants>("responsible_person", "Виновник ДТП").SetCompoundVariants(TReninsCatalogueEntryAdapter<TKaskoGuiltyEntry>::Cast()).SetRequired(true);
        scheme.Add<TFSVariants>("place_type", "Тип места происшествия").SetCompoundVariants(TReninsCatalogueEntryAdapter<TKaskoLocationTypeEntry>::Cast()).SetRequired(true);
        scheme.Add<TFSVariants>("town", "Город").SetCompoundVariants(TReninsCatalogueEntryAdapter<TKaskoTerritoryEntry>::Cast()).SetRequired(true);
        scheme.Add<TFSString>("address", "Адрес").SetMaxLength(100).SetRequired(true);
        scheme.Add<TFSNumeric>("notification_timestamp", "Дата и время уведомления").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(true);
        scheme.Add<TFSNumeric>("registration_timestamp", "Дата и время регистрации").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(true);
        scheme.Add<TFSVariants>("registration_type", "Регистрация").SetCompoundVariants(TReninsCatalogueEntryAdapter<TKaskoRegistrationTypeEntry>::Cast()).SetRequired(true);
        scheme.Add<TFSBoolean>("is_europrotocol", "Европротокол");
        scheme.Add<TFSNumeric>("inquiry_date", "Дата подачи заявления").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(true);
        scheme.Add<TFSBoolean>("has_injured_persons", "Есть ли травмированные").SetRequired(true);
        scheme.Add<TFSBoolean>("has_lost_persons", "Есть ли жертвы").SetRequired(true);
        scheme.Add<TFSBoolean>("has_multiple_damage_faces", "Повреждено имущество 3-х лиц");
        scheme.Add<TFSBoolean>("on_the_move", "ТС находилось в движении");
        scheme.Add<TFSVariants>("regulation_type", "Форма возмещения").SetCompoundVariants(TReninsCatalogueEntryAdapter<TKaskoCompensationTypeEntry>::Cast()).SetRequired(true);
        scheme.Add<TFSString>("additional_info", "Доп. информация").SetMaxLength(2000);

        SetEntrySchemeDefaults(scheme, entry, { "loss_code_info" });

        return scheme;
    }

    NJson::TJsonValue TKaskoClaim::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "policy_number", PolicyNumber);
        NJson::InsertNonNull(result, "claim_number", ClaimNumber);
        NJson::InsertField(result, "incident_timestamp", NJson::Seconds(IncidentTimestamp));
        NJson::InsertField(result, "loss_code_info", LossCodeInfo.SerializeToJson());
        NJson::InsertField(result, "location", Location);
        NJson::InsertField(result, "responsible_person", ResponsiblePerson);
        NJson::InsertField(result, "place_type", PlaceType);
        NJson::InsertField(result, "town", Town);
        NJson::InsertField(result, "address", Address);
        NJson::InsertField(result, "notification_timestamp", NJson::Seconds(NotificationTimestamp));
        NJson::InsertField(result, "registration_timestamp", NJson::Seconds(RegistrationTimestamp));
        NJson::InsertField(result, "registration_type", RegistrationType);
        NJson::InsertNonNull(result, "is_europrotocol", IsEuroprotocol);
        NJson::InsertField(result, "inquiry_date", NJson::Seconds(InquiryDate));
        NJson::InsertField(result, "has_injured_persons", HasInjuredPersons);
        NJson::InsertField(result, "has_lost_persons", HasLostPersons);
        NJson::InsertNonNull(result, "has_multiple_damage_faces", HasMultipleDamageFaces);
        NJson::InsertField(result, "on_the_move", OnTheMove);
        NJson::InsertField(result, "regulation_type", RegulationType);
        NJson::InsertNonNull(result, "additional_info", AdditionalInfo);
        return result;
    }

    bool TKaskoClaim::DeserializeFromJson(const NJson::TJsonValue& data) {
        const auto& lossCodeInfoJson = data["loss_code_info"];
        if (lossCodeInfoJson.IsDefined() && !(lossCodeInfoJson.IsString() && lossCodeInfoJson.GetString().empty())) {
            auto lossCodeInfo = TKaskoLossCodeEntry::Restore(NDrive::HasServer() ? &NDrive::GetServer() : nullptr, lossCodeInfoJson);
            if (!lossCodeInfo) {
                return false;
            }
            LossCodeInfo = std::move(*lossCodeInfo);
        }
        return NJson::ParseField(data["policy_number"], PolicyNumber) &&
               NJson::ParseField(data["claim_number"], ClaimNumber) &&
               NJson::ParseField(data["incident_timestamp"], NJson::Seconds(IncidentTimestamp)) &&
               NJson::ParseField(data["location"], Location) &&
               NJson::ParseField(data["responsible_person"], ResponsiblePerson) &&
               NJson::ParseField(data["place_type"], PlaceType) &&
               NJson::ParseField(data["town"], Town) &&
               NJson::ParseField(data["address"], Address) &&
               NJson::ParseField(data["notification_timestamp"], NJson::Seconds(NotificationTimestamp)) &&
               NJson::ParseField(data["registration_timestamp"], NJson::Seconds(RegistrationTimestamp)) &&
               NJson::ParseField(data["registration_type"], RegistrationType) &&
               NJson::ParseField(data["is_europrotocol"], IsEuroprotocol) &&
               NJson::ParseField(data["inquiry_date"], NJson::Seconds(InquiryDate)) &&
               NJson::ParseField(data["has_injured_persons"], HasInjuredPersons) &&
               NJson::ParseField(data["has_lost_persons"], HasLostPersons) &&
               NJson::ParseField(data["has_multiple_damage_faces"], HasMultipleDamageFaces) &&
               NJson::ParseField(data["on_the_move"], OnTheMove) &&
               NJson::ParseField(data["regulation_type"], RegulationType) &&
               NJson::ParseField(data["additional_info"], AdditionalInfo);
    }

    NJson::TJsonValue TKaskoClaim::MakeRequestData(const TReninsClaimClientConfig& config) const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "PolicyNumber", PolicyNumber);
        NJson::InsertNonNull(result, "ClaimNumber", ClaimNumber);
        NJson::InsertField(result, "IncidentDate", NJson::ToJson(NJson::ReninsDateTime(IncidentTimestamp)));
        NJson::MergeJson(LossCodeInfo.MakeRequestData(config), result);
        NJson::InsertField(result, "Location", Location);
        NJson::InsertField(result, "IncidentRespPerson", ResponsiblePerson);
        NJson::InsertField(result, "IncidentPlaceType", PlaceType);
        NJson::InsertField(result, "IncidentTown", Town);
        NJson::InsertField(result, "IncidentAddress", Address);
        NJson::InsertField(result, "NotificationDate", NJson::ToJson(NJson::ReninsDateTime(NotificationTimestamp)));
        NJson::InsertField(result, "RegistrationDate", NJson::ToJson(NJson::ReninsDateTime(RegistrationTimestamp)));
        NJson::InsertField(result, "Registration", RegistrationType);
        InsertOptionalReninsBoolean(result, "Europrotocol", IsEuroprotocol);
        NJson::InsertField(result, "InquiryDate", NJson::ToJson(NJson::ReninsDate(InquiryDate)));
        NJson::InsertField(result, "InjuredAnswer", NJson::ToJson(NJson::ReninsBoolean(HasInjuredPersons)));
        NJson::InsertField(result, "LostAnswer", NJson::ToJson(NJson::ReninsBoolean(HasLostPersons)));
        NJson::InsertField(result, "OnTheMove", NJson::ToJson(NJson::ReninsBoolean(OnTheMove)));
        InsertOptionalReninsBoolean(result, "DamageFace", HasMultipleDamageFaces);
        NJson::InsertField(result, "RegulationType", RegulationType);
        NJson::InsertNonNull(result, "AddInformation", AdditionalInfo);
        return result;
    }

    bool TKaskoClaim::ParseResponse(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["PolicyNumber"], PolicyNumber) &&
               NJson::ParseField(data["ClaimNumber"], ClaimNumber) &&
               NJson::ParseField(data["IncidentDate"], NJson::ReninsDateTime(IncidentTimestamp)) &&
               LossCodeInfo.ParseResponse(data) &&
               NJson::ParseField(data["Location"], Location) &&
               NJson::ParseField(data["IncidentRespPerson"], ResponsiblePerson) &&
               NJson::ParseField(data["IncidentPlaceType"], PlaceType) &&
               NJson::ParseField(data["IncidentTown"], Town) &&
               NJson::ParseField(data["IncidentAddress"], Address) &&
               NJson::ParseField(data["NotificationDate"], NJson::ReninsDateTime(NotificationTimestamp)) &&
               NJson::ParseField(data["RegistrationDate"], NJson::ReninsDateTime(RegistrationTimestamp)) &&
               NJson::ParseField(data["Registration"], RegistrationType) &&
               ParseOptionalReninsBoolean(data, "Europrotocol", IsEuroprotocol) &&
               NJson::ParseField(data["InquiryDate"], NJson::ReninsDate(InquiryDate)) &&
               NJson::ParseField(data["InjuredAnswer"], NJson::ReninsBoolean(HasInjuredPersons)) &&
               NJson::ParseField(data["LostAnswer"], NJson::ReninsBoolean(HasLostPersons)) &&
               ParseOptionalReninsBoolean(data, "DamageFace", HasMultipleDamageFaces) &&
               NJson::ParseField(data["RegulationType"], RegulationType) &&
               NJson::ParseField(data["AddInformation"], AdditionalInfo);
    }

    NDrive::TScheme TKaskoPolicy::GetScheme(const IServerBase& /* server */, const TMaybe<TKaskoPolicy>& entry) {
        NDrive::TScheme scheme;

        scheme.Add<TFSString>("insurance_company", "Где застрахован").SetMaxLength(100);
        scheme.Add<TFSString>("address", "Адрес").SetMaxLength(100);
        scheme.Add<TFSNumeric>("policy_start_date", "Действует с").SetVisual(TFSNumeric::EVisualType::DateTime);
        scheme.Add<TFSNumeric>("policy_end_date", "Действует по").SetVisual(TFSNumeric::EVisualType::DateTime);
        scheme.Add<TFSString>("policy_number", "Номер полиса").SetMaxLength(30);

        SetEntrySchemeDefaults(scheme, entry);

        return scheme;
    }
    NJson::TJsonValue TKaskoPolicy::MakeRequestData(const TReninsClaimClientConfig& /* config */) const {
        NJson::TJsonValue result;
        NJson::InsertNonNull(result, "InsuranceCompany", InsuranceCompany);
        NJson::InsertNonNull(result, "Address", Address);
        if (PolicyStartDate) {
            NJson::InsertField(result, "PolicyStartDate", NJson::ToJson(NJson::ReninsDateTime(PolicyStartDate)));
        }
        if (PolicyEndDate) {
            NJson::InsertField(result, "PolicyEndDate", NJson::ToJson(NJson::ReninsDateTime(PolicyEndDate)));
        }
        NJson::InsertNonNull(result, "PolicyNumber", PolicyNumber);
        return result;
    }

    bool TKaskoPolicy::ParseResponse(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["InsuranceCompany"], InsuranceCompany) &&
               NJson::ParseField(data["Address"], Address) &&
               NJson::ParseField(data["PolicyStartDate"], NJson::ReninsDateTime(PolicyStartDate)) &&
               NJson::ParseField(data["PolicyEndDate"], NJson::ReninsDateTime(PolicyEndDate)) &&
               NJson::ParseField(data["PolicyNumber"], PolicyNumber);
    }

    NJson::TJsonValue TKaskoPolicy::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertNonNull(result, "insurance_company", InsuranceCompany);
        NJson::InsertNonNull(result, "address", Address);
        if (PolicyStartDate) {
            NJson::InsertField(result, "policy_start_date", NJson::Seconds(PolicyStartDate));
        }
        if (PolicyEndDate) {
            NJson::InsertField(result, "policy_end_date", NJson::Seconds(PolicyEndDate));
        }
        NJson::InsertNonNull(result, "policy_number", PolicyNumber);
        return result;
    }

    bool TKaskoPolicy::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["insurance_company"], InsuranceCompany) &&
               NJson::ParseField(data["address"], Address) &&
               NJson::ParseField(data["policy_start_date"], NJson::Seconds(PolicyStartDate)) &&
               NJson::ParseField(data["policy_end_date"], NJson::Seconds(PolicyEndDate)) &&
               NJson::ParseField(data["policy_number"], PolicyNumber);
    }

    TKaskoPolicy::operator bool() const {
        return InsuranceCompany ||
               Address ||
               PolicyStartDate ||
               PolicyEndDate ||
               PolicyNumber;
    }

    NDrive::TScheme TKaskoDeclarant::GetScheme(const IServerBase& server, const TMaybe<TKaskoDeclarant>& entry) {
        NDrive::TScheme scheme;

        scheme.Add<TFSString>("last_name", "Фамилия").SetMaxLength(50).SetRequired(true);
        scheme.Add<TFSString>("first_name", "Имя").SetMaxLength(50).SetRequired(true);
        // scheme.Add<TFSString>("middle_name", "Отчество").SetMaxLength(50);  // hide as not required
        scheme.Add<TFSString>("email", "Email").SetMaxLength(30);  // suddenly required
        scheme.Add<TFSString>("emergency_phone_number", "Телефон оперативной обратной связи").SetMaxLength(30).SetRequired(true);
        scheme.Add<TFSString>("address", "Адрес").SetMaxLength(100);
        scheme.Add<TFSVariants>("role_in_accident", "Роль заявителя").SetCompoundVariants(TReninsCatalogueEntryAdapter<TKaskoRoleEntry>::Cast());
        scheme.Add<TFSString>("birth_date", "Дата рождения").SetRequired(true);
        scheme.Add<TFSString>("experience_year", "Стаж с");  // required depends

        SetEntrySchemeDefaults(scheme, entry);

        const auto& policyEntry = entry ? MakeMaybe(entry->GetPolicyOsagoDeclarant()) : Nothing();
        scheme.Add<TFSStructure>("policy_info", "Информация о страховщике").SetStructure(TKaskoPolicy::GetScheme(server, policyEntry));

        return scheme;
    }

    NJson::TJsonValue TKaskoDeclarant::MakeRequestData(const TReninsClaimClientConfig& config) const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "LastName", LastName);
        NJson::InsertField(result, "FirstName", FirstName);
        // NJson::InsertField(result, "MiddleName", MiddleName);  // do not provide as not required
        NJson::InsertField(result, "Email", Email);  // suddenly required
        NJson::InsertField(result, "EmergencyPhoneNm", NJson::ToJson(NJson::ReninsPhoneNumber(EmergencyPhoneNumber)));
        NJson::InsertNonNull(result, "Address", Address);
        NJson::InsertField(result, "RoleInAccident", RoleInAccident);
        NJson::InsertField(result, "BirthDate", NJson::ToJson(NJson::ReninsDate(BirthDate)));
        if (ExperienceYear) {
            NJson::InsertField(result, "ExperienceYear", NJson::ToJson(NJson::ReninsDate(ExperienceYear)));  // required depends
        }
        if (PolicyOsagoDeclarant) {
            NJson::InsertField(result, "PolicyOSAGODeclarant", PolicyOsagoDeclarant.MakeRequestData(config));
        }
        return result;
    }

    bool TKaskoDeclarant::ParseResponse(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["LastName"], LastName) &&
               NJson::ParseField(data["FirstName"], FirstName) &&
               NJson::ParseField(data["MiddleName"], MiddleName) &&
               NJson::ParseField(data["Email"], Email) &&
               NJson::ParseField(data["EmergencyPhoneNm"], EmergencyPhoneNumber) &&
               NJson::ParseField(data["Address"], Address) &&
               NJson::ParseField(data["RoleInAccident"], RoleInAccident) &&
               NJson::ParseField(data["BirthDate"], BirthDate) &&
               NJson::ParseField(data["ExperienceYear"], ExperienceYear) &&
               PolicyOsagoDeclarant.ParseResponse(data["PolicyOSAGODeclarant"]);
    }

    NJson::TJsonValue TKaskoDeclarant::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "last_name", LastName);
        NJson::InsertField(result, "first_name", FirstName);
        NJson::InsertNonNull(result, "middle_name", MiddleName);
        NJson::InsertField(result, "email", Email);
        NJson::InsertField(result, "emergency_phone_number", EmergencyPhoneNumber);
        NJson::InsertNonNull(result, "address", Address);
        NJson::InsertField(result, "role_in_accident", RoleInAccident);
        NJson::InsertField(result, "birth_date", BirthDate);
        NJson::InsertNonNull(result, "experience_year", ExperienceYear);
        if (PolicyOsagoDeclarant) {
            NJson::InsertField(result, "policy_info", PolicyOsagoDeclarant.SerializeToJson());
        }
        return result;
    }

    bool TKaskoDeclarant::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["last_name"], LastName) &&
               NJson::ParseField(data["first_name"], FirstName) &&
               NJson::ParseField(data["middle_name"], MiddleName) &&
               NJson::ParseField(data["email"], Email) &&
               NJson::ParseField(data["emergency_phone_number"], EmergencyPhoneNumber) &&
               NJson::ParseField(data["address"], Address) &&
               NJson::ParseField(data["role_in_accident"], RoleInAccident) &&
               NJson::ParseField(data["birth_date"], BirthDate) &&
               NJson::ParseField(data["experience_year"], ExperienceYear) &&
               PolicyOsagoDeclarant.DeserializeFromJson(data["policy_info"]);
    }

    NDrive::TScheme TKaskoParticipant::GetScheme(const IServerBase& server, const TMaybe<TKaskoParticipant>& entry) {
        NDrive::TScheme scheme;

        scheme.Add<TFSVariants>("role_in_accident", "Роль в происшествии").SetCompoundVariants(TReninsCatalogueEntryAdapter<TKaskoRoleEntry>::Cast());
        scheme.Add<TFSString>("last_name", "Фамилия").SetMaxLength(50).SetRequired(true);
        scheme.Add<TFSString>("first_name", "Имя").SetMaxLength(50).SetRequired(true);
        scheme.Add<TFSString>("middle_name", "Отчество").SetMaxLength(50);
        scheme.Add<TFSString>("email", "Email").SetMaxLength(50);  // suddenly required
        scheme.Add<TFSString>("emergency_phone_number", "Телефон оперативной обратной связи").SetMaxLength(30);
        scheme.Add<TFSString>("birth_date", "Дата рождения").SetRequired(true);
        scheme.Add<TFSString>("experience_year", "Стаж с");  // required for driver only
        // scheme.Add<TFSString>("address", "Адрес").SetMaxLength(100);  // hide as not required
        scheme.Add<TFSBoolean>("is_guilty", "Виновник");
        scheme.Add<TFSBoolean>("is_beneficiary", "Выгодоприобретатель");
        scheme.Add<TFSBoolean>("is_mediator", "Посредник");
        scheme.Add<TFSBoolean>("is_injured", "Травмированный");

        SetEntrySchemeDefaults(scheme, entry);

        const auto& policyEntry = entry ? MakeMaybe(entry->GetPolicyOsagoParticipant()) : Nothing();
        scheme.Add<TFSStructure>("policy_info", "Информация о страховщике").SetStructure(TKaskoPolicy::GetScheme(server, policyEntry));

        return scheme;
    }

    NJson::TJsonValue TKaskoParticipant::MakeRequestData(const TReninsClaimClientConfig& config) const {
        NJson::TJsonValue result;
        {
            const bool role1 = true;
            NJson::InsertField(result, "Role1", NJson::ToJson(NJson::ReninsBoolean(role1)));
        }
        NJson::InsertField(result, "RoleInAccident", RoleInAccident);
        NJson::InsertField(result, "LastName", LastName);
        NJson::InsertField(result, "FirstName", FirstName);
        NJson::InsertNonNull(result, "MiddleName", MiddleName);
        NJson::InsertField(result, "Email", Email);  // suddenly required
        if (EmergencyPhoneNumber) {
            NJson::InsertField(result, "EmergencyPhoneNm", NJson::ToJson(NJson::ReninsPhoneNumber(EmergencyPhoneNumber)));
        }
        NJson::InsertField(result, "BirthDate", NJson::ToJson(NJson::ReninsDate(BirthDate)));
        if (ExperienceYear) {
            NJson::InsertField(result, "ExperienceYear", NJson::ToJson(NJson::ReninsDate(ExperienceYear)));  // required for driver only
        }
        // NJson::InsertField(result, "Address", Address);  // do not provide as not required
        InsertOptionalReninsBoolean(result, "IsGuilty", IsGuilty);
        InsertOptionalReninsBoolean(result, "IsBeneficiary", IsBeneficiary);
        InsertOptionalReninsBoolean(result, "IsMediator", IsMediator);
        InsertOptionalReninsBoolean(result, "IsInjured", IsInjured);
        if (PolicyOsagoParticipant) {
            NJson::InsertField(result, "PolicyOSAGOParticipant", PolicyOsagoParticipant.MakeRequestData(config));
        }
        return result;
    }

    bool TKaskoParticipant::ParseResponse(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["RoleInAccident"], RoleInAccident) &&
               NJson::ParseField(data["LastName"], LastName) &&
               NJson::ParseField(data["FirstName"], FirstName) &&
               NJson::ParseField(data["MiddleName"], MiddleName) &&
               NJson::ParseField(data["Email"], Email) &&
               NJson::ParseField(data["EmergencyPhoneNm"], EmergencyPhoneNumber) &&
               NJson::ParseField(data["BirthDate"], BirthDate) &&
               NJson::ParseField(data["ExperienceYear"], ExperienceYear) &&
               NJson::ParseField(data["Address"], Address) &&
               ParseOptionalReninsBoolean(data, "IsGuilty", IsGuilty) &&
               ParseOptionalReninsBoolean(data, "IsBeneficiary", IsBeneficiary) &&
               ParseOptionalReninsBoolean(data, "IsMediator", IsMediator) &&
               ParseOptionalReninsBoolean(data, "IsInjured", IsInjured) &&
               PolicyOsagoParticipant.ParseResponse(data["PolicyOSAGOParticipant"]);
    }

    NJson::TJsonValue TKaskoParticipant::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "role_in_accident", RoleInAccident);
        NJson::InsertField(result, "last_name", LastName);
        NJson::InsertField(result, "first_name", FirstName);
        NJson::InsertNonNull(result, "middle_name", MiddleName);
        NJson::InsertField(result, "email", Email);
        NJson::InsertNonNull(result, "emergency_phone_number", EmergencyPhoneNumber);
        NJson::InsertField(result, "birth_date", BirthDate);
        NJson::InsertNonNull(result, "experience_year", ExperienceYear);
        NJson::InsertNonNull(result, "address", Address);
        NJson::InsertNonNull(result, "is_guilty", IsGuilty);
        NJson::InsertNonNull(result, "is_beneficiary", IsBeneficiary);
        NJson::InsertNonNull(result, "is_mediator", IsMediator);
        NJson::InsertNonNull(result, "is_injured", IsInjured);
        if (PolicyOsagoParticipant) {
            NJson::InsertField(result, "policy_info", PolicyOsagoParticipant.SerializeToJson());
        }
        return result;
    }

    bool TKaskoParticipant::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["role_in_accident"], RoleInAccident) &&
               NJson::ParseField(data["last_name"], LastName) &&
               NJson::ParseField(data["first_name"], FirstName) &&
               NJson::ParseField(data["middle_name"], MiddleName) &&
               NJson::ParseField(data["email"], Email) &&
               NJson::ParseField(data["emergency_phone_number"], EmergencyPhoneNumber) &&
               NJson::ParseField(data["birth_date"], BirthDate) &&
               NJson::ParseField(data["experience_year"], ExperienceYear) &&
               NJson::ParseField(data["address"], Address) &&
               NJson::ParseField(data["is_guilty"], IsGuilty) &&
               NJson::ParseField(data["is_beneficiary"], IsBeneficiary) &&
               NJson::ParseField(data["is_mediator"], IsMediator) &&
               NJson::ParseField(data["is_injured"], IsInjured) &&
               PolicyOsagoParticipant.DeserializeFromJson(data["policy_info"]);
    }

    NDrive::TScheme TKaskoVehicle::GetScheme(const IServerBase& /* server */, const TMaybe<TKaskoVehicle>& entry) {
        NDrive::TScheme scheme;

        scheme.Add<TFSString>("plate_number", "Гос. номер").SetMaxLength(30).SetRequired(true);
        scheme.Add<TFSVariants>("color", "Цвет").SetCompoundVariants(TReninsCatalogueEntryAdapter<TKaskoVehicleColorEntry>::Cast()).SetRequired(true);
        scheme.Add<TFSString>("vin", "VIN").SetMaxLength(50).SetRequired(true);
        scheme.Add<TFSString>("pts_number", "Номер СТС/ПТС").SetMaxLength(30);
        scheme.Add<TFSBoolean>("is_transportable", "ТС на ходу").SetRequired(true);
        // scheme.Add<TFSNumeric>("official_inspection_date", "Дата осмотра").SetVisual(TFSNumeric::EVisualType::DateTime);  // hide as not required
        scheme.Add<TFSBoolean>("no_truck_needed", "Без вызова эвакуатора");
        // scheme.Add<TFSNumeric>("reserve_amount", "Стоимость ремонта (ущерба), руб.").SetVisual(TFSNumeric::EVisualType::Money);  // hide as not required

        SetEntrySchemeDefaults(scheme, entry);

        return scheme;
    }

    NJson::TJsonValue TKaskoVehicle::MakeRequestData(const TReninsClaimClientConfig& /* config */) const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "PlateNumber", PlateNumber);
        NJson::InsertField(result, "RENSelectColor", Color);
        NJson::InsertField(result, "VIN", VIN);
        NJson::InsertNonNull(result, "PTSNumber", PTSNumber);
        NJson::InsertField(result, "VehicleTransportability", NJson::ToJson(NJson::ReninsBoolean(IsTransportable)));
        // NJson::InsertField(result, "OfficialInspectionDate", NJson::ToJson(NJson::ReninsDate(OfficialInspectionDate)));  // do not provide as not required
        InsertOptionalReninsBoolean(result, "NoTowTruckCall", NoTruckNeeded);
        // NJson::InsertField(result, "ReserveAmount", ReserveAmount);  // do not provide as not required
        return result;
    }

    bool TKaskoVehicle::ParseResponse(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["PlateNumber"], PlateNumber) &&
               NJson::ParseField(data["RENSelectColor"], Color) &&
               NJson::ParseField(data["VIN"], VIN) &&
               NJson::ParseField(data["PTSNumber"], PTSNumber) &&
               NJson::ParseField(data["VehicleTransportability"], NJson::ReninsBoolean(IsTransportable)) &&
               NJson::ParseField(data["OfficialInspectionDate"], NJson::ReninsDate(OfficialInspectionDate)) &&
               ParseOptionalReninsBoolean(data, "NoTowTruckCall", NoTruckNeeded) &&
               NJson::ParseField(data["ReserveAmount"], ReserveAmount);
    }

    NJson::TJsonValue TKaskoVehicle::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "plate_number", PlateNumber);
        NJson::InsertField(result, "color", Color);
        NJson::InsertField(result, "vin", VIN);
        NJson::InsertNonNull(result, "pts_number", PTSNumber);
        NJson::InsertField(result, "is_transportable", IsTransportable);
        NJson::InsertNonNull(result, "official_inspection_date", OfficialInspectionDate);
        NJson::InsertNonNull(result, "no_truck_needed", NoTruckNeeded);
        NJson::InsertNonNull(result, "reserve_amount", ReserveAmount);
        return result;
    }

    bool TKaskoVehicle::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["plate_number"], PlateNumber) &&
               NJson::ParseField(data["color"], Color) &&
               NJson::ParseField(data["vin"], VIN) &&
               NJson::ParseField(data["pts_number"], PTSNumber) &&
               NJson::ParseField(data["is_transportable"], IsTransportable) &&
               NJson::ParseField(data["official_inspection_date"], OfficialInspectionDate) &&
               NJson::ParseField(data["no_truck_needed"], NoTruckNeeded) &&
               NJson::ParseField(data["reserve_amount"], ReserveAmount);
    }

    NDrive::TScheme TKaskoDamages::GetScheme(const IServerBase& /* server */, const TMaybe<TKaskoDamages>& entry) {
        NDrive::TScheme scheme;

        scheme.Add<TFSVariants>("damages", "Список поврежденных элементов лев./прав.").SetCompoundVariants(TReninsCatalogueEntryAdapter<TKaskoDamageListEntry>::Cast()).SetMultiSelect(true);

        SetEntrySchemeDefaults(scheme, entry);

        return scheme;
    }

    NJson::TJsonValue TKaskoDamages::MakeRequestData(const TReninsClaimClientConfig& /* config */) const {
        return NJson::TMapBuilder("Damage", NJson::ToJson(Damages));
    }

    bool TKaskoDamages::ParseResponse(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["Damage"], Damages);
    }

    NJson::TJsonValue TKaskoDamages::SerializeToJson() const {
        return NJson::TMapBuilder("damages", NJson::ToJson(Damages));
    }

    bool TKaskoDamages::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["damages"], Damages);
    }

    NDrive::TScheme TKaskoSTOAInfo::GetScheme(const IServerBase& /* server */, const TMaybe<TKaskoSTOAInfo>& entry) {
        NDrive::TScheme scheme;

        auto& stoaAddressScheme = scheme.Add<TFSVariants>("stoa_info", "Идентификатор адреса сервиса");
        stoaAddressScheme.SetCompoundVariants(TReninsCatalogueEntryAdapter<TKaskoSTOAAddressEntry>::Cast()).SetRequired(true);
        if (entry) {
            stoaAddressScheme.SetDefault(entry->GetSTOAAddressInfo().GetId());
        }

        return scheme;
    }

    NJson::TJsonValue TKaskoSTOAInfo::MakeRequestData(const TReninsClaimClientConfig& config) const {
        NJson::TJsonValue result;
        NJson::MergeJson(STOAAddressInfo.MakeRequestData(config), result);
        return result;
    }

    bool TKaskoSTOAInfo::ParseResponse(const NJson::TJsonValue& data) {
        return STOAAddressInfo.ParseResponse(data);
    }

    NJson::TJsonValue TKaskoSTOAInfo::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "stoa_info", STOAAddressInfo.SerializeToJson());
        return result;
    }

    bool TKaskoSTOAInfo::DeserializeFromJson(const NJson::TJsonValue& data) {
        const auto& stoaInfoJson = data["stoa_info"];
        if (stoaInfoJson.IsDefined() && !(stoaInfoJson.IsString() && stoaInfoJson.GetString().empty())) {
            auto stoaAddressInfo = TKaskoSTOAAddressEntry::Restore(NDrive::HasServer() ? &NDrive::GetServer() : nullptr, stoaInfoJson);
            if (!stoaAddressInfo) {
                return false;
            }
            STOAAddressInfo = std::move(*stoaAddressInfo);
        }
        return true;
    }

    NDrive::TScheme TKaskoReserveAmountInfo::GetScheme(const IServerBase& /* server */, const TMaybe<TKaskoReserveAmountInfo>& entry) {
        NDrive::TScheme scheme;

        scheme.Add<TFSNumeric>("amount", "Размер франшизы (руб.)").SetRequired(true);

        SetEntrySchemeDefaults(scheme, entry);

        return scheme;
    }

    NJson::TJsonValue TKaskoReserveAmountInfo::MakeRequestData(const TReninsClaimClientConfig& /* config */) const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "DeductibleAmount", Amount);
        return result;
    }

    bool TKaskoReserveAmountInfo::ParseResponse(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["DeductibleAmount"], Amount);
    }

    NJson::TJsonValue TKaskoReserveAmountInfo::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "amount", Amount);
        return result;
    }

    bool TKaskoReserveAmountInfo::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["amount"], Amount);
    }

    NDrive::TScheme TKaskoClaimEntry::GetScheme(const IServerBase& server, const TMaybe<TKaskoClaimEntry>& entry, bool useTabs) {
        NDrive::TScheme scheme;
        {
            auto gTab = scheme.StartTabGuard(useTabs ? "General" : NDrive::ISchemeElement::DefaultTabName);
            const auto& tabEntry = entry ? MakeMaybe(entry->GetGeneralClaimInfo()) : Nothing();
            scheme.Add<TFSStructure>("general_claim_info", "Общая информация").SetStructure(TKaskoGeneralClaimInfo::GetScheme(server, tabEntry));
        }
        {
            auto gTab = scheme.StartTabGuard(useTabs ? "Claim" : NDrive::ISchemeElement::DefaultTabName);
            const auto& tabEntry = entry ? MakeMaybe(entry->GetClaim()) : Nothing();
            scheme.Add<TFSStructure>("claim_info", "Информация о заявителе, месте и условиях происшествия и пр.").SetStructure(TKaskoClaim::GetScheme(server, tabEntry));
        }
        {
            auto gTab = scheme.StartTabGuard(useTabs ? "Declarant" : NDrive::ISchemeElement::DefaultTabName);
            const auto& tabEntry = entry ? MakeMaybe(entry->GetDeclarant()) : Nothing();
            scheme.Add<TFSStructure>("declarant_info", "Информация о заявителе").SetStructure(TKaskoDeclarant::GetScheme(server, tabEntry));
        }
        {
            auto gTab = scheme.StartTabGuard(useTabs ? "Participants" : NDrive::ISchemeElement::DefaultTabName);
            auto& participantsScheme = scheme.Add<TFSArray>("participants_info", "Другие участники физ. лицо");
            participantsScheme.SetElement(TKaskoParticipant::GetScheme(server));
            if (entry) {
                participantsScheme.SetDefaults(entry->GetParticipants());
            }
        }
        {
            auto gTab = scheme.StartTabGuard(useTabs ? "Vehicle" : NDrive::ISchemeElement::DefaultTabName);
            const auto& tabEntry = entry ? MakeMaybe(entry->GetVehicleDescription()) : Nothing();
            scheme.Add<TFSStructure>("vehicle_info", "Информация о застрахованном ТС").SetStructure(TKaskoVehicle::GetScheme(server, tabEntry));
        }
        {
            auto gTab = scheme.StartTabGuard(useTabs ? "Damages" : NDrive::ISchemeElement::DefaultTabName);
            const auto& tabEntry = entry ? MakeMaybe(entry->GetDamages()) : Nothing();
            scheme.Add<TFSStructure>("damages_info", "Список поврежденных элементов").SetStructure(TKaskoDamages::GetScheme(server, tabEntry));
        }
        {
            auto gTab = scheme.StartTabGuard(useTabs ? "Stoa" : NDrive::ISchemeElement::DefaultTabName);
            const auto& tabEntry = entry ? MakeMaybe(entry->GetSTOAInfo()) : Nothing();
            scheme.Add<TFSStructure>("stoa_info", "Адрес сервиса / тип урегулирования").SetStructure(TKaskoSTOAInfo::GetScheme(server, tabEntry));
        }
        {
            auto gTab = scheme.StartTabGuard(useTabs ? "ReserveAmount" : NDrive::ISchemeElement::DefaultTabName);
            const auto& tabEntry = entry ? MakeMaybe(entry->GetReserveAmountInfo()) : Nothing();
            scheme.Add<TFSStructure>("reserve_amount_info", "Франшизы по договору страхования").SetStructure(TKaskoReserveAmountInfo::GetScheme(server, tabEntry));
        }
        {
            // NB. Documents are readonly and processed in other way

            // auto gTab = scheme.StartTabGuard(useTabs ? "Documents" : NDrive::ISchemeElement::DefaultTabName);
            // const auto& tabEntry = entry ? MakeMaybe(entry->GetDocuments()) : Nothing();
            // auto& documentsScheme = scheme.Add<TFSStructure>("documents_info", "Документы");
            // documentsScheme.SetStructure(TDocuments::GetScheme(server, TReninsCatalogueEntryAdapter<TKaskoDocumentTypeEntry>(), tabEntry));
        }
        return scheme;
    }

    NJson::TJsonValue TKaskoClaimEntry::MakeRequestData(const TReninsClaimClientConfig& config) const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "GENL", GeneralClaimInfo.MakeRequestData(config));
        NJson::InsertField(result, "Claim", Claim.MakeRequestData(config));
        NJson::InsertField(result, "Declarant", Declarant.MakeRequestData(config));
        {
            NJson::TJsonValue innerParticipants = NJson::JSON_ARRAY;
            for (auto&& participant: Participants) {
                innerParticipants.AppendValue(participant.MakeRequestData(config));
            }

            NJson::TJsonValue participants = NJson::JSON_MAP;
            participants.InsertValue("Participant", std::move(innerParticipants));

            result.InsertValue("ParticipantList", std::move(participants));
        }
        NJson::InsertField(result, "Vehicle", VehicleDescription.MakeRequestData(config));
        NJson::InsertField(result, "Damages", Damages.MakeRequestData(config));
        NJson::InsertField(result, "Stoa", STOAInfo.MakeRequestData(config));
        NJson::InsertField(result, "Reserve", ReserveAmountInfo.MakeRequestData(config));
        NJson::InsertField(result, "DocList", Documents.MakeRequestData(config));
        return result;
    }

    bool TKaskoClaimEntry::ParseResponse(const NJson::TJsonValue& data) {
        if (data.Has("ParticipantList")) {
            auto& rawParticipants = data["ParticipantList"]["Participant"];
            if (!rawParticipants.IsArray()) {
                return false;
            }
            for (auto&& rawParticipant: rawParticipants.GetArray()) {
                TKaskoParticipant participant;
                if (!participant.ParseResponse(rawParticipant)) {
                    return false;
                }
                Participants.push_back(std::move(participant));
            }
        }
        if (data.Has("DocList") && !Documents.ParseResponse(data["DocList"])) {
            return false;
        }
        if (data.Has("ClaimNumber") && !NJson::ParseField(data["ClaimNumber"], Claim.MutableClaimNumber())) {
            return false;
        }
        return GeneralClaimInfo.ParseResponse(data["GENL"]) &&
               Claim.ParseResponse(data["Claim"]) &&
               Declarant.ParseResponse(data["Declarant"]) &&
               VehicleDescription.ParseResponse(data["Vehicle"]) &&
               Damages.ParseResponse(data["Damages"]) &&
               STOAInfo.ParseResponse(data["Stoa"]) &&
               ReserveAmountInfo.ParseResponse(data["Reserve"]);
    }

    NJson::TJsonValue TKaskoClaimEntry::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "general_claim_info", GeneralClaimInfo.SerializeToJson());
        NJson::InsertField(result, "claim_info", Claim.SerializeToJson());
        NJson::InsertField(result, "declarant_info", Declarant.SerializeToJson());
        NJson::InsertField(result, "participants_info", NJson::ToJson(Participants));
        NJson::InsertField(result, "vehicle_info", VehicleDescription.SerializeToJson());
        NJson::InsertField(result, "damages_info", Damages.SerializeToJson());
        NJson::InsertField(result, "stoa_info", STOAInfo.SerializeToJson());
        NJson::InsertField(result, "reserve_amount_info", ReserveAmountInfo.SerializeToJson());
        // NJson::InsertField(result, "documents_info", Documents.SerializeToJson());  // readonly field, used in scheme and external data only
        return result;
    }

    bool TKaskoClaimEntry::DeserializeFromJson(const NJson::TJsonValue& data) {
        return GeneralClaimInfo.DeserializeFromJson(data["general_claim_info"]) &&
               Claim.DeserializeFromJson(data["claim_info"]) &&
               Declarant.DeserializeFromJson(data["declarant_info"]) &&
               NJson::TryFromJson(data["participants_info"], Participants) &&
               VehicleDescription.DeserializeFromJson(data["vehicle_info"]) &&
               Damages.DeserializeFromJson(data["damages_info"]) &&
               STOAInfo.DeserializeFromJson(data["stoa_info"]) &&
               ReserveAmountInfo.DeserializeFromJson(data["reserve_amount_info"]) &&
               // Documents.DeserializeFromJson(data["documents_info"]) &&  // readonly field, used in scheme and external data  only
               true;
    }

    bool TKaskoClaimEntry::Validate(TMessagesCollector& errors) const {
        TSet<TString> requiredDocumentCodes = {
          //"I040",  // Акт осмотра ГРС
          //"G010",  // Заявление о страховом случае по риску УЩЕРБ
            "I190",  // Фото первичного осмотра ТС
        };

        if (Claim.OptionalIsEuroprotocol().GetOrElse(false)) {
            requiredDocumentCodes.emplace("J020");  // Извещение о ДТП
        }

        if (Claim.IsOnTheMove()) {
            requiredDocumentCodes.insert("D045"); // Комплект документов (СТС/ПТС+Вод.удост)
        }

        if (Claim.GetRegistrationType() == "МВД") {
            requiredDocumentCodes.emplace("C150");  // Постановление о возбуждении/отказе в возбуждении уголовного дела
        } else if (Claim.GetRegistrationType() == "ГИБДД") {
            requiredDocumentCodes.emplace("C080");  // Протокол + Постановление ГИБДД / Определение ГИБДД
        }

        for (const auto& document: Documents.GetDocuments()) {
            requiredDocumentCodes.erase(document.GetAttachmentCode());
        }

        if (requiredDocumentCodes) {
            TStringBuilder errorMessage;
            errorMessage << "Some required documents are not provided: ";

            TKaskoDocumentTypeEntry availableDocuments;
            bool descriptionsLoaded = NDrive::HasServer() && availableDocuments.LoadSettingValue(&NDrive::GetServer());

            bool isFirstDocument = true;
            for (auto&& code: requiredDocumentCodes) {
                errorMessage << (isFirstDocument ? "" : ", ") << code << ((descriptionsLoaded) ? " (" + availableDocuments.GetValues().Value(code, "unknown") + ")" : "");
                isFirstDocument = false;
            }

            errors.AddMessage(__LOCATION__, errorMessage);
            return false;
        }

        return true;
    }

    bool TKaskoClaimEntry::IsInitialClaimApplication() const {
        return Claim.GetClaimNumber().empty();
    }
}
