#include "client.h"

#include <drive/backend/logging/events.h>

#include <drive/library/cpp/threading/future.h>

#include <kernel/reqid/reqid.h>

#include <library/cpp/http/simple/http_client.h>
#include <library/cpp/json/json_reader.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/types/accessor.h>

namespace {
    class TReninsResponse {
    public:
        static TMaybe<TReninsResponse> Construct(const NJson::TJsonValue& data) {
            TReninsResponse response;
            return response.DeserializeFromJson(data) ? response : TMaybe<TReninsResponse>();
        }

        bool IsErrorneous() const {
            return !ErrorCode.empty();
        }

        bool DeserializeFromJson(const NJson::TJsonValue& data) {
            return NJson::ParseField(data["ErrorCode"], ErrorCode, /* required = */ true) &&
                   NJson::ParseField(data["ErrorMessage"], ErrorMessage, /* required = */ true) &&
                   NJson::ParseField(data["ClaimNumber"], ClaimNumber) &&
                   NJson::ParseField(data["RequestInstanceId"], RequestId);
        }

        TString GetReport() const {
            return TStringBuilder()
                   << "Error code: " << ErrorCode
                   << "; error message: " << ErrorMessage
                   << "; request id: " << RequestId
                   << ((ClaimNumber) ? "; claim number: " + ClaimNumber : "");
        }

    private:
        R_FIELD(TString, ErrorCode);
        R_FIELD(TString, ErrorMessage);
        R_FIELD(TString, ClaimNumber);
        R_FIELD(TString, RequestId);
    };
}

namespace NDrive::NRenins {
    TReninsClaimClient::TReninsClaimClient(const TReninsClaimClientConfig& config)
        : TBase(config, "renins_claim_api")
    {
        if (config.GetUseSimpleHttpClient()) {
            TStringBuilder host;
            if (config.GetIsHttps()) {
                host << "https://";
            }
            host << config.GetHost();
            SimpleHttpClient = MakeHolder<TSimpleHttpClient>(
                host,
                config.GetPort(),
                config.GetRequestTimeout(),
                config.GetRequestConfig().GetConnectTimeout()
            );
        }
    }

    TReninsClaimClient::~TReninsClaimClient() {
    }

    NThreading::TFuture<void> TReninsClaimClient::ReportKaskoAsync(TKaskoClaimEntry& requestEntry) const {
        return SendAsyncPostRequest(EReninsClaimOperationType::CreateKasko, Config.GetCreateKaskoApiPath(), requestEntry);
    }

    bool TReninsClaimClient::ReportKaskoSync(TKaskoClaimEntry& requestEntry, TMessagesCollector& errors) const {
        return SendSyncPostRequest(EReninsClaimOperationType::CreateKasko, Config.GetCreateKaskoApiPath(), requestEntry, errors);
    }

    NThreading::TFuture<void> TReninsClaimClient::UploadDocumentsAsync(TDocumentsEntry& requestEntry) const {
        return SendAsyncPostRequest(EReninsClaimOperationType::UploadDocuments, Config.GetUploadDocumentsApiPath(), requestEntry);
    }

    bool TReninsClaimClient::UploadDocumentsSync(TDocumentsEntry& requestEntry, TMessagesCollector& errors) const {
        return SendSyncPostRequest(EReninsClaimOperationType::UploadDocuments, Config.GetUploadDocumentsApiPath(), requestEntry, errors);
    }

    NThreading::TFuture<void> TReninsClaimClient::GetCatalogueValuesAsync(ICatalogueEntry& requestEntry) const {
        return SendAsyncPostRequest(EReninsClaimOperationType::GetCatalogue, Config.GetCatalogueApiPath(), requestEntry);
    }

    bool TReninsClaimClient::GetCatalogueValuesSync(ICatalogueEntry& requestEntry, TMessagesCollector& errors) const {
        return SendSyncPostRequest(EReninsClaimOperationType::GetCatalogue, Config.GetCatalogueApiPath(), requestEntry, errors);
    }

    NThreading::TFuture<void> TReninsClaimClient::GetSTOAAddressesAsync(ICatalogueEntry& requestEntry) const {
        return SendAsyncPostRequest(EReninsClaimOperationType::GetSTOAList, Config.GetSTOAListApiPath(), requestEntry);
    }

    bool TReninsClaimClient::GetSTOAAddressesSync(ICatalogueEntry& requestEntry, TMessagesCollector& errors) const {
        return SendSyncPostRequest(EReninsClaimOperationType::GetSTOAList, Config.GetSTOAListApiPath(), requestEntry, errors);
    }

    bool TReninsClaimClient::SendSyncPostRequest(EReninsClaimOperationType operationType, const TString& apiPath, IRequestEntry& requestEntry, TMessagesCollector& errors) const {
        auto r = SendAsyncPostRequest(operationType, apiPath, requestEntry);
        if (r.Wait(Config.GetRequestTimeout()) && r.HasValue()) {
            return true;
        }
        errors.AddMessage(__LOCATION__, NThreading::GetExceptionMessage(r));
        return false;
    }

    NThreading::TFuture<void> TReninsClaimClient::SendAsyncPostRequest(EReninsClaimOperationType operationType, const TString& apiPath, IRequestEntry& requestEntry) const {
        TMessagesCollector errors;
        if (!requestEntry.Validate(errors)) {
            return NThreading::TExceptionFuture() << "Request is invalid: " << errors.GetStringReport();
        }
        auto reqid = ReqIdGenerate();
        auto request = CreateCommonRequest(apiPath, ERequestMethod::POST, requestEntry.MakeRequestData(Config), ERequestContentType::Json);
        request.AddCgiData("&reqid=" + reqid);
        request.SetDumpEventLog(true);
        auto result = SendRequest(operationType, request);
        return result.Apply([&requestEntry](const NThreading::TFuture<NJson::TJsonValue>& r) -> NThreading::TFuture<void> {
            if (r.HasValue()) {
                if (!requestEntry.ParseResponse(r.GetValue())) {
                    return NThreading::TExceptionFuture() << "Error parsing response: " << r.GetValue().GetStringRobust();
                }
                return NThreading::MakeFuture();
            }
            return NThreading::TExceptionFuture() << NThreading::GetExceptionMessage(r);
        });
    }

    NThreading::TFuture<NJson::TJsonValue> TReninsClaimClient::SendRequest(EReninsClaimOperationType operationType, const NNeh::THttpRequest& request) const {
        Logger.ProcessStart(operationType);

        const TInstant deadline = Now() + Config.GetRequestTimeout();
        NThreading::TFuture<NNeh::THttpReply> response;
        if (SimpleHttpClient) {
            TStringBuilder relativeUrl;
            if (!request.GetUri().StartsWith('/')) {
                relativeUrl << '/';
            }
            relativeUrl << request.GetUri() << '?' << request.GetCgiData();

            TStringBuf body = request.GetPostData().AsStringBuf();
            THashMap<TString, TString> headers = { request.GetHeaders().begin(), request.GetHeaders().end() };

            TString result;
            TStringOutput output(result);
            SimpleHttpClient->DoPost(relativeUrl, body, &output, headers);

            NNeh::THttpReply reply;
            reply.SetCode(HTTP_OK);
            reply.SetContent(result);
            response = NThreading::MakeFuture(std::move(reply));
        } else {
            response = Impl->SendAsync(request, deadline);
        }
        {
            NDrive::TEventLog::Log("ReninsClaimRequest", NJson::TMapBuilder
                ("host", Config.GetHost())
                ("port", Config.GetPort())
                ("request", request.Serialize())
            );
        }
        auto eventLogState = NDrive::TEventLog::CaptureState();
        return response.Apply([
                eventLogState = std::move(eventLogState),
                &logger = Logger,
                operationType
            ](const NThreading::TFuture<NUtil::THttpReply>& r) -> NThreading::TFuture<NJson::TJsonValue> {
                auto report = r.GetValue();
                auto elsg = NDrive::TEventLog::Guard(eventLogState);
                NDrive::TEventLog::Log("ReninsClaimResponse", report.Serialize());
                if (!report.HasReply()) {
                    logger.ProcessError(operationType);
                    return NThreading::TExceptionFuture() << "No reply: " << report.ErrorMessage();
                }

                ui32 code = report.Code();
                logger.ProcessReply(operationType, code);

                NJson::TJsonValue result;
                if (!NJson::ReadJsonFastTree(report.Content(), &result)) {
                    logger.ProcessError(operationType);
                    return NThreading::TExceptionFuture() << "Incorrect json format: " << report.Content();
                }

                // error response may have successful response code, so try parse it as an error first
                auto parsedResponse = TReninsResponse::Construct(result);
                if (!IsRequestSuccessful(code) || (parsedResponse && parsedResponse->IsErrorneous())) {
                    logger.ProcessError(operationType);
                    return NThreading::TExceptionFuture() << "Unsuccessful request: " << code
                                                            << "; details: " << ((parsedResponse) ? parsedResponse->GetReport() : "cannot obtain details");
                }

                return NThreading::MakeFuture(std::move(result));
            }
        );
    }
}
