#include "common_entity.h"

#include "scheme_adapters.h"

#include <drive/backend/abstract/base.h>

#include <drive/library/cpp/mds/client.h>
#include <drive/library/cpp/threading/future.h>

#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/string_utils/url/url.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/network/neh.h>

namespace {
    auto GetDefaultConfig(const TDuration timeout) {
        NSimpleMeta::TConfig config;
        config.SetGlobalTimeout(timeout).SetConnectTimeout(TDuration::MilliSeconds(100));
        return config;
    }

    NThreading::TFuture<TString> DownloadAttachmentAsync(const TString& attachmentUrl, const NDrive::NRenins::TReninsClaimClientConfig& config, NNeh::THttpClient& client) {
        TString host, path;
        SplitUrlToHostAndPath(attachmentUrl, host, path);

        if (!client.HasSource(host)) {
            client.RegisterSource(attachmentUrl, GetDefaultConfig(config.GetAttachmentDownloadTimeout()), host);
        }

        NNeh::THttpRequest request;
        request.SetUri(path);

        auto asyncResult = client.SendAsync(host, request);
        return asyncResult.Apply([](const NThreading::TFuture<NUtil::THttpReply>& r) -> NThreading::TFuture<TString> {
            const auto& report = r.GetValue();
            if (!report.HasReply()) {
                return NThreading::TExceptionFuture() << "No reply: " << report.ErrorMessage();
            }
            if (!report.IsSuccessReply()) {
                return NThreading::TExceptionFuture() << "Error downloading attachment: " << report.GetDebugReply();
            }
            return NThreading::MakeFuture(Base64Encode(report.Content()));
        });
    }
}

namespace NJson {
    template <>
    TJsonValue ToJson(const NDrive::NRenins::TDocument& object) {
        return object.SerializeToJson();
    }

    template <>
    bool TryFromJson(const NJson::TJsonValue& value, NDrive::NRenins::TDocument& result) {
        return result.DeserializeFromJson(value);
    }
}

namespace NDrive::NRenins {
    TDocument::TDocument(const TString& url, const TString& attachmentCode, const TString& name)
        : Url(url)
        , AttachmentCode(attachmentCode)
        , Name(name)
    {
    }

    NDrive::TScheme TDocument::GetScheme(const IServerBase& /* server */, const IVariantsAdapter& variantsAdapter) {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("url", "Ссылка на документ").SetRequired(true);
        scheme.Add<TFSVariants>("attachment_code", "Код шаблона").SetCompoundVariants(static_cast<TFSVariants::TCompoundVariants>(variantsAdapter)).SetRequired(true);
        return scheme;
    }

    NJson::TJsonValue TDocument::MakeRequestData(const TReninsClaimClientConfig& config) const {
        NNeh::THttpClient client;
        auto asyncResult = MakeRequestDataAsync(config, client);
        asyncResult.Wait(config.GetAttachmentDownloadTimeout());
        return (asyncResult.HasValue()) ? asyncResult.ExtractValue() : NJson::TJsonValue();
    }

    NThreading::TFuture<NJson::TJsonValue> TDocument::MakeRequestDataAsync(const TReninsClaimClientConfig& config, NNeh::THttpClient& client) const {
        if (Data) {
            NJson::TJsonValue result = NJson::TMapBuilder
                ("DocCode", AttachmentCode)
                ("DocName", Name ? Name : "Фото")
                ("DocFormat", "jpg")
                ("Data", Base64Encode(Data))
            ;
            return NThreading::MakeFuture(std::move(result));
        }

        // NB. DocFormat is required. Available variants: "jpg", "pdf", "gif", "png", "bmp", "tif"
        // NB. DocName is not required

        TString attachmentFormat = (Url.EndsWith(".pdf")) ? "pdf" : "jpg";
        TString docName = Name;
        if (!docName) {
            docName = attachmentFormat == "pdf" ? "Вложение" : "Фото";
        }

        auto asyncAttachmentBase64Data = DownloadAttachmentAsync(Url, config, client);
        if (!asyncAttachmentBase64Data.Wait(config.GetAttachmentDownloadTimeout()) || !asyncAttachmentBase64Data.HasValue()) {
            return NThreading::TExceptionFuture() << NThreading::GetExceptionMessage(asyncAttachmentBase64Data);
        }

        return NThreading::MakeFuture<NJson::TJsonValue>(
            NJson::TMapBuilder
            ("DocCode", AttachmentCode)
            ("DocName", docName)
            ("DocFormat", attachmentFormat)
            ("Data", asyncAttachmentBase64Data.ExtractValue())
        );
    }

    bool TDocument::ParseResponse(const NJson::TJsonValue& /* data */) {
        return true;  // no documents response should to be parsed
    }

    NJson::TJsonValue TDocument::SerializeToJson() const {
        return NJson::TMapBuilder
            ("url", Url)
            ("attachment_code", AttachmentCode);
    }

    bool TDocument::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["url"], Url) &&
               NJson::ParseField(data["attachment_code"], AttachmentCode);
    }

    NDrive::TScheme TDocuments::GetScheme(const IServerBase& server, const IVariantsAdapter& variantsAdapter, const TMaybe<TDocuments>& entry) {
        NDrive::TScheme scheme;
        auto& documentsScheme = scheme.Add<TFSArray>("documents", "Документы");
        documentsScheme.SetElement(TDocument::GetScheme(server, variantsAdapter));
        if (entry) {
            documentsScheme.SetDefaults(entry->GetDocuments());
        }
        return scheme;
    }

    NJson::TJsonValue TDocuments::MakeRequestData(const TReninsClaimClientConfig& config) const {
        NNeh::THttpClient client;

        TVector<NThreading::TFuture<NJson::TJsonValue>> futures;
        for (auto&& document: Documents) {
            futures.push_back(document.MakeRequestDataAsync(config, client));
        }

        if (!NThreading::WaitExceptionOrAll(futures).Wait(config.GetAttachmentDownloadTimeout())) {
            return {};  // to return maybe or expected
        }

        NJson::TJsonValue innerDocuments = NJson::JSON_ARRAY;
        for (auto&& f: futures) {
            if (f.HasException()) {
                return {};  // to return maybe or expected
            }
            innerDocuments.AppendValue(f.ExtractValue());
        }

        NJson::TJsonValue documents = NJson::JSON_MAP;
        documents.InsertValue("Doc", std::move(innerDocuments));

        return documents;
    }

    bool TDocuments::ParseResponse(const NJson::TJsonValue& data) {
        auto& rawDocuments = data["Doc"];
        if (!rawDocuments.IsArray()) {
            return false;
        }
        for (auto&& rawDocument: rawDocuments.GetArray()) {
            TDocument document;
            if (!document.ParseResponse(rawDocument)) {
                return false;
            }
            Documents.push_back(std::move(document));
        }
        return true;
    }

    NJson::TJsonValue TDocuments::SerializeToJson() const {
        return NJson::TMapBuilder("documents", NJson::ToJson(Documents));
    }

    bool TDocuments::DeserializeFromJson(const NJson::TJsonValue& data) {
        return data.Has("documents") ? NJson::ParseField(data["documents"], Documents) : NJson::TryFromJson(data, Documents);  // temporary backward compatibility
    }

    NDrive::TScheme TDocumentsEntry::GetScheme(const IServerBase& server, const IVariantsAdapter& variantsAdapter) {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("claim_number", "Номер убытка");
        scheme.Add<TFSStructure>("documents_info", "Документы").SetStructure(TDocuments::GetScheme(server, variantsAdapter));
        return scheme;
    }

    NJson::TJsonValue TDocumentsEntry::MakeRequestData(const TReninsClaimClientConfig& config) const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "Login", config.GetLogin());
        NJson::InsertField(result, "Password", config.GetPassword());
        NJson::InsertField(result, "ClaimNumber", ClaimNumber);
        NJson::InsertField(result, "DocList", Documents.MakeRequestData(config));
        return result;
    }

    bool TDocumentsEntry::ParseResponse(const NJson::TJsonValue& data) {
        if (data.Has("DocList") && !Documents.ParseResponse(data["DocList"])) {
            return false;
        }
        return NJson::ParseField(data["ClaimNumber"], ClaimNumber);
    }

    NJson::TJsonValue TDocumentsEntry::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "claim_number", ClaimNumber);
        NJson::InsertField(result, "documents_info", Documents.SerializeToJson());
        return result;
    }

    bool TDocumentsEntry::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["claim_number"], ClaimNumber) &&
               Documents.DeserializeFromJson(data["documents_info"]);
    }

    bool TDocumentsEntry::Validate(TMessagesCollector& errors) const {
        if (!ClaimNumber) {
            errors.AddMessage(__LOCATION__, "Claim number is not filled");
            return false;
        }
        if (Documents.GetDocuments().empty()) {
            errors.AddMessage(__LOCATION__, "No documents to upload provided");
            return false;
        }
        return true;
    }
}
