#pragma once

#include "constants.h"

#include <drive/library/cpp/raw_text/datetime.h>

#include <rtline/library/json/cast.h>

#include <util/string/cast.h>
#include <util/string/strip.h>

namespace NJson {
    namespace NPrivate {
        TJsonValue ReninsPhoneNumberToJson(const TString& value);
        TJsonValue ReninsBooleanToJson(bool value);
        TJsonValue ReninsDateTimeToJson(TInstant value);

        bool TryReninsPhoneNumberFromJson(const TJsonValue& value, TString& result);
        bool TryReninsBooleanFromJson(const TJsonValue& value, bool& result);
        bool TryReninsDateTimeFromJson(const TJsonValue& value, TInstant& result);
    }
}

namespace NJson {
    namespace NPrivate {
        template <typename T, std::enable_if_t<std::is_same<std::decay_t<T>, TString>::value, int> = 0>
        struct TReninsPhoneNumber {
            TReninsPhoneNumber(T& value)
                : Value(value)
            {
            }

            T& Value;
        };

        template <typename T, std::enable_if_t<std::is_same<std::decay_t<T>, bool>::value, int> = 0>
        struct TReninsBoolean {
            TReninsBoolean(T& value)
                : Value(value)
            {
            }

            T& Value;
        };

        template <typename T>
        struct TReninsDate {
            TReninsDate(T& value)
                : Value(value)
            {
            }

            T& Value;
        };

        template <typename T, std::enable_if_t<std::is_same<std::decay_t<T>, TInstant>::value, int> = 0>
        struct TReninsDateTime {
            TReninsDateTime(T& value)
                : Value(value)
            {
            }

            T& Value;
        };

        template <typename T>
        struct TInnerKeyValue {
        public:
            TInnerKeyValue(T& container, TStringBuf key)
                : Container(container)
                , Key(key)
            {
            }

        public:
            T& Container;
            TStringBuf Key;
        };
    }

    template <typename T>
    NPrivate::TReninsPhoneNumber<T> ReninsPhoneNumber(T& object) {
        return { object };
    }

    template <typename T>
    NPrivate::TReninsBoolean<T> ReninsBoolean(T& object) {
        return { object };
    }

    template <typename T>
    NPrivate::TReninsDate<T> ReninsDate(T& object) {
        return { object };
    }

    template <typename T>
    NPrivate::TReninsDateTime<T> ReninsDateTime(T& object) {
        return { object };
    }

    template <typename T>
    NPrivate::TInnerKeyValue<T> InnerKeyValue(T& object, TStringBuf key) {
        return { object, key };
    }
}

namespace NJson {
    template <typename T>
    TJsonValue ToJson(const NPrivate::TReninsPhoneNumber<T>& object) {
        return NPrivate::ReninsPhoneNumberToJson(object.Value);
    }

    template <typename T>
    TJsonValue ToJson(const NPrivate::TReninsBoolean<T>& object) {
        return NPrivate::ReninsBooleanToJson(object.Value);
    }

    template <typename T, std::enable_if_t<std::is_same<std::decay_t<T>, TString>::value, int> = 0>
    TJsonValue ToJson(const NPrivate::TReninsDate<T>& object) {
        return ToJson(::ToString(object.Value));  // TString basically
    }

    template <typename T, std::enable_if_t<std::is_same<std::decay_t<T>, TInstant>::value, int> = 0>
    TJsonValue ToJson(const NPrivate::TReninsDate<T>& object) {
        return ToJson(NUtil::FormatDatetime(object.Value, NDrive::NRenins::DefaultDateFormat));
    }

    template <typename T>
    TJsonValue ToJson(const NPrivate::TReninsDateTime<T>& object) {
        return NPrivate::ReninsDateTimeToJson(object.Value);
    }

    template <typename T>
    TJsonValue ToJson(const NPrivate::TInnerKeyValue<T>& object) {
        TJsonValue result = JSON_ARRAY;
        for (auto&&[key, value] : object.Container) {
            result.AppendValue(ToJson(value));
        }
        return result;
    }

    template <typename T>
    bool TryFromJson(const NJson::TJsonValue& value, NPrivate::TReninsPhoneNumber<T>&& result) {
        return NPrivate::TryReninsPhoneNumberFromJson(value, result.Value);
    }

    template <typename T>
    bool TryFromJson(const NJson::TJsonValue& value, NPrivate::TReninsBoolean<T>&& result) {
        return NPrivate::TryReninsBooleanFromJson(value, result.Value);
    }

    template <typename T, std::enable_if_t<std::is_same<T, TString>::value, int> = 0>
    bool TryFromJson(const NJson::TJsonValue& value, NPrivate::TReninsDate<T>&& result) {
        if (!value.IsString()) {
            return false;
        }
        result.Value = value.GetString();
        return true;
    }

    template <typename T, std::enable_if_t<std::is_same<T, TInstant>::value, int> = 0>
    bool TryFromJson(const NJson::TJsonValue& value, NPrivate::TReninsDate<T>&& result) {
        if (!value.IsString()) {
            return false;
        }
        result.Value = NUtil::ParseFomattedLocalDatetime(value.GetString(), NDrive::NRenins::DefaultDateFormat);
        return true;
    }

    template <typename T>
    bool TryFromJson(const NJson::TJsonValue& value, NPrivate::TReninsDateTime<T>&& result) {
        return NPrivate::TryReninsDateTimeFromJson(value, result.Value);
    }

    template <typename T>
    bool TryFromJson(const NJson::TJsonValue& value, NPrivate::TInnerKeyValue<T>&& result) {
        if (!value.IsArray()) {
            return false;
        }
        for (auto&& element : value.GetArray()) {
            typename T::key_type key;
            typename T::mapped_type value;
            if (!element.IsMap() || !TryFromJson(element[result.Key], key) || !TryFromJson(element, value)) {
                return false;
            }
            result.Container.emplace(std::move(key), std::move(value));
        }
        return true;
    }
}
