#pragma once

#include <library/cpp/object_factory/object_factory.h>

#include <rtline/util/types/accessor.h>
#include <rtline/util/types/messages_collector.h>

#include <util/generic/set.h>
#include <util/generic/vector.h>

/*
Interfaces below are intended to summarize common principles dealing with action flow.

1. A complex object is supposed to be in a state
2. There are multiple transitions to other states (moreover, multiple transitions from one state to another are allowed)
3. States are not fixed, can be constructed and added dynamically and not limited to be initial or terminal.
4. Source and destination states are proposed to be different regardless error executing a transition.
  An explicit error state simplified error processing.
*/

namespace NDrive {
    class TEntitySession;
}

namespace NDrive::NState {
    // Encloses an object to be modified
    class IContext {
    public:
        virtual ~IContext() = default;

        // Current object state (executing transitions are ignored as they should be managed externally)
        virtual TString GetCurrentState() const = 0;
    };

    template <typename TContext, typename = std::enable_if_t<std::is_base_of<IContext, TContext>::value>>
    class ITransition {
    public:
        using TContextType = TContext;

        virtual ~ITransition() = default;

        virtual TString GetType() const = 0;

        virtual TSet<TString> GetAllowedSourceStates() const = 0;

        virtual TString GetDestinationState(const TContextType& context) const = 0;

        // Check state and internal conditions
        TMaybe<bool> IsApplicable(const TContextType& context, NDrive::TEntitySession& session) const {
            if (!GetAllowedSourceStates().contains(context.GetCurrentState())) {
                return false;
            }
            if (auto isApplicable = DoCheckIsApplicable(context, session)) {
                return *isApplicable;
            }
            return {};
        }

        // Perform an action and changes state
        virtual bool Perform(TContextType& context, NDrive::TEntitySession& session) const = 0;

    private:
        virtual TMaybe<bool> DoCheckIsApplicable(const TContextType& context, NDrive::TEntitySession& session) const = 0;
    };

    template <typename TTransitionImpl, typename = std::enable_if_t<std::is_base_of<ITransition<typename TTransitionImpl::TContextType>, TTransitionImpl>::value>>
    class IManager {
    public:
        using TContextType = typename TTransitionImpl::TContextType;

        using TTransitionPtr = TAtomicSharedPtr<TTransitionImpl>;
        using TTransitions = TVector<TTransitionPtr>;

        // Use template argument to
        //  1) provide distinct state managers for distinct transition groups and
        //  2) avoid template product implementation
        using TFactory = NObjectFactory::TParametrizedObjectFactory<TTransitionImpl, TString>;

        TMaybe<TTransitions> GetAvailableTransitions(const TContextType& context, NDrive::TEntitySession& session, const TSet<TString>& idFilter = {}) const {
            TTransitions transitions;

            TSet<TString> availableTransitions;
            TFactory::GetRegisteredKeys(availableTransitions);

            for (auto&& key: availableTransitions) {
                TTransitionPtr transition = TFactory::Construct(key);
                if (!idFilter || idFilter.contains(transition->GetType())) {
                    auto isApplicable = transition->IsApplicable(context, session);
                    if (!isApplicable) {
                        return {};
                    }
                    if (*isApplicable) {
                        transitions.push_back(transition);
                    }
                }
            }

            return transitions;
        }

        TMaybe<TSet<TString>> GetAvailableTransitionNames(const TContextType& context, NDrive::TEntitySession& session) const {
            TSet<TString> typeNames;
            auto transitions = GetAvailableTransitions(context, session);
            if (!transitions) {
                return {};
            }
            for (auto&& transition: *transitions) {
                typeNames.emplace(transition->GetType());
            }
            return typeNames;
        }
    };
}
