#include "states.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/transaction/tx.h>

namespace NDrive {
    TIncidentStateContext::TIncidentStateContext(const NDrive::IServer* server, TMaybe<TIncidentData> incidentInstance, const TString& performerId)
        : Server(server)
        , Instance(incidentInstance)
        , PerformerId(performerId)
    {
    }

    EIncidentStatus TIncidentStateContext::GetCurrentStatus() const {
        return Instance ? Instance->GetStatus() : EIncidentStatus::Undefined;
    }

    TString TIncidentStateContext::GetCurrentState() const {
        return ::ToString(GetCurrentStatus());
    }

    NDrive::TScheme IBaseIncidentTransition::GetScheme(const TIncidentStateContext& /* context */) const {
        return {};
    }

    bool IBaseIncidentTransition::Initialize(const NJson::TJsonValue& /* data */, const TIncidentStateContext& /* context */, TMessagesCollector& /* errors */) {
        return true;
    }

    TSet<TString> IBaseIncidentTransition::GetAllowedSourceStates() const {
        TSet<TString> allowedSourceStates;
        for (const auto& status: GetAllowedSourceStatuses()) {
            allowedSourceStates.emplace(::ToString(status));
        }
        return allowedSourceStates;
    }

    TString IBaseIncidentTransition::GetDestinationState(const TIncidentStateContext& context) const {
        return ::ToString(GetDestinationStatus(context));
    }

    bool IBaseIncidentTransition::Perform(TIncidentStateContext& context, NDrive::TEntitySession& session) const {
        if (!context.HasInstance()) {
            return false;
        }
        const bool isPerformSuccessful = DoPerform(context, session);
        context.SetIsPerformSuccessful(isPerformSuccessful);
        context.MutableInstance()->SetStatus(GetDestinationStatus(context));
        return isPerformSuccessful;
    }

    TMaybe<bool> IBaseIncidentTransition::DoCheckIsApplicable(const TIncidentStateContext& /* context */, NDrive::TEntitySession& /* session */) const {
        return true;
    }

    bool IBaseIncidentTransition::DoPerform(TIncidentStateContext& /* context */, NDrive::TEntitySession& /* session */) const {
        return true;
    }

    TIncidentStateManager::TTransitionPtr TIncidentStateManager::ConstructTransition(const TString& transitionType, const NJson::TJsonValue& transitionData, const TIncidentStateContext& context, TMessagesCollector& errors) const {
        TIncidentStateManager::TTransitionPtr transitionPtr = TFactory::Construct(transitionType);
        if (!transitionPtr || !transitionPtr->Initialize(transitionData, context, errors)) {
            return nullptr;
        }
        return transitionPtr;
    }
}
