#pragma once

#include "constants.h"
#include "incident.h"
#include "state.h"

#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/json/json_value.h>

#include <rtline/util/types/messages_collector.h>

#include <util/string/builder.h>
#include <util/string/cast.h>

namespace NDrive {
    class IServer;
    class TEntitySession;
}

namespace NDrive {
    #define FACTORY_INCIDENT_TRANSITION(ClassName, EnumValue)                               \
    public:                                                                                 \
        using TRegistrator = TIncidentStateManager::TFactory::TRegistrator<ClassName>;      \
        static TString GetTypeName() {                                                      \
            return ::ToString(EnumValue);                                                   \
        }                                                                                   \
        virtual TString GetType() const override {                                          \
            return ::ToString(EnumValue);                                                   \
        }                                                                                   \
        virtual EIncidentTransition GetTransitionType() const override {                    \
            return (EnumValue);                                                             \
        }                                                                                   \
    private:                                                                                \
        static TRegistrator Registrator;

    class TIncidentStateContext: public NDrive::NState::IContext {
    public:
        TIncidentStateContext(const NDrive::IServer* server, TMaybe<TIncidentData> incidentInstance, const TString& performerId);

        const NDrive::IServer* GetServer() const noexcept {
            return Server;
        }

        EIncidentStatus GetCurrentStatus() const;
        virtual TString GetCurrentState() const override;

    private:
        const NDrive::IServer* Server;

        R_OPTIONAL(TIncidentData, Instance);
        R_FIELD(TString, PerformerId);
        R_OPTIONAL(bool, IsPerformSuccessful);
    };

    class IBaseIncidentTransition: public NDrive::NState::ITransition<TIncidentStateContext> {
    public:
        // Provide additional settings and context
        virtual NDrive::TScheme GetScheme(const TIncidentStateContext& context) const;
        virtual bool Initialize(const NJson::TJsonValue& data, const TIncidentStateContext& context, TMessagesCollector& errors);

        virtual EIncidentTransition GetTransitionType() const = 0;

        virtual TSet<EIncidentStatus> GetAllowedSourceStatuses() const = 0;
        virtual EIncidentStatus GetDestinationStatus(const TIncidentStateContext& context) const = 0;

        virtual TSet<TString> GetAllowedSourceStates() const override final;
        virtual TString GetDestinationState(const TIncidentStateContext& context) const override final;

        virtual bool Perform(TIncidentStateContext& context, NDrive::TEntitySession& session) const override final;

    private:
        virtual TMaybe<bool> DoCheckIsApplicable(const TIncidentStateContext& context, NDrive::TEntitySession& session) const override;
        virtual bool DoPerform(TIncidentStateContext& context, NDrive::TEntitySession& session) const;
    };

    class TIncidentStateManager: public NDrive::NState::IManager<IBaseIncidentTransition> {
    public:
        TTransitionPtr ConstructTransition(const TString& transitionType, const NJson::TJsonValue& transitionData, const TIncidentStateContext& context, TMessagesCollector& errors) const;
    };
}
