#include "apply_renins_kasko_claim.h"

#include <drive/backend/incident/contexts/renins_kasko_claim.h>

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/incident/renins_claims/client.h>
#include <drive/backend/incident/renins_claims/claim_entity_kasko.h>
#include <drive/backend/incident/renins_claims/common_entity.h>
#include <drive/backend/user_document_photos/manager.h>

namespace {
    void TransferIncidentLinks(const NDrive::TIncidentData& incident, const NDrive::IServer& server, NDrive::NRenins::TKaskoClaimEntry& claimEntry) {
        TVector<NDrive::NRenins::TDocument> documents;

        for (const auto& documentLink: incident.GetDocumentLinks()) {
            documents.emplace_back(NDrive::NRenins::TDocument(documentLink.GetDocumentUrl(), documentLink.GetAttachmentCode(), documentLink.GetDocumentType()));
        }

        for (const auto& photoLink: incident.GetPhotoLinks()) {
            TString photoUrl = photoLink.GetImageUrl();
            if (photoUrl) {
                documents.emplace_back(NDrive::NRenins::TDocument(photoUrl, photoLink.GetAttachmentCode(), ""));
            }
        }

        for (const auto& link: incident.GetUserDocumentPhotoLinks()) {
            TString data;
            Y_ENSURE_BT(server.GetDriveAPI()->GetDocumentPhotosManager().GetDocumentPhoto(link.GetDocumentId(), data, server));
            documents.emplace_back()
                .SetAttachmentCode(link.GetAttachmentCode())
                .SetData(data)
            ;
        }

        NDrive::NRenins::TDocuments documentsInfo;
        documentsInfo.SetDocuments(std::move(documents));
        claimEntry.SetDocuments(std::move(documentsInfo));
    }

    void UpdateIncidentLinksSentTimestamp(NDrive::TIncidentData& incident) {
        const auto timestamp = Now();

        for (auto&& documentLink: incident.MutableDocumentLinks()) {
            documentLink.SetLastSentAt(timestamp);
        }

        for (auto&& photoLink: incident.MutablePhotoLinks()) {
            photoLink.SetLastSentAt(timestamp);
        }
    }
}

namespace NDrive {
    TApplyReninsKaskoClaimIncidentTransition::TRegistrator TApplyReninsKaskoClaimIncidentTransition::Registrator;

    bool TApplyReninsKaskoClaimIncidentTransition::Initialize(const NJson::TJsonValue& /* data */, const TIncidentStateContext& context, TMessagesCollector& errors) {
        if (!context.GetServer()->GetDriveAPI()->HasReninsClaimClient()) {
            errors.AddMessage(__LOCATION__, "No renins claim client configured");
            return false;
        }

        if (!context.HasInstance()) {
            errors.AddMessage(__LOCATION__, "No incident instance provided");
            return false;
        }

        if (!context.GetInstanceRef().HasContext<TReninsKaskoClaimIncidentContext>()) {
            errors.AddMessage(__LOCATION__, "Incident has no required context: " + ::ToString(TReninsKaskoClaimIncidentContext::GetTypeName()));
            return false;
        }

        return true;
    }

    TSet<EIncidentStatus> TApplyReninsKaskoClaimIncidentTransition::GetAllowedSourceStatuses() const {
        return { EIncidentStatus::ObjectTagsProcessed, EIncidentStatus::KaskoClaimApplied, EIncidentStatus::KaskoClaimApplyError };
    }

    EIncidentStatus TApplyReninsKaskoClaimIncidentTransition::GetDestinationStatus(const TIncidentStateContext& context) const {
        return (context.GetIsPerformSuccessfulDef(false)) ? EIncidentStatus::KaskoClaimApplied : EIncidentStatus::KaskoClaimApplyError;
    }

    TMaybe<bool> TApplyReninsKaskoClaimIncidentTransition::DoCheckIsApplicable(const TIncidentStateContext& context, NDrive::TEntitySession& /* session */) const {
        return (context.HasInstance()) ? (context.OptionalInstance()->GetIncidentType() != EIncidentType::Evacuation) : false;
    }

    bool TApplyReninsKaskoClaimIncidentTransition::DoPerform(TIncidentStateContext& context, NDrive::TEntitySession& session) const {
        const auto& server = *Yensured(context.GetServer());
        const auto& reninsClient = server.GetDriveAPI()->GetReninsClaimClient();

        auto alwaysTransferIncidentLinks = server.GetSettings().GetValue<bool>(
            JoinSeq(".", { IncidentTransitionsSettingPrefix, GetType(), "always_transfer_incident_links" })
        ).GetOrElse(true);
        auto claimContextPtr = context.GetInstanceRef().GetContext<TReninsKaskoClaimIncidentContext>();
        if (!claimContextPtr) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "incorrect context type: expected TReninsKaskoClaimIncidentContext", EDriveSessionResult::InternalError);
            return false;
        }
        auto claimEntryPtr = claimContextPtr->GetClaimEntryPtr();
        if (!claimEntryPtr) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "no claim entry", EDriveSessionResult::InternalError);
            return false;
        }

        NDrive::NRenins::TKaskoClaimEntry& claimEntry = *claimEntryPtr;
        if (claimEntry.IsInitialClaimApplication() || alwaysTransferIncidentLinks) {
            TransferIncidentLinks(context.GetInstanceRef(), server, claimEntry);  // documents to be updated in a separate transition
        }

        TMessagesCollector errors;
        if (!reninsClient.ReportKaskoSync(claimEntry, errors)) {
            session.SetErrorInfo(TIncidentData::GetTableName(), errors.GetStringReport(), EDriveSessionResult::InternalError);
            return false;
        }

        // upsert with result (e.g. claim number) and data filled
        context.MutableInstance()->UpsertContext(claimContextPtr);
        UpdateIncidentLinksSentTimestamp(context.MutableInstanceRef());

        return true;
    }
}
