#include "fetch_related_documents.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/car_attachments/registry/registry.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/history/event.h>
#include <drive/backend/database/transaction/tx.h>
#include <drive/backend/device_snapshot/snapshots/image.h>
#include <drive/backend/images/database.h>
#include <drive/backend/incident/contexts/renins_kasko_claim.h>
#include <drive/backend/incident/renins_claims/claim_entity_kasko.h>
#include <drive/backend/tags/history.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/user_document_photos/manager.h>

#include <drive/library/cpp/mds/client.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/cast.h>
#include <rtline/util/algorithm/container.h>

#include <util/generic/algorithm.h>
#include <util/string/split.h>

namespace NDrive {
    TFetchRelatedDocumentsIncidentTransition::TRegistrator TFetchRelatedDocumentsIncidentTransition::Registrator;

    bool TFetchRelatedDocumentsIncidentTransition::Initialize(const NJson::TJsonValue& /* data */, const TIncidentStateContext& context, TMessagesCollector& errors) {
        if (!context.HasInstance()) {
            errors.AddMessage(__LOCATION__, "No incident instance provided");
            return false;
        }

        const TString settingPrefix = GenericIncidentSettingPrefix + ".document_fetcher";
        const auto& settings = context.GetServer()->GetSettings();

        if (settings.GetValueDef<bool>(settingPrefix + ".do_fetch_images_from_linked_tags_only", false)) {
            for (const auto& tagLink: context.GetInstanceRef().GetTagLinks()) {
                if (tagLink.GetTagEntityType() == NEntityTagsManager::EEntityType::Car) {
                    RelatedDocumentTagIds.insert(tagLink.GetTagId());
                }
            }
        } else {
            if (!NJson::TryFromJson(settings.GetJsonValue(settingPrefix + ".related_document_tag_names"), RelatedDocumentTagNames)) {
                errors.AddMessage(__LOCATION__, "Cannot restore tag names to fetch from");
                return false;
            }
        }

        if (!settings.GetValue(settingPrefix + ".default_attachment_code", DefaultAttachmentCode)) {
            errors.AddMessage(__LOCATION__, "Default attachment code is not set");
            return false;
        }

        if (!NJson::TryFromJson(settings.GetJsonValue(settingPrefix + ".attachment_code_mapping"), NJson::Dictionary(AttachmentCodeMapping))) {
            errors.AddMessage(__LOCATION__, "Default attachment code is not set");
            return false;
        }

        return true;
    }

    TSet<EIncidentStatus> TFetchRelatedDocumentsIncidentTransition::GetAllowedSourceStatuses() const {
        return { EIncidentStatus::ObjectTagsProcessed, EIncidentStatus::KaskoClaimApplied, EIncidentStatus::KaskoClaimApplyError };
    }

    EIncidentStatus TFetchRelatedDocumentsIncidentTransition::GetDestinationStatus(const TIncidentStateContext& context) const {
        return context.GetCurrentStatus();
    }

    TMaybe<bool> TFetchRelatedDocumentsIncidentTransition::DoCheckIsApplicable(const TIncidentStateContext& context, NDrive::TEntitySession& /* session */) const {
        return (context.HasInstance()) ? (context.OptionalInstance()->GetIncidentType() != EIncidentType::Evacuation) : false;
    }

    bool TFetchRelatedDocumentsIncidentTransition::DoPerform(TIncidentStateContext& context, NDrive::TEntitySession& session) const {
        auto server = Yensured(context.GetServer());
        auto api = Yensured(server->GetDriveAPI());
        auto& instance = context.MutableInstanceRef();
        auto reninsKaskoClaimContext = instance.GetContext<TReninsKaskoClaimIncidentContext>();
        auto reninsKaskoClaim = reninsKaskoClaimContext ? reninsKaskoClaimContext->GetClaimEntryPtr() : nullptr;

        const TString& carId = instance.GetCarId();
        if (!carId) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "No car id assigned to the incident", EDriveSessionResult::InconsistencySystem);
            return false;
        }

        TImageIds imageIds;
        if (!GetRelatedImageIds(context, imageIds, session)) {
            return false;
        }

        const auto& imagesStorage = api->GetImagesDB();
        auto images = imagesStorage.GetRecords(MakeVector(imageIds), session);
        if (!images) {
            return false;
        }

        for (auto&& image: *images) {
            TString imageUri = api->GetMDSClient().GetTmpFilePath("carsharing-tags", image.GetPath());

            TString attachmentCode;

            // NB. MetaData parsing is a feasible but not a reliable solution.
            const auto& metaData = image.GetMetaData();
            if (metaData.Has("vin")) {
                attachmentCode = AttachmentCodeMapping.Value("vin", DefaultAttachmentCode);
            } else if (metaData.Has("mileage")) {
                attachmentCode = AttachmentCodeMapping.Value("mileage", DefaultAttachmentCode);
            } else if (metaData.Has("has_documents")) {
                attachmentCode = AttachmentCodeMapping.Value("has_documents", DefaultAttachmentCode);
            } else {
                attachmentCode = DefaultAttachmentCode;
            }

            instance.UpsertPhotoLink(TIncidentPhotoLink(image.GetImageId(), imageUri, attachmentCode, GetTransitionType(), context.GetPerformerId()), image.GetImageId());
        }

        {
            TCarGenericAttachment currentRegistryDocument;
            if (!api->GetCarAttachmentAssignments().TryGetAttachmentOfType(carId, EDocumentAttachmentType::CarRegistryDocument, currentRegistryDocument)) {
                session.SetErrorInfo(TIncidentData::GetTableName(), "No registry entry found for car " + carId, EDriveSessionResult::InternalError);
                return false;
            }

            auto baseDocuments = dynamic_cast<const TCarRegistryDocument*>(currentRegistryDocument.Get());
            if (!baseDocuments) {
                session.SetErrorInfo(TIncidentData::GetTableName(), "No registry entry found for car " + carId, EDriveSessionResult::InternalError);
                return false;
            }

            // NB. There is no document id and type specification currently as no specific intent exists besides visualization
            const TString registrationDocumentId = baseDocuments->GetRegistrationMDSKey();
            constexpr TStringBuf registrationDocumentType = "STS";

            // NB. No specific need to parameterize bucket and attachment code
            const TString registrationDocumentUri = api->GetMDSClient().GetTmpFilePath("carsharing-car-documents", registrationDocumentId);
            constexpr TStringBuf attachmentCode = "D020";

            TIncidentDocumentLink documentLink;
            documentLink.SetDocumentId(registrationDocumentId)
                        .SetDocumentType(registrationDocumentType)
                        .SetDocumentUrl(registrationDocumentUri)
                        .SetAttachmentCode(attachmentCode)
                        .SetSourceTransitionId(GetTransitionType())
                        .SetOriginatorId(context.GetPerformerId())
                        .SetLastModifiedAt(Now());

            auto existingLinkIt = FindIf(instance.MutableDocumentLinks(), [&registrationDocumentType](const auto& existingLink){ return registrationDocumentType == existingLink.GetDocumentType(); });
            if (existingLinkIt != instance.MutableDocumentLinks().end()) {
                *existingLinkIt = documentLink;
            } else {
                instance.MutableDocumentLinks().push_back(documentLink);
            }

            if (reninsKaskoClaim && reninsKaskoClaim->GetClaim().IsOnTheMove()) {
                documentLink.SetAttachmentCode("D045");
                documentLink.SetDocumentId("D045_" + documentLink.GetDocumentId());
                documentLink.SetDocumentType("STS_D045");
                existingLinkIt = FindIf(instance.MutableDocumentLinks(), [](const auto& existingLink) {
                    return existingLink.GetDocumentType() == "STS_D045";
                });
                if (existingLinkIt != instance.MutableDocumentLinks().end()) {
                    *existingLinkIt = documentLink;
                } else {
                    instance.MutableDocumentLinks().push_back(documentLink);
                }
            }
        }

        if (api->HasDocumentPhotosManager()) {
            const auto& userDocumentsManager = api->GetDocumentPhotosManager();
            TSet<NUserDocument::EType> documentTypes;
            if (reninsKaskoClaim && reninsKaskoClaim->GetClaim().IsOnTheMove()) {
                documentTypes.insert(NUserDocument::LicenseFront);
                documentTypes.insert(NUserDocument::LicenseBack);
            }
            for (auto&& documentType : documentTypes) {
                auto photoFetchResult = userDocumentsManager.GetUserPhotosDB().GetPhotosForUsers({ instance.GetUserId() }, documentType);
                auto bestPhotoId = TString();
                auto bestPhotoTimestamp = TInstant::Zero();
                for (auto&& [id, photo] : photoFetchResult) {
                    if (photo.GetVerificationStatus() != NUserDocument::Ok) {
                        continue;
                    }
                    if (bestPhotoTimestamp < photo.GetSubmittedAt()) {
                        bestPhotoTimestamp = photo.GetSubmittedAt();
                        bestPhotoId = id;
                    }
                }
                if (!bestPhotoId) {
                    session.SetErrorInfo(
                        "FetchRelatedDocumentsIncidentTransition::DoPerform",
                        TStringBuilder() << "Ok " << documentType << " photo cannot be found"
                    );
                    return false;
                }
                TUserDocumentPhotoLink link;
                link.SetAttachmentCode("D045");
                link.SetDocumentId(bestPhotoId);
                link.SetDocumentType(documentType);
                auto existing = FindIf(instance.MutableUserDocumentPhotoLinks(), [&](const auto& existingLink) {
                    return existingLink.GetDocumentType() == documentType;
                });
                if (existing != instance.MutableUserDocumentPhotoLinks().end()) {
                    *existing = std::move(link);
                } else {
                    instance.MutableUserDocumentPhotoLinks().push_back(std::move(link));
                }
            }
        }

        return true;
    }

    bool TFetchRelatedDocumentsIncidentTransition::GetRelatedImageIds(const TIncidentStateContext& context, TImageIds& imageIds, NDrive::TEntitySession& session) const {
        // NB. Tag is supposed to be linked during incident creation, so tag ids could be determined directly - to be discussed and simplified

        const auto& instance = context.GetInstanceRef();

        TSet<TString> incidentRelatedIssues;
        for (const auto& ticketLink: instance.GetStartrekTicketLinks()) {
            incidentRelatedIssues.insert(ticketLink.GetTicketKey());
        }

        const auto& tagsManagerImpl = context.GetServer()->GetDriveAPI()->GetTagsManager().GetDeviceTags();

        IEntityTagsManager::TQueryOptions queryOptions;
        queryOptions.SetActions({EObjectHistoryAction::AddSnapshot});
        if (RelatedDocumentTagNames) {
            queryOptions.SetTags(RelatedDocumentTagNames);
        }
        if (RelatedDocumentTagIds) {
            queryOptions.SetTagIds(RelatedDocumentTagIds);
        }

        auto events = tagsManagerImpl.GetEventsByObject(instance.GetCarId(), session, 0, TInstant::Zero(), queryOptions);
        if (!events) {
            return false;
        }

        for (auto&& event: *events) {
            // check tag related issue only if tag ids not specified
            if (!RelatedDocumentTagIds && IsIntersectionEmpty(incidentRelatedIssues, GetEventRelatedIssues(event))) {
                continue;
            }

            const auto* snapshotPtr = event->GetObjectSnapshotAs<TImagesSnapshot>();
            if (snapshotPtr) {
                for (auto&& image: snapshotPtr->GetImages()) {
                    imageIds.insert(*image.ImageId);
                }
            }
        }

        return true;
    }

    TSet<TString> TFetchRelatedDocumentsIncidentTransition::GetEventRelatedIssues(const TTagHistoryEvent& event) const {
        // the better way is to broadcast links to all tags

        auto deviceRepairTag = event.GetTagAs<TRepairTagRecord>();
        if (deviceRepairTag) {
            return {deviceRepairTag->GetTicketNumber()};
        }

        TSet<TString> relatedIssues;

        TString comment = event.GetData()->GetComment();
        for (TStringBuf line: StringSplitter(comment).SplitBySet("\r\n").SkipEmpty()) {
            auto possibleIssue = line.substr(line.find_last_of("/") + 1);
            if (possibleIssue.Contains('-')) {
                relatedIssues.insert(TString{possibleIssue});
            }
        }

        return relatedIssues;
    }
}
