#include "fetch_renins_kasko_defaults.h"

#include <drive/backend/incident/contexts/incident_ticket.h>
#include <drive/backend/incident/contexts/renins_kasko_claim.h>

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/car_attachments/documents/insurance.h>
#include <drive/backend/car_attachments/registry/registry.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/private_data.h>
#include <drive/backend/incident/renins_claims/catalogue_entity_kasko.h>
#include <drive/backend/incident/renins_claims/claim_entity_kasko.h>
#include <drive/backend/incident/renins_claims/constants.h>
#include <drive/backend/users/user.h>

#include <rtline/library/json/cast.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/string/join.h>

namespace NDrive {
    TFetchReninsKaskoDefaultsIncidentTransition::TRegistrator TFetchReninsKaskoDefaultsIncidentTransition::Registrator;

    TSet<EIncidentStatus> TFetchReninsKaskoDefaultsIncidentTransition::GetAllowedSourceStatuses() const {
        return { EIncidentStatus::ObjectTagsProcessed };
    }

    EIncidentStatus TFetchReninsKaskoDefaultsIncidentTransition::GetDestinationStatus(const TIncidentStateContext& context) const {
        return context.GetCurrentStatus();
    }

    bool TFetchReninsKaskoDefaultsIncidentTransition::Initialize(const NJson::TJsonValue& /* data */, const TIncidentStateContext& context, TMessagesCollector& errors) {
        if (!context.HasInstance()) {
            errors.AddMessage(__LOCATION__, "No incident instance provided");
            return false;
        }

        ContextPtr = context.OptionalInstance()->GetContext<TReninsKaskoClaimIncidentContext>();
        if (!ContextPtr) {
            ContextPtr = MakeAtomicShared<TReninsKaskoClaimIncidentContext>();
        }

        return true;
    }

    TMaybe<bool> TFetchReninsKaskoDefaultsIncidentTransition::DoCheckIsApplicable(const TIncidentStateContext& context, NDrive::TEntitySession& /* session */) const {
        if (!context.HasInstance() || context.OptionalInstance()->GetIncidentType() == EIncidentType::Evacuation) {
            return false;
        }

        const auto& instance = context.GetInstanceRef();

        auto checkOnceExecution = context.GetServer()->GetSettings().GetValue<bool>(JoinSeq(".", { IncidentTransitionsSettingPrefix, GetType(), "check_once_execution" }));
        if (checkOnceExecution.GetOrElse(true) && instance.HasContext<TReninsKaskoClaimIncidentContext>()) {
            return false;
        }

        if (!instance.GetCarId() || !instance.GetUserId()) {
            return false;
        }

        {
            NDrive::TInfoEntitySession tx;
            TCarInsurancePolicy insurancePolicy;
            if (!FillPolicy(context, insurancePolicy, tx)) {
                return false;
            }
        }

        return true;
    }

    bool TFetchReninsKaskoDefaultsIncidentTransition::DoPerform(TIncidentStateContext& context, NDrive::TEntitySession& session) const {
        TCarInsurancePolicy insurancePolicy;  // kasko
        NDrive::NRenins::TKaskoPolicy driverPolicy;  // osago

        auto claimEntryPtr = ContextPtr->GetClaimEntryPtr();
        if (!FillGeneralClaimInfo(context, claimEntryPtr->MutableGeneralClaimInfo(), session) ||
            !FillPolicy(context, insurancePolicy, session) ||
            !FillVehicleDescription(context, claimEntryPtr->MutableVehicleDescription(), driverPolicy, session) ||
            !FillClaim(context, insurancePolicy, claimEntryPtr->MutableClaim(), session) ||
            !FillDeclarant(context, claimEntryPtr->MutableDeclarant(), session) ||
            !FillParticipants(context, driverPolicy, claimEntryPtr->MutableParticipants(), session)
        ) {
            return false;
        }

        context.MutableInstance()->UpsertContext(ContextPtr);
        return true;
    }

    bool TFetchReninsKaskoDefaultsIncidentTransition::FillGeneralClaimInfo(const TIncidentStateContext& /* context */, NDrive::NRenins::TKaskoGeneralClaimInfo& generalClaimInfo, NDrive::TEntitySession& /* session */) const {
        generalClaimInfo.SetPartnerRole("Ремонт");
        generalClaimInfo.SetDoRegisterClaim(true);
        return true;
    }

    bool TFetchReninsKaskoDefaultsIncidentTransition::FillPolicy(const TIncidentStateContext& context, TCarInsurancePolicy& insurancePolicy, NDrive::TInfoEntitySession& session) const {
        TString carId = context.GetInstanceRef().GetCarId();
        if (!carId) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "No car id assigned to the incident", EDriveSessionResult::InconsistencySystem);
            return false;
        }

        const auto& gAttachments = context.GetServer()->GetDriveAPI()->GetCarAttachmentAssignments();
        if (!gAttachments.GetEffectiveInsurancePolicy(carId, Now(), insurancePolicy)) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "Cannot obtain insurance policy for car " + carId, EDriveSessionResult::InternalError);
            return false;
        }

        if (insurancePolicy.GetProvider() != NDrive::EInsuranceProvider::Renins) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "Car insurance policy does not belong to Renins, actual is " + ::ToString(insurancePolicy.GetProvider()), EDriveSessionResult::IncorrectRequest);
            return false;
        }

        return true;
    }

    bool TFetchReninsKaskoDefaultsIncidentTransition::FillClaim(const TIncidentStateContext& context, const TCarInsurancePolicy& insurancePolicy, NDrive::NRenins::TKaskoClaim& claim, NDrive::TEntitySession& /* session */) const {
        claim.SetPolicyNumber(insurancePolicy.GetAgreementNumber());

        {
            auto ticketContextPtr = context.GetInstanceRef().GetContext<TIncidentTicketIncidentContext>();
            if (ticketContextPtr) {
                auto incidentInstant = NJson::TryFromJson<TInstant>(ticketContextPtr->GetData()["incident_instant"]);
                claim.SetIncidentTimestamp(incidentInstant.GetOrElse(Now()));
            } else {
                claim.SetIncidentTimestamp(Now());
            }
        }

        claim.SetInquiryDate(Now());

        if (auto lossCodeInfo = NDrive::NRenins::TKaskoLossCodeEntry::Restore(context.GetServer(), "Столкновение с другим(и) ТС")) {
            claim.SetLossCodeInfo(std::move(*lossCodeInfo));
        }

        claim.SetLocation("Москва");
        claim.SetHasInjuredPersons(false);
        claim.SetHasLostPersons(false);
        claim.SetHasMultipleDamageFaces(false);
        claim.SetRegulationType("Ремонт на СТОА");

        return true;
    }

    bool TFetchReninsKaskoDefaultsIncidentTransition::FillDeclarant(const TIncidentStateContext& context, NDrive::NRenins::TKaskoDeclarant& declarant, NDrive::TEntitySession& /* session */) const {
        auto defaultDeclarantJson = context.GetServer()->GetSettings().GetJsonValue(NDrive::NRenins::KaskoSettingPrefix + ".default_declarant");
        return declarant.DeserializeFromJson(defaultDeclarantJson);
    }

    bool TFetchReninsKaskoDefaultsIncidentTransition::FillParticipants(const TIncidentStateContext& context, const NDrive::NRenins::TKaskoPolicy& driverPolicy, TVector<NDrive::NRenins::TKaskoParticipant>& participants, NDrive::TEntitySession& session) const {
        participants.clear();

        TString userId = context.GetInstanceRef().GetUserId();
        if (!userId) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "No user id assigned to the incident", EDriveSessionResult::InconsistencySystem);
            return false;
        }

        auto gUsers = context.GetServer()->GetDriveAPI()->GetUsersData()->FetchInfo(userId, session);
        if (!gUsers) {
            return false;
        }

        auto userDataPtr = gUsers.GetResultPtr(userId);
        if (!userDataPtr) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "No user found: " + userId, EDriveSessionResult::InconsistencySystem);
            return false;
        }

        NDrive::NRenins::TKaskoParticipant driver;
        driver.SetRoleInAccident("Водитель");

        driver.SetLastName(userDataPtr->GetLastName());
        driver.SetFirstName(userDataPtr->GetFirstName());
        driver.SetMiddleName(userDataPtr->GetPName());

        {
            TUserPassportData passportData;
            if (!context.GetServer()->GetDriveAPI()->GetPrivateDataClient().GetPassportSync(*userDataPtr, userDataPtr->GetPassportDatasyncRevision(), passportData)) {
                session.SetErrorInfo(TIncidentData::GetTableName(), "Unable to fetch user passport data: " + userId, EDriveSessionResult::InternalError);
                return false;
            }

            driver.SetBirthDate(passportData.GetBirthDate());
        }

        driver.SetPolicyOsagoParticipant(driverPolicy);

        participants.push_back(std::move(driver));

        return true;
    }

    bool TFetchReninsKaskoDefaultsIncidentTransition::FillVehicleDescription(const TIncidentStateContext& context, NDrive::NRenins::TKaskoVehicle& vehicleDescription, NDrive::NRenins::TKaskoPolicy& driverPolicy, NDrive::TEntitySession& session) const {
        TString carId = context.GetInstanceRef().GetCarId();
        if (!carId) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "No car id assigned to the incident", EDriveSessionResult::InconsistencySystem);
            return false;
        }

        {
            auto gCars = context.GetServer()->GetDriveAPI()->GetCarsData()->FetchInfo(carId, session);
            if (!gCars) {
                return false;
            }

            auto carDataPtr = gCars.GetResultPtr(carId);
            if (!carDataPtr) {
                session.SetErrorInfo(TIncidentData::GetTableName(), "No car found: " + carId, EDriveSessionResult::InternalError);
                return false;
            }

            vehicleDescription.SetPlateNumber(carDataPtr->GetNumber());
            vehicleDescription.SetVIN(carDataPtr->GetVin());
            vehicleDescription.SetPTSNumber(::ToString(carDataPtr->GetRegistrationID()));
        }

        {
            TCarGenericAttachment currentRegistryDocument;
            if (!context.GetServer()->GetDriveAPI()->GetCarAttachmentAssignments().TryGetAttachmentOfType(carId, EDocumentAttachmentType::CarRegistryDocument, currentRegistryDocument)) {
                session.SetErrorInfo(TIncidentData::GetTableName(), "No registry entry found for car " + carId, EDriveSessionResult::InternalError);
                return false;
            }

            auto baseDocuments = dynamic_cast<const TCarRegistryDocument*>(currentRegistryDocument.Get());
            if (!baseDocuments) {
                session.SetErrorInfo(TIncidentData::GetTableName(), "No registry entry found for car " + carId, EDriveSessionResult::InternalError);
                return false;
            }

            vehicleDescription.SetColor(baseDocuments->GetColor());

            driverPolicy.SetInsuranceCompany("ГРУППА РЕНЕССАНС СТРАХОВАНИЕ");  // renins only, checked fetching insurance policy
            driverPolicy.SetPolicyStartDate(baseDocuments->GetOsagoDate());
            driverPolicy.SetPolicyEndDate(baseDocuments->GetOsagoDateTo());
            driverPolicy.SetPolicyNumber(baseDocuments->GetOsagoNumber());
        }

        return true;
    }
}
