#include "handle_attached_photos.h"

#include <drive/backend/incident/renins_claims/catalogue_entity_kasko.h>

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/common/scheme_adapters.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/images/database.h>

#include <drive/library/cpp/mds/client.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/json/merge.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/container.h>

#include <util/generic/algorithm.h>
#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/string/join.h>

namespace NJson {
    template <>
    bool TryFromJson(const TJsonValue& value, NDrive::THandleAttachedPhotosIncidentTransition::TAction& result) {
        auto unnestedData = NJson::UnnestJson(value);
        return NJson::ParseField(unnestedData["action_type"], NJson::Stringify(result.MutableActionType())) &&
               NJson::ParseField(unnestedData["image_id"], result.MutableImageId()) &&
               NJson::ParseField(unnestedData["attachment_code"], result.MutableAttachmentCode()) &&
               NJson::ParseField(unnestedData["comment"], result.MutableComment());
    }
}

namespace NDrive {
    THandleAttachedPhotosIncidentTransition::TRegistrator THandleAttachedPhotosIncidentTransition::Registrator;

    NDrive::TScheme THandleAttachedPhotosIncidentTransition::TAction::GetScheme(const NDrive::IServer& /* server */, const IVariantsAdapter& variantsAdapter) {
        NDrive::TScheme scheme;
        auto& actionScheme = scheme.Add<TFSVariable>("action_type", "Действие");
        {
            NDrive::TScheme upsertScheme;
            upsertScheme.Add<TFSString>("image_id", "id фотографии").SetRequired(true);
            upsertScheme.Add<TFSVariants>("attachment_code", "Код шаблона").SetCompoundVariants(static_cast<TFSVariants::TCompoundVariants>(variantsAdapter)).SetRequired(true);
            upsertScheme.Add<TFSString>("comment", "Комментарий");
            actionScheme.AddVariant(TLocalizedVariantAdapter::FromEnum(EAttachedPhotosHandlingAction::Upsert), upsertScheme);
        }
        {
            NDrive::TScheme removeScheme;
            removeScheme.Add<TFSString>("image_id", "id фотографии").SetRequired(true);
            actionScheme.AddVariant(TLocalizedVariantAdapter::FromEnum(EAttachedPhotosHandlingAction::Remove), removeScheme);
        }
        actionScheme.MutableCondition().SetDefault(::ToString(EAttachedPhotosHandlingAction::Upsert));
        return scheme;
    }

    NDrive::TScheme THandleAttachedPhotosIncidentTransition::GetScheme(const TIncidentStateContext& context) const {
        auto variantsAdapter = TReninsCatalogueEntryAdapter<NDrive::NRenins::TKaskoDocumentTypeEntry>();

        NDrive::TScheme scheme;
        scheme.Add<TFSArray>("actions", "Действия с фотографиями").SetElement(TAction::GetScheme(*context.GetServer(), variantsAdapter));
        return scheme;
    }

    bool THandleAttachedPhotosIncidentTransition::Initialize(const NJson::TJsonValue& data, const TIncidentStateContext& context, TMessagesCollector& errors) {
        if (!context.HasInstance()) {
            errors.AddMessage(__LOCATION__, "Incident instance is not provided");
            return false;
        }

        if (!NJson::TryFromJson(data["actions"], Actions)) {
            errors.AddMessage(__LOCATION__, "Invalid photo actions format");
            return false;
        }

        return ValidateImageActions(context, errors);
    }

    bool THandleAttachedPhotosIncidentTransition::ValidateImageActions(const TIncidentStateContext& context, TMessagesCollector& errors) const {
        bool isValid = true;

        auto checkImageOwnerMismatch = context.GetServer()->GetSettings().GetValue<bool>(JoinSeq(".", { IncidentTransitionsSettingPrefix, GetType(), "check_image_owner_mismatch" }));

        TSet<TIncidentPhotoLink::TImageId> imageIdsToUpsert;
        for (auto&& action: Actions) {
            if (action.GetActionType() == EAttachedPhotosHandlingAction::Upsert) {
                imageIdsToUpsert.insert(action.GetImageId());
            }
        }

        TSet<TIncidentPhotoLink::TImageId> processedImageIds;
        TVector<TString> imageOwnerMismatchErrorMessages;

        {
            const auto& imagesStorage = context.GetServer()->GetDriveAPI()->GetImagesDB();

            auto session = imagesStorage.BuildSession(/* readonly = */ true);
            auto imageRecords = imagesStorage.GetRecords(MakeVector(imageIdsToUpsert), session);
            if (!imageRecords) {
                errors.MergeMessages(session.GetMessages());
                return false;
            }

            const auto& instance = context.GetInstanceRef();

            for (auto&& image: *imageRecords) {
                TString referenceObjectId;
                if (image.GetObjectType() == NEntityTagsManager::EEntityType::User) {
                    referenceObjectId = instance.GetUserId();
                } else if (image.GetObjectType() == NEntityTagsManager::EEntityType::Car) {
                    referenceObjectId = instance.GetCarId();
                } else {
                    referenceObjectId = {};
                }

                if (checkImageOwnerMismatch.GetOrElse(true) && image.GetObjectId() != referenceObjectId) {
                    imageOwnerMismatchErrorMessages.emplace_back(
                        TStringBuilder() << "Image does not relates to the incident: "
                        << "image id - " << image.GetImageId() << ", "
                        << "object type - " << ::ToString(image.GetObjectType()) << ", "
                        <<  "object id - " << image.GetObjectId()
                    );
                }

                processedImageIds.insert(image.GetImageId());

                TString imageUri = context.GetServer()->GetDriveAPI()->GetMDSClient().GetTmpFilePath("carsharing-tags", image.GetPath());
                ImageUrlMapping.emplace(image.GetImageId(), imageUri);
            }
        }

        if (imageOwnerMismatchErrorMessages) {
            isValid = false;

            for (auto&& errorMessage: imageOwnerMismatchErrorMessages) {
                errors.AddMessage(__LOCATION__, errorMessage);
            }
        }

        if (imageIdsToUpsert.size() != processedImageIds.size()) {
            isValid = false;

            TVector<TIncidentPhotoLink::TImageId> remainedImageIds(std::max(imageIdsToUpsert.size(), processedImageIds.size()));
            remainedImageIds.resize(SetDifference(imageIdsToUpsert.begin(), imageIdsToUpsert.end(), processedImageIds.begin(), processedImageIds.end(), remainedImageIds.begin()) - remainedImageIds.begin());

            errors.AddMessage(__LOCATION__, "Some images cannot be referenced (maybe they has been removed already): " + JoinSeq(", ", remainedImageIds));
        }

        return isValid;
    }

    TSet<EIncidentStatus> THandleAttachedPhotosIncidentTransition::GetAllowedSourceStatuses() const {
        return { EIncidentStatus::ObjectTagsProcessed, EIncidentStatus::KaskoClaimApplied, EIncidentStatus::KaskoClaimApplyError };
    }

    EIncidentStatus THandleAttachedPhotosIncidentTransition::GetDestinationStatus(const TIncidentStateContext& context) const {
        return context.GetCurrentStatus();
    }

    TMaybe<bool> THandleAttachedPhotosIncidentTransition::DoCheckIsApplicable(const TIncidentStateContext& context, NDrive::TEntitySession& /* session */) const {
        return (context.HasInstance()) ? (context.OptionalInstance()->GetIncidentType() != EIncidentType::Evacuation) : false;
    }

    bool THandleAttachedPhotosIncidentTransition::DoPerform(TIncidentStateContext& context, NDrive::TEntitySession& session) const {
        for (auto&& action: Actions) {
            switch (action.GetActionType()) {
            case EAttachedPhotosHandlingAction::Upsert:
                if (!UpsertPhotoLink(action, context, session)) {
                    return false;
                }
                break;
            case EAttachedPhotosHandlingAction::Remove:
                if (!RemovePhotoLink(action, context, session)) {
                    return false;
                }
                break;
            }
        }
        return true;
    }

    bool THandleAttachedPhotosIncidentTransition::UpsertPhotoLink(const TAction& action, TIncidentStateContext& context, NDrive::TEntitySession& /* session */) const {
        const auto& imageId = action.GetImageId();
        TString imageUrl = ImageUrlMapping.Value(imageId, "");  // empty value already checked

        TIncidentPhotoLink link(imageId, imageUrl, action.GetAttachmentCode(), GetTransitionType(), context.GetPerformerId());
        link.SetComment(action.GetComment());

        auto& instance = context.MutableInstanceRef();
        instance.UpsertPhotoLink(std::move(link), imageId);
        return true;
    }

    bool THandleAttachedPhotosIncidentTransition::RemovePhotoLink(const TAction& action, TIncidentStateContext& context, NDrive::TEntitySession& /* session */) const {
        auto& instance = context.MutableInstanceRef();
        instance.RemovePhotoLink(action.GetImageId());
        return true;
    }
}
