#include "handle_avarcom_tags.h"

#include <drive/backend/cars/hardware.h>
#include <drive/backend/car_attachments/documents/insurance.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>

namespace NDrive {
    TSet<EIncidentStatus> IHandleAvarcomTagsTransition::GetAllowedSourceStatuses() const {
        auto result = THandleObjectTagsBaseTransition::GetAllowedSourceStatuses();
        result.insert(EIncidentStatus::ObjectTagsProcessed);
        return result;
    }

    TString IHandleAvarcomTagsTransition::GetSettingPrefix() const {
        return AvarcomTransitionsSettingPrefix + "_" + ToString(GetInsuranceProvider());
    }

    bool IHandleAvarcomTagsTransition::Initialize(const NJson::TJsonValue& data, const TIncidentStateContext& context, TMessagesCollector& errors) {
        if (!TBase::Initialize(data, context, errors)) {
            return false;
        }
        AvarcomUserIds = MakeSet(SplitString(context.GetServer()->GetSettings().GetValueDef<TString>(GetSettingPrefix() + ".avarcom_user_ids", ""), ","));
        return true;
    }

    EIncidentContextType IHandleAvarcomTagsTransition::GetContextType() const {
        return EIncidentContextType::IncidentTicket;
    }

    TMaybe<bool> IHandleAvarcomTagsTransition::DoCheckIsApplicable(const TIncidentStateContext& context, NDrive::TEntitySession& session) const {
        if (AvarcomUserIds.empty()) {
            return false;
        }
        auto& incident = context.OptionalInstance();
        if (!incident || incident->GetIncidentType() != EIncidentType::RoadAccident || !incident->GetCarId()) {
            return false;
        }
        auto contextIncident = incident->GetContext(GetContextType());
        if (!contextIncident) {
            return false;
        }
        auto json = contextIncident->SerializeToJson();
        auto checkField = [&json] (const TString& key) {
            return json.Has(key) && json[key].IsBoolean() && json[key].GetBooleanRobust();
        };
        const bool applicable = checkField("incident_is_other_participant_carsharing")
            || checkField("incident_is_other_participant_motorcycle")
            || checkField("incident_are_other_participant_airbags_deployed")
            || checkField("incident_does_other_participant_get_off_the_road")
            || (json.Has("incident_related_car_models") && !json["incident_related_car_models"].GetArray().empty());
        if (!applicable) {
            return false;
        }
        auto optionalAttachments = context.GetServer()->GetDriveAPI()->GetCarAttachmentAssignments().GetActiveAttachments(incident->GetCarId(), EDocumentAttachmentType::CarInsurancePolicy, session);
        if (!optionalAttachments) {
            return {};
        }
        TCarInsurancePolicy policy;
        for (auto&& element : *optionalAttachments) {
            auto insurance = dynamic_cast<const TCarInsurancePolicy*>(element.Get());
            if (insurance && insurance->GetValidFrom() <= Now() && Now() <= insurance->GetValidUntil()) {
                policy = *insurance;
                break;
            }
        }
        return policy.GetProvider() == GetInsuranceProvider();
    }

    bool IHandleAvarcomTagsTransition::DoPerformTags(TIncidentStateContext& context, const TJsonFetchContext& fetchContext, NDrive::TEntitySession& session) const {
        if (!TBase::DoPerformTags(context, fetchContext, session)) {
            return false;
        }
        const NDrive::IServer* server = context.GetServer();
        TMessagesCollector errors;
        auto userTagsData = ParseTags(server, fetchContext, GetSettingPrefix() + ".avarcom_user_tags_data", errors);
        if (!userTagsData) {
            session.SetErrorInfo(TIncidentData::GetTableName(), errors.GetStringReport(), EDriveSessionResult::IncorrectRequest);
            return false;
        }
        const auto& tagsManager = server->GetDriveAPI()->GetTagsManager().GetUserTags();
        for (auto&& userId : AvarcomUserIds) {
            TSet<TString> processedTagIds;
            if (!AddTags(tagsManager, *userTagsData, context, processedTagIds, session, userId)) {
                return false;
            }
        }
        return true;
    }

    THandleAvarcomReninsTagsTransition::TRegistrator THandleAvarcomReninsTagsTransition::Registrator;

    NDrive::EInsuranceProvider THandleAvarcomReninsTagsTransition::GetInsuranceProvider() const {
        return NDrive::EInsuranceProvider::Renins;
    }

    THandleAvarcomIngosTagsTransition::TRegistrator THandleAvarcomIngosTagsTransition::Registrator;

    NDrive::EInsuranceProvider THandleAvarcomIngosTagsTransition::GetInsuranceProvider() const {
        return NDrive::EInsuranceProvider::Ingos;
    }
}
