#include "handle_object_tags_base.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/data/common/serializable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/tx.h>
#include <drive/backend/tags/tag_description.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/merge.h>
#include <rtline/library/json/parse.h>

#include <util/generic/algorithm.h>

namespace NDrive {
    TSet<EIncidentStatus> THandleObjectTagsBaseTransition::GetAllowedSourceStatuses() const {
        return { EIncidentStatus::StartrekTicketsProcessed, EIncidentStatus::ObjectTagsProcessingError };
    }

    EIncidentStatus THandleObjectTagsBaseTransition::GetDestinationStatus(const TIncidentStateContext& context) const {
        return (context.GetIsPerformSuccessfulDef(false)) ? EIncidentStatus::ObjectTagsProcessed : EIncidentStatus::ObjectTagsProcessingError;
    }

    bool THandleObjectTagsBaseTransition::DoPerform(TIncidentStateContext& context, NDrive::TEntitySession& session) const {
        auto fetchContext = PrepareFetchContext(context, session);
        if (!fetchContext) {
            return false;
        }
        return DoPerformTags(context, *fetchContext, session);
    }

    bool THandleObjectTagsBaseTransition::DoPerformTags(TIncidentStateContext& context, const TJsonFetchContext& fetchContext, NDrive::TEntitySession& session) const {
        const NDrive::IServer* server = context.GetServer();

        TMessagesCollector errors;
        auto carTagsData = ParseTags(server, fetchContext, GetSettingPrefix() + ".car_tags_data", errors);
        auto userTagsData = ParseTags(server, fetchContext, GetSettingPrefix() + ".user_tags_data", errors);
        auto carTagEvolutionsMapping = ParseTagEvolutions(server, fetchContext, GetSettingPrefix() + ".car_tag_evolutions_mapping", errors);
        if (!carTagsData || !userTagsData || !carTagEvolutionsMapping) {
            session.SetErrorInfo(TIncidentData::GetTableName(), errors.GetStringReport(), EDriveSessionResult::IncorrectRequest);
            return false;
        }

        TSet<TString> processedTagIds;

        const auto& tagsManager = server->GetDriveAPI()->GetTagsManager();
        {
            const auto& tagsManagerImpl = tagsManager.GetUserTags();
            if (!AddTags(tagsManagerImpl, *userTagsData, context, processedTagIds, session)) {
                return false;
            }
            if (!RemoveExtraTags(tagsManagerImpl, processedTagIds, context, session)) {
                return false;
            }
        }
        {
            const auto& tagsManagerImpl = tagsManager.GetDeviceTags();
            if (!AddTags(tagsManagerImpl, *carTagsData, context, processedTagIds, session)) {
                return false;
            }
            if (!EvolveTags(tagsManagerImpl, *carTagEvolutionsMapping, context, processedTagIds, session)) {
                return false;
            }
            if (!RemoveExtraTags(tagsManagerImpl, processedTagIds, context, session)) {
                return false;
            }
        }
        return true;
    }

    TMaybe<TJsonFetchContext> THandleObjectTagsBaseTransition::PrepareFetchContext(TIncidentStateContext& context, NDrive::TEntitySession& session) const {
        TIncidentData& incidentInstance = context.MutableInstanceRef();
        const NDrive::IServer& server = *context.GetServer();

        TString relatedStartrekTicketKey, relatedStartrekTicketUrl;
        {
            const auto& ticketLinks = incidentInstance.GetStartrekTicketLinks();
            if (ticketLinks.size() != 1) {
                session.SetErrorInfo(TIncidentData::GetTableName(), "Explected exactly one startrek ticket linked but have " + ::ToString(ticketLinks.size()), EDriveSessionResult::IncorrectRequest);
                return {};
            }
            relatedStartrekTicketKey = ticketLinks.front().GetTicketKey();
            relatedStartrekTicketUrl = ticketLinks.front().GetTicketUrl(server);
        }

        NJson::TJsonValue contextEntry(NJson::JSON_MAP);

        auto ticketContextPtr = incidentInstance.GetContext(GetContextType());
        if (ticketContextPtr) {
            NJson::MergeJson(ticketContextPtr->SerializeToJson(), contextEntry);
        }

        TJsonFetchContext::TDynamicContext dynamicContext = {
            {"performer_id", context.GetPerformerId()},
            {"car_id", incidentInstance.GetCarId()},
            {"user_id", incidentInstance.GetUserId()},
            {"session_id", incidentInstance.GetSessionId()},
            {"startrek.related_ticket_key", relatedStartrekTicketKey},
            {"startrek.related_ticket_url", relatedStartrekTicketUrl}
        };

        return TJsonFetchContext(server, contextEntry, dynamicContext);
    }

    TMaybe<TVector<NJson::TJsonValue>> THandleObjectTagsBaseTransition::ParseTags(const NDrive::IServer* server, const TJsonFetchContext& context, const TString& settingKey, TMessagesCollector& errors) const {
        TVector<NJson::TJsonValue> tagsData;

        auto rawTagsDataJson = server->GetSettings().GetJsonValue(settingKey);
        auto rawTagsData = NJson::TryFromJson<TVector<TString>>(rawTagsDataJson);
        if (!rawTagsData) {
            errors.AddMessage(__LOCATION__, "Invalid tags data: " + rawTagsDataJson.GetStringRobust());
            return {};
        }

        for (auto&& rawTagData: *rawTagsData) {
            auto tagData = ParseTagData(rawTagData, context, errors);
            if (!tagData) {
                return {};
            }
            if (tagData->IsDefined()) {
                tagsData.push_back(std::move(*tagData));
            }
        }

        return tagsData;
    }

    TMaybe<TMap<TString, NJson::TJsonValue>> THandleObjectTagsBaseTransition::ParseTagEvolutions(const NDrive::IServer* server, const TJsonFetchContext& context, const TString& settingKey, TMessagesCollector& errors) const {
        TMap<TString, NJson::TJsonValue> tagEvolutionsMapping;

        auto rawTagsDataJson = server->GetSettings().GetJsonValue(settingKey);

        TMap<TString, TString> rawTagsDataMapping;
        if (!NJson::TryFromJson(rawTagsDataJson, NJson::Dictionary(rawTagsDataMapping))) {
            errors.AddMessage(__LOCATION__, "Invalid tags data: " + rawTagsDataJson.GetStringRobust());
            return {};
        }

        for (auto&& [sourceTagName, targetRawTagData]: rawTagsDataMapping) {
            auto tagData = ParseTagData(targetRawTagData, context, errors);
            if (!tagData) {
                return {};
            }
            if (tagData->IsDefined()) {
                tagEvolutionsMapping.emplace(sourceTagName, std::move(*tagData));
            }
        }

        return tagEvolutionsMapping;
    }

    TMaybe<NJson::TJsonValue> THandleObjectTagsBaseTransition::ParseTagData(const TString& rawTagData, const TJsonFetchContext& context, TMessagesCollector& errors) const {
        TString tagData = IJsonContextFetcher::ProcessText(rawTagData, context, errors);
        if (!tagData) {
            return NJson::JSON_NULL;  // allow some tags to be conditional
        }

        NJson::TJsonValue tagDataJson;
        if (!NJson::ReadJsonTree(tagData, &tagDataJson)) {
            errors.AddMessage(__LOCATION__, "Error parsing tag data json: " + tagData);
            return {};
        }

        return tagDataJson;
    }

    TString THandleObjectTagsBaseTransition::GetRobotUserId(const TIncidentStateContext& context) const {
        return context.GetServer()->GetSettings().GetValueDef<TString>("incident.default_robot_id", "robot-frontend");
    }

    TString THandleObjectTagsBaseTransition::GetTagObjectId(const IEntityTagsManager& tagsManagerImpl, const TIncidentStateContext& context) const {
        TString objectId;
        switch (tagsManagerImpl.GetEntityType()) {
            case NEntityTagsManager::EEntityType::User:
                objectId = context.OptionalInstance()->GetUserId();
                break;
            case NEntityTagsManager::EEntityType::Car:
                objectId = context.OptionalInstance()->GetCarId();
                break;
            default:
                break;
        }
        return objectId;
    }

    bool THandleObjectTagsBaseTransition::AddTags(const IEntityTagsManager& tagsManagerImpl, const TVector<NJson::TJsonValue>& tagsData, TIncidentStateContext& context, TSet<TString>& processedTagIds, NDrive::TEntitySession& session, const TString& entityId /* = "" */) const {
        // NB. 1) Two tags with same name are not allowed. 2) Both tag to evolve from and target tags must be indicated as skipIfExistTags

        const TString& objectId = entityId ? entityId : GetTagObjectId(tagsManagerImpl, context);
        if (!objectId) {
            return true;
        }

        TVector<TDBTag> existingTags;
        if (!tagsManagerImpl.RestoreEntityTags(objectId, {}, existingTags, session)) {
            return false;
        }

        TSet<TString> linkedTagIds;
        for (auto&& tagLink: context.GetInstanceRef().GetTagLinks()) {
            if (GetTransitionType() == tagLink.OptionalSourceTransitionId() && tagLink.GetTagEntityType() == tagsManagerImpl.GetEntityType()) {
                linkedTagIds.emplace(tagLink.GetTagId());
            }
        }

        for (auto&& tagData: tagsData) {
            TSet<TString> skipIfExistTags;
            if (!NJson::ParseField(tagData["skip_if_exist_tags"], skipIfExistTags)) {
                session.SetErrorInfo(TIncidentData::GetTableName(), "Invalid tag data: " + tagData.GetStringRobust(), EDriveSessionResult::IncorrectRequest);
                return false;
            }

            if (AnyOf(existingTags, [&skipIfExistTags](const auto& existingTag){ return skipIfExistTags.contains(existingTag->GetName()); })) {
                continue;
            }

            ITag::TPtr tagPtr = IJsonSerializableTag::BuildFromJson(context.GetServer()->GetDriveAPI()->GetTagsManager(), tagData);
            if (!tagPtr) {
                session.SetErrorInfo(TIncidentData::GetTableName(), "Invalid tag data: " + tagData.GetStringRobust(), EDriveSessionResult::IncorrectRequest);
                return false;
            }

            TMaybe<TDBTag> relatedTag;
            for (auto&& existingTag: existingTags) {
                if (existingTag.GetObjectId() == objectId && linkedTagIds.contains(existingTag.GetTagId()) && tagPtr->GetName() == existingTag->GetName()) {
                    if (existingTag->HasPerformer()) {
                        session.SetErrorInfo(TIncidentData::GetTableName(), "Cannot update tag data as tag is performing: " + existingTag.GetTagId(), EDriveSessionResult::IncorrectRequest);
                        return false;
                    }
                    relatedTag = existingTag;
                    break;
                }
            }

            TString relatedTagId;

            if (!relatedTag) {
                auto addedTags = tagsManagerImpl.AddTag(tagPtr, context.GetPerformerId(), objectId, context.GetServer(), session);
                if (!addedTags || addedTags->empty()) {
                    return false;
                }

                relatedTagId = addedTags->front().GetTagId();
            } else {
                relatedTag->SetData(tagPtr);
                if (!tagsManagerImpl.UpdateTagData(*relatedTag, context.GetPerformerId(), session)) {
                    return false;
                }

                relatedTagId = relatedTag->GetTagId();
            }

            context.MutableInstance()->UpsertTagLink(TIncidentTagLink(relatedTagId, tagsManagerImpl.GetEntityType(), GetTransitionType(), context.GetPerformerId()), relatedTagId);
            processedTagIds.insert(relatedTagId);
        }

        return true;
    }

    bool THandleObjectTagsBaseTransition::EvolveTags(const IEntityTagsManager& tagsManagerImpl, const TMap<TString, NJson::TJsonValue>& tagsDataMapping, TIncidentStateContext& context, TSet<TString>& processedTagIds, NDrive::TEntitySession& session) const {
        auto robotUserPermissions = context.GetServer()->GetDriveAPI()->GetUserPermissions(GetRobotUserId(context), {});
        if (!robotUserPermissions) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "Cannot initialize robot permissions", EDriveSessionResult::InternalError);
        }

        TString objectId = GetTagObjectId(tagsManagerImpl, context);
        if (!objectId) {
            return true;
        }

        TVector<TDBTag> existingTags;
        if (!tagsManagerImpl.RestoreEntityTags(objectId, {}, existingTags, session)) {
            return false;
        }

        for (auto&& [sourceTagName, targetTagData]: tagsDataMapping) {
            ITag::TPtr targetTagPtr = IJsonSerializableTag::BuildFromJson(context.GetServer()->GetDriveAPI()->GetTagsManager(), targetTagData);
            if (!targetTagPtr) {
                session.SetErrorInfo(TIncidentData::GetTableName(), "Invalid tag data: " + targetTagData.GetStringRobust(), EDriveSessionResult::IncorrectRequest);
                return false;
            }

            TMaybe<TDBTag> sourceTag;
            TString targetTagId;

            for (auto&& existingTag: existingTags) {
                if (existingTag->GetName() == sourceTagName) {
                    if (sourceTag) {
                        session.SetErrorInfo(TIncidentData::GetTableName(), "Multiple source tags to evolve are not supported", EDriveSessionResult::IncorrectRequest);
                        return false;
                    }
                    sourceTag = existingTag;
                }
                if (existingTag->GetName() == targetTagPtr->GetName()) {
                    if (targetTagId) {
                        session.SetErrorInfo(TIncidentData::GetTableName(), "Multiple evolution target tags are not supported", EDriveSessionResult::IncorrectRequest);
                        return false;
                    }
                    targetTagId = existingTag.GetTagId();
                }
            }

            if (!sourceTag && !targetTagId) {
                session.SetErrorInfo(TIncidentData::GetTableName(), "Object does not have neither source tags to evolve nor evolution target tags", EDriveSessionResult::IncorrectRequest);
                return false;
            }

            if (sourceTag && targetTagId) {
                session.SetErrorInfo(TIncidentData::GetTableName(), "Both source tag to evolve and evolution target tag are not allowed", EDriveSessionResult::IncorrectRequest);
                return false;
            }

            if (sourceTag) {
                auto existingLink = context.OptionalInstance()->GetTagLink(sourceTag->GetTagId());
                if (existingLink && GetTransitionType() != existingLink->OptionalSourceTransitionId()) {
                    session.SetErrorInfo(TIncidentData::GetTableName(), "Tag to evolve is already linked by another transition", EDriveSessionResult::InconsistencySystem);
                    return false;
                }
                targetTagId = sourceTag->GetTagId();
            } else {
                auto existingLink = context.OptionalInstance()->GetTagLink(targetTagId);
                if (!existingLink || (existingLink && GetTransitionType() != existingLink->OptionalSourceTransitionId())) {
                    session.SetErrorInfo(TIncidentData::GetTableName(), "Another evolution target tag unbound or linked to another transition exists", EDriveSessionResult::InconsistencySystem);
                    return false;
                }
            }

            if (sourceTag) {
                if (!targetTagPtr->CopyOnEvolve(**sourceTag, nullptr, *context.GetServer())) {
                    session.SetErrorInfo(TIncidentData::GetTableName(), "Cannot copy tag data on evolve", EDriveSessionResult::IncorrectRequest);
                    return false;
                }

                const TString& originalComment = (*sourceTag)->GetComment();
                if (originalComment && originalComment != targetTagPtr->GetComment()) {
                    targetTagPtr->SetComment(originalComment + "\r\n" + targetTagPtr->GetComment());
                }

                if (!tagsManagerImpl.EvolveTag(*sourceTag, targetTagPtr, *robotUserPermissions, context.GetServer(), session)) {
                    return false;
                }
            }

            context.MutableInstance()->UpsertTagLink(TIncidentTagLink(targetTagId, tagsManagerImpl.GetEntityType(), GetTransitionType(), context.GetPerformerId()), targetTagId);
            processedTagIds.insert(targetTagId);
        }

        return true;
    }

    bool THandleObjectTagsBaseTransition::RemoveExtraTags(const IEntityTagsManager& tagsManagerImpl, const TSet<TString>& processedTagIds, TIncidentStateContext& context, NDrive::TEntitySession& session) const {
        TSet<TString> tagIdsToRemove;
        for (auto&& tagLink: context.GetInstanceRef().GetTagLinks(GetTransitionType())) {
            if (tagsManagerImpl.GetEntityType() == tagLink.GetTagEntityType() && !processedTagIds.contains(tagLink.GetTagId())) {
                tagIdsToRemove.insert(tagLink.GetTagId());
            }
        }

        context.OptionalInstance()->RemoveTagLinks(tagIdsToRemove);

        TVector<TDBTag> tagsToRemove;
        return tagsManagerImpl.RestoreTags(tagIdsToRemove, tagsToRemove, session) &&
               tagsManagerImpl.RemoveTags(tagsToRemove, context.GetPerformerId(), context.GetServer(), session);
    }
}
